<?php
// Runtime error handling (prevent HTML notices in JSON responses)
@ini_set('display_errors', '0');
@ini_set('log_errors', '1');
@error_reporting(E_ALL);
/**
 * Database Configuration for SuperbHR Platform
 */

// Generate or retrieve a per-request correlation ID
function get_request_id() {
    static $rid = null;
    if ($rid !== null) return $rid;
    // Try to reuse incoming header if any (for proxied calls)
    $rid = isset($_SERVER['HTTP_X_REQUEST_ID']) && $_SERVER['HTTP_X_REQUEST_ID'] !== ''
        ? $_SERVER['HTTP_X_REQUEST_ID']
        : (function(){ $t = microtime(true); return sprintf('RID-%s-%04d', bin2hex(random_bytes(6)), (int)(($t - floor($t))*10000));})();
    return $rid;
}

/**
 * Domain activity logger (writes to `activities` if present)
 */
function recordActivity(PDO $db, int $companyId, ?int $userId, string $action, ?string $targetType = null, ?int $targetId = null, ?string $description = null, $metadata = null): void {
    try {
        $meta = is_string($metadata) ? $metadata : json_encode($metadata, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE);
        $st = $db->prepare("INSERT INTO activities (company_id, user_id, action, target_type, target_id, description, metadata, ip_address, user_agent, created_at)
                            VALUES (:cid, :uid, :act, :tt, :tid, :desc, :meta, :ip, :ua, NOW())");
        $st->bindValue(':cid', $companyId, PDO::PARAM_INT);
        $st->bindValue(':uid', $userId);
        $st->bindValue(':act', $action);
        $st->bindValue(':tt', $targetType);
        $st->bindValue(':tid', $targetId);
        $st->bindValue(':desc', $description);
        $st->bindValue(':meta', $meta);
        $st->bindValue(':ip', $_SERVER['REMOTE_ADDR'] ?? null);
        $st->bindValue(':ua', $_SERVER['HTTP_USER_AGENT'] ?? null);
        $st->execute();
    } catch (Throwable $e) {
        // Best-effort only; ignore if table doesn't exist or FKs fail
        app_log('warning', 'recordActivity_failed', ['error'=>$e->getMessage(), 'action'=>$action]);
    }
}

// Very lightweight file logger
function app_log($level, $message, $context = []) {
    try {
        $base = __DIR__ . '/../storage/logs';
        if (!is_dir($base)) { @mkdir($base, 0777, true); }
        $file = $base . '/app.log';
        $line = sprintf(
            "%s [%s] %s %s %s\n",
            date('Y-m-d H:i:s'),
            strtoupper($level),
            get_request_id(),
            $message,
            $context ? json_encode($context, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE) : ''
        );
        @file_put_contents($file, $line, FILE_APPEND);
    } catch (Throwable $e) {
        // As a last resort, write to PHP error log
        error_log('[SuperbHR logger failure] ' . $e->getMessage());
    }
}

class Database {
    private $host = 'localhost';
    private $db_name = 'superbhr';
    private $username = 'root';
    private $password = 'pass';
    private $conn;

    public function getConnection() {
        $this->conn = null;

        try {
            $this->conn = new PDO(
                "mysql:host=" . $this->host . ";dbname=" . $this->db_name,
                $this->username,
                $this->password,
                array(
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8"
                )
            );
        } catch(PDOException $exception) {
            app_log('error', 'DB connection error', ['error' => $exception->getMessage()]);
            // Fail fast with a structured error so front-end can parse
            ApiResponse::error('Database connection error', 500);
        }

        return $this->conn;
    }
}

/**
 * Base API Response Handler
 */
class ApiResponse {
    public static function success($data = null, $message = 'Success') {
        // Include request id header for correlation
        header('X-Request-ID: ' . get_request_id());
        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: no-store');
        if (function_exists('header_remove')) { @header_remove('X-Powered-By'); }
        if (ob_get_length()) { @ob_clean(); }
        http_response_code(200);
        echo json_encode([
            'success' => true,
            'message' => $message,
            'data' => $data,
            'request_id' => get_request_id()
        ]);
        exit;
    }

    public static function error($message = 'Error', $code = 400) {
        header('X-Request-ID: ' . get_request_id());
        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: no-store');
        if (function_exists('header_remove')) { @header_remove('X-Powered-By'); }
        if (ob_get_length()) { @ob_clean(); }
        // Log server-side error summary
        try {
            app_log('error', 'api_error', [
                'code' => $code,
                'message' => $message,
                'uri' => $_SERVER['REQUEST_URI'] ?? null,
                'method' => $_SERVER['REQUEST_METHOD'] ?? null,
            ]);
        } catch (Throwable $e) { /* ignore */ }
        http_response_code($code);
        echo json_encode([
            'success' => false,
            'message' => $message,
            'data' => null,
            'request_id' => get_request_id()
        ]);
        exit;
    }

    public static function unauthorized($message = 'Unauthorized') {
        self::error($message, 401);
    }

    public static function forbidden($message = 'Forbidden') {
        self::error($message, 403);
    }

    public static function notFound($message = 'Not Found') {
        self::error($message, 404);
    }
}

/**
 * Authentication Helper
 */
class Auth {
    private $db;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();
    }

    public function authenticate($username, $password) {
        $query = "SELECT u.*, ur.role_id, r.slug as role_slug, r.name as role_name, r.permissions 
                  FROM users u 
                  LEFT JOIN user_roles ur ON u.id = ur.user_id 
                  LEFT JOIN roles r ON ur.role_id = r.id 
                  WHERE u.username = :username AND u.status = 'active'";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':username', $username);
        try {
            $stmt->execute();
        } catch (Throwable $e) {
            app_log('error', 'Auth query failure', ['error' => $e->getMessage(), 'username' => $username]);
            return false;
        }

        if ($stmt->rowCount() > 0) {
            $user = $stmt->fetch();
            
            if (password_verify($password, $user['password'])) {
                // Update last login
                $updateQuery = "UPDATE users SET last_login_at = NOW() WHERE id = :id";
                $updateStmt = $this->db->prepare($updateQuery);
                $updateStmt->bindParam(':id', $user['id']);
                $updateStmt->execute();

                // Log activity
                $this->logActivity($user['id'], 'user_login');

                unset($user['password']);
                app_log('info', 'Login success', ['user_id' => $user['id'], 'username' => $username]);
                return $user;
            }
            // Wrong password for existing active user
            app_log('warning', 'Login failed: invalid password', ['username' => $username]);
            // record activity_logs
            try {
                $fail = $this->db->prepare("INSERT INTO activity_logs (user_id, action, data, ip_address, user_agent, created_at) VALUES (NULL, 'login_failed', :data, :ip, :ua, NOW())");
                $fail->bindValue(':data', json_encode(['username' => $username, 'reason' => 'invalid_password']));
                $fail->bindValue(':ip', $_SERVER['REMOTE_ADDR'] ?? null);
                $fail->bindValue(':ua', $_SERVER['HTTP_USER_AGENT'] ?? null);
                $fail->execute();
            } catch (Throwable $e) { /* ignore */ }
        }
        // User not found or inactive
        app_log('warning', 'Login failed: user_not_found_or_inactive', ['username' => $username]);
        try {
            $fail = $this->db->prepare("INSERT INTO activity_logs (user_id, action, data, ip_address, user_agent, created_at) VALUES (NULL, 'login_failed', :data, :ip, :ua, NOW())");
            $fail->bindValue(':data', json_encode(['username' => $username, 'reason' => 'user_not_found_or_inactive']));
            $fail->bindValue(':ip', $_SERVER['REMOTE_ADDR'] ?? null);
            $fail->bindValue(':ua', $_SERVER['HTTP_USER_AGENT'] ?? null);
            $fail->execute();
        } catch (Throwable $e) { /* ignore */ }
        return false;
    }

    public function hasPermission($userId, $permission) {
        $query = "SELECT r.permissions FROM users u 
                  JOIN user_roles ur ON u.id = ur.user_id 
                  JOIN roles r ON ur.role_id = r.id 
                  WHERE u.id = :user_id";
        
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':user_id', $userId);
        $stmt->execute();

        if ($stmt->rowCount() > 0) {
            $result = $stmt->fetch();
            $permissions = json_decode($result['permissions'], true);
            
            return in_array('*', $permissions) || in_array($permission, $permissions);
        }
        
        return false;
    }

    public function logActivity($userId, $action, $data = null) {
        try {
            // Gracefully ensure table exists (no-op if already present)
            $this->db->exec("CREATE TABLE IF NOT EXISTS activity_logs (
                id INT(11) NOT NULL AUTO_INCREMENT,
                user_id INT(11) DEFAULT NULL,
                action VARCHAR(255) NOT NULL,
                data LONGTEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL,
                ip_address VARCHAR(45) DEFAULT NULL,
                user_agent TEXT DEFAULT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (id),
                KEY idx_user_action (user_id, action),
                KEY idx_created_at (created_at)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");

            $query = "INSERT INTO activity_logs (user_id, action, data, ip_address, user_agent, created_at)
                      VALUES (:user_id, :action, :data, :ip, :user_agent, NOW())";

            $stmt = $this->db->prepare($query);
            $stmt->bindValue(':user_id', $userId);
            $stmt->bindValue(':action', $action);
            $stmt->bindValue(':data', json_encode($data, JSON_UNESCAPED_SLASHES|JSON_UNESCAPED_UNICODE));
            $stmt->bindValue(':ip', $_SERVER['REMOTE_ADDR'] ?? null);
            $stmt->bindValue(':user_agent', $_SERVER['HTTP_USER_AGENT'] ?? null);
            $stmt->execute();
        } catch (Throwable $e) {
            // Best-effort only: don't break main flow
            app_log('error', 'logActivity_failed', ['error' => $e->getMessage(), 'action' => $action]);
        }
    }
}

/**
 * Session Management
 */
session_start();
// Idle timeout (seconds)
if (!defined('IDLE_TIMEOUT_SECONDS')) {
    define('IDLE_TIMEOUT_SECONDS', 600); // 10 minutes
}

function requireAuth() {
    if (!isset($_SESSION['user_id'])) {
        ApiResponse::unauthorized('Authentication required');
    }
    // Inactivity enforcement
    $now = time();
    $last = isset($_SESSION['last_activity']) ? (int)$_SESSION['last_activity'] : $now;
    if (($now - $last) > IDLE_TIMEOUT_SECONDS) {
        // Optional: log timeout event (best-effort)
        try { if (isset($_SESSION['user_id'])) { $a = new Auth(); $a->logActivity((int)$_SESSION['user_id'], 'session_timeout', ['idle_seconds' => ($now - $last)]); } } catch (Throwable $e) {}
        @session_unset();
        @session_destroy();
        ApiResponse::unauthorized('Session expired due to inactivity');
    }
    // Refresh activity timestamp for active session
    $_SESSION['last_activity'] = $now;
}

function getCurrentUser() {
    return $_SESSION['user'] ?? null;
}

function hasRole($role) {
    $user = getCurrentUser();
    return $user && $user['role_slug'] === $role;
}

function canApprove($type) {
    $user = getCurrentUser();
    if (!$user) return false;
    
    $approvalRoles = [
        'leave' => ['super_admin', 'admin', 'hr_head', 'manager'],
        'training' => ['super_admin', 'admin', 'hr_head', 'manager'],
        'appraisal' => ['super_admin', 'admin', 'hr_head'],
        'payroll' => ['super_admin', 'admin', 'hr_head'],
        // New: employee exit approvals are handled by HR Head and Admin (and Super Admin)
        'exit' => ['super_admin', 'admin', 'hr_head']
    ];
    
    return isset($approvalRoles[$type]) && in_array($user['role_slug'], $approvalRoles[$type]);
}
?>
