// User Management page
(function(){
  window.pages = window.pages || {};

  const allowedRoles = ['super_admin','admin','hr_head','hr_officer'];
  function role(){ return (window.auth?.currentUser?.role_slug) || (window.auth?.currentUser?.role) || 'employee'; }
  function isAllowed(){ return allowedRoles.includes(role()); }
  function notify(msg, type){ try{ window.auth?.showNotification?.(msg, type||'info'); }catch(_){} }
  const esc = s => { try{ return String(s).replace(/[&<>"']/g,c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c])); }catch(_){ return String(s||''); } };

  async function render(container){
    if (!isAllowed()) { container.innerHTML = '<div class="alert alert-warning">Access denied. User Management is available to Admin and HR roles.</div>'; return; }

    container.innerHTML = `
      <div class="row g-3">
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header d-flex align-items-center justify-content-between"><h6 class="mb-0">Create User</h6></div>
            <div class="card-body">
              <div class="mb-2">
                <label class="form-label">Employee</label>
                <select id="umEmp" class="form-select"></select>
              </div>
              <div class="mb-2">
                <label class="form-label">Username</label>
                <input id="umUsername" class="form-control" placeholder="username" />
              </div>
              <div class="mb-2">
                <label class="form-label">Email</label>
                <input id="umEmail" type="email" class="form-control" placeholder="email@company.com" />
              </div>
              <div class="mb-2">
                <label class="form-label">Phone</label>
                <input id="umPhone" class="form-control" placeholder="+233..." />
              </div>
              <div class="mb-3">
                <label class="form-label">Roles</label>
                <div id="umRoles" class="d-flex flex-wrap gap-2"></div>
              </div>
              <div class="text-end">
                <button id="umCreateBtn" class="btn btn-primary"><i class="fas fa-user-plus"></i> Create User</button>
              </div>
              <div class="form-text mt-2">Default password is <code>password123</code>. You can reset after creation.</div>
            </div>
          </div>
        </div>
        <div class="col-lg-8">
          <div class="card h-100">
            <div class="card-header d-flex align-items-center justify-content-between">
              <h6 class="mb-0">Users</h6>
              <div class="d-flex align-items-center gap-2">
                <button class="btn btn-sm btn-outline-secondary" id="umRefresh"><i class="fas fa-rotate"></i> Refresh</button>
              </div>
            </div>
            <div class="card-body">
              <div class="table-responsive">
                <table class="table table-sm align-middle">
                  <thead><tr>
                    <th>Username</th><th>Name</th><th>Email</th><th>Phone</th><th>Status</th><th>Roles</th><th></th>
                  </tr></thead>
                  <tbody id="umTBody"></tbody>
                </table>
              </div>
            </div>
          </div>
        </div>
      </div>`;

    await Promise.all([
      loadRoles(container),
      loadUnlinkedEmployees(container),
      loadUsers(container)
    ]);

    container.querySelector('#umRefresh')?.addEventListener('click', ()=>loadUsers(container));
    container.querySelector('#umCreateBtn')?.addEventListener('click', ()=>createUser(container));

    // Auto-fill username/email from selected employee
    const empSel = container.querySelector('#umEmp');
    empSel?.addEventListener('change', ()=>{
      const opt = empSel.options[empSel.selectedIndex];
      const email = opt?.getAttribute('data-email') || '';
      const phone = opt?.getAttribute('data-phone') || '';
      const un = (opt?.getAttribute('data-suggest') || '').toLowerCase();
      const uIn = container.querySelector('#umUsername'); if (uIn && !uIn.value) uIn.value = un;
      const eIn = container.querySelector('#umEmail'); if (eIn && !eIn.value) eIn.value = email;
      const pIn = container.querySelector('#umPhone'); if (pIn && !pIn.value) pIn.value = phone;
    });
  }

  // Open inline role assignment form for a user
  async function openAssignRoles(container, userId, currentRoleIds){
    try{
      const res = await fetch('api/users.php?action=roles', { credentials:'same-origin' });
      const js = await res.json();
      const allRoles = (js?.success && Array.isArray(js.data)) ? js.data : [];
      const row = container.querySelector(`#rform-${userId}`);
      if (!row) return;
      row.classList.remove('d-none');
      const box = row.querySelector('[data-role-box]');
      box.innerHTML = allRoles.map(r=>{
        const checked = Array.isArray(currentRoleIds) && currentRoleIds.includes(Number(r.id)) ? 'checked' : '';
        return `<label class="form-check me-3">
          <input type="checkbox" class="form-check-input" value="${r.id}" ${checked} />
          <span class="form-check-label">${esc(r.name)}</span>
        </label>`;
      }).join('');

      // Save
      row.querySelector(`[data-save-roles="${userId}"]`)?.addEventListener('click', async ()=>{
        const ids = Array.from(row.querySelectorAll('input[type="checkbox"]:checked')).map(i=>Number(i.value));
        try{
          const r = await fetch('api/users.php?action=assign_roles', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ user_id: userId, role_ids: ids }) });
          const j = await r.json();
          if (j?.success){ notify('Roles updated','success'); loadUsers(container); row.classList.add('d-none'); }
          else notify(j?.message||'Update failed','error');
        }catch(_){ notify('Update failed','error'); }
      }, { once: true });

      // Cancel
      row.querySelector(`[data-cancel-roles="${userId}"]`)?.addEventListener('click', ()=>{ row.classList.add('d-none'); }, { once: true });
    }catch(_){ notify('Failed to load roles','error'); }
  }

  async function loadRoles(container){
    try{
      const res = await fetch('api/users.php?action=roles', { credentials:'same-origin' });
      const js = await res.json();
      if (!js?.success) throw 0;
      const wrap = container.querySelector('#umRoles');
      if (wrap) {
        const arr = Array.isArray(js.data) ? js.data : [];
        wrap.innerHTML = arr.map(r=>`
          <label class="form-check me-3">
            <input type="checkbox" class="form-check-input" value="${r.id}" />
            <span class="form-check-label">${esc(r.name)}</span>
          </label>`).join('');
        if (!arr.length) wrap.innerHTML = '<span class="text-muted">No roles available</span>';
      }
    }catch(err){ console.error('Failed to load roles', err); const wrap = container.querySelector('#umRoles'); if (wrap) wrap.innerHTML = '<span class="text-danger">Failed to load roles</span>'; }
  }

  async function loadUnlinkedEmployees(container){
    try{
      const res = await fetch('api/users.php?action=unlinked_employees', { credentials:'same-origin' });
      const js = await res.json();
      const sel = container.querySelector('#umEmp'); if (!sel) return;
      const rows = js?.success ? (js.data||[]) : [];
      sel.innerHTML = '<option value="">Select employee...</option>' + rows.map(e=>`<option value="${e.id}" data-email="${esc(e.email||'')}" data-phone="${esc(e.phone||'')}" data-suggest="${esc((e.first_name||'').charAt(0)+(e.last_name||''))}">${esc(e.employee_number||'')} - ${esc((e.first_name||'')+' '+(e.last_name||''))}</option>`).join('');
    }catch(_){ /* ignore */ }
  }

  async function loadUsers(container){
    const tb = container.querySelector('#umTBody'); if (!tb) return;
    tb.innerHTML = '<tr><td colspan="7" class="text-muted">Loading...</td></tr>';
    try{
      const res = await fetch('api/users.php', { credentials:'same-origin' });
      const js = await res.json();
      const rows = js?.success ? (js.data || []) : [];
      tb.innerHTML = rows.length ? rows.map(u=>{
        const roles = Array.isArray(u.roles)?u.roles:[];
        const roleStr = roles.map(r=>esc(r.name)).join(', ');
        const active = (u.status||'')==='active';
        const roleIds = roles.map(r=>r.id).join(',');
        return `
        <tr>
          <td>${esc(u.username||'')}</td>
          <td>${esc((u.first_name||'')+' '+(u.last_name||''))}</td>
          <td>${esc(u.email||'')}</td>
          <td>${esc(u.phone||'')}</td>
          <td>${active?'<span class="badge bg-success">active</span>':'<span class="badge bg-secondary">inactive</span>'}</td>
          <td>${roleStr||'<span class="text-muted">none</span>'}</td>
          <td class="text-end">
            <div class="btn-group btn-group-sm">
              <button class="btn btn-outline-primary" data-assign="${u.id}" data-roles="${roleIds}"><i class="fas fa-user-shield"></i></button>
              <button class="btn btn-outline-warning" data-reset="${u.id}"><i class="fas fa-key"></i></button>
              <button class="btn ${active?'btn-outline-danger':'btn-outline-success'}" data-toggle="${u.id}" data-next="${active?'inactive':'active'}">${active?'Deactivate':'Activate'}</button>
            </div>
          </td>
        </tr>
        <tr class="d-none" id="rform-${u.id}">
          <td colspan="7">
            <div class="border rounded p-2 bg-light">
              <div class="d-flex flex-wrap gap-2" data-role-box></div>
              <div class="text-end mt-2">
                <button class="btn btn-sm btn-primary" data-save-roles="${u.id}"><i class="fas fa-save"></i> Save Roles</button>
                <button class="btn btn-sm btn-outline-secondary" data-cancel-roles="${u.id}">Cancel</button>
              </div>
            </div>
          </td>
        </tr>`;
      }).join('') : '<tr><td colspan="7" class="text-muted">No users found.</td></tr>';

      // Bind actions per row
      tb.querySelectorAll('[data-assign]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const uid = Number(btn.getAttribute('data-assign'));
          const currentRoleIds = (btn.getAttribute('data-roles')||'').split(',').filter(Boolean).map(x=>Number(x));
          await openAssignRoles(container, uid, currentRoleIds);
        });
      });
      tb.querySelectorAll('[data-reset]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const uid = Number(btn.getAttribute('data-reset'));
          const np = prompt('Enter new password (leave blank to reset to default password123):','');
          try{
            const r = await fetch('api/users.php?action=reset_password', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ user_id: uid, new_password: np||null }) });
            const j = await r.json();
            if (j?.success) notify('Password reset','success'); else notify(j?.message||'Reset failed','error');
          }catch(_){ notify('Reset failed','error'); }
        });
      });
      tb.querySelectorAll('[data-toggle]').forEach(btn=>{
        if (btn._bound) return; btn._bound = true;
        btn.addEventListener('click', async ()=>{
          const uid = Number(btn.getAttribute('data-toggle'));
          const next = btn.getAttribute('data-next');
          try{
            const r = await fetch('api/users.php?action=toggle_status', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ user_id: uid, status: next }) });
            const j = await r.json();
            if (j?.success){ notify('Status updated','success'); loadUsers(container); } else notify(j?.message||'Update failed','error');
          }catch(_){ notify('Update failed','error'); }
        });
      });
    }catch(_){ /* ignore */ }
  }

  async function createUser(container){
    const emp = container.querySelector('#umEmp').value;
    const username = container.querySelector('#umUsername').value.trim();
    const email = container.querySelector('#umEmail').value.trim();
    const phone = container.querySelector('#umPhone').value.trim();
    const roles = Array.from(container.querySelectorAll('#umRoles input[type="checkbox"]:checked')).map(i=>Number(i.value));
    if (!emp || !username){ notify('Employee and username are required','warning'); return; }
    try{
      const r = await fetch('api/users.php?action=create', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ employee_id: Number(emp), username, email, phone, role_ids: roles }) });
      const j = await r.json();
      if (j?.success){ notify('User created','success'); loadUsers(container); loadUnlinkedEmployees(container); container.querySelector('#umUsername').value=''; }
      else notify(j?.message||'Create failed','error');
    }catch(_){ notify('Create failed','error'); }
  }

  window.pages.users = { render };
})();
