// Super Admin Portal: Dashboard & Companies management
(function(){
  window.pages = window.pages || {};

  // Helpers
  async function api(path, options){
    const res = await fetch(path, Object.assign({ credentials: 'same-origin' }, options||{}));
    const js = await res.json();
    if (!js?.success) throw new Error(js?.message || 'Request failed');
    return js.data;
  }

  function cardStat(title, value, icon){
    return `<div class="col-md-3">
      <div class="card h-100">
        <div class="card-body d-flex align-items-center justify-content-between">
          <div>
            <div class="text-muted small">${title}</div>
            <div class="h4 mb-0">${value}</div>
          </div>
          <div class="text-primary fs-3"><i class="${icon}"></i></div>
        </div>
      </div>
    </div>`
  }

  // Page 1: Super Admin Dashboard
  async function renderSuperAdminDashboard(container){
    container.innerHTML = `<div class="row g-3" id="sadash-stats"></div>
      <div class="row g-3 mt-1">
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Role Distribution (All Companies)</h6>
              <small class="text-muted">Users per role</small>
            </div>
            <div class="card-body">
              <canvas id="rolePie"></canvas>
            </div>
          </div>
        </div>
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Companies Overview</h6>
              <button class="btn btn-sm btn-outline-primary" id="btnRefreshCompanies">Refresh</button>
            </div>
            <div class="card-body table-responsive" style="max-height: 360px;">
              <table class="table table-sm align-middle">
                <thead>
                  <tr>
                    <th>Company</th>
                    <th class="text-end">Users</th>
                    <th>Admins</th>
                    <th>HR Head</th>
                    <th>HR Officer</th>
                    <th>Mgr</th>
                    <th>Emp</th>
                  </tr>
                </thead>
                <tbody id="companiesTableBody"></tbody>
              </table>
            </div>
          </div>
        </div>
      </div>`;

    // Load stats
    const stats = await api('api/superadmin.php?action=overall_stats');
    const statsRow = container.querySelector('#sadash-stats');
    statsRow.innerHTML = [
      cardStat('Total Companies', stats.total_companies ?? 0, 'fas fa-building'),
      cardStat('Total Users', stats.total_users ?? 0, 'fas fa-users'),
      cardStat('Active Users (7d)', stats.active_users_7d ?? 0, 'fas fa-user-check'),
      cardStat('Events (7d)', stats.events_7d ?? 0, 'fas fa-chart-line')
    ].join('');

    // Role distribution chart
    const labels = Object.keys(stats.roles || {});
    const values = labels.map(k=> stats.roles[k]);
    const ctx = container.querySelector('#rolePie');
    if (ctx && window.Chart){
      new Chart(ctx, {
        type: 'doughnut',
        data: {
          labels,
          datasets: [{ data: values, backgroundColor: ['#2563eb','#22c55e','#eab308','#ef4444','#06b6d4','#8b5cf6'] }]
        },
        options: { plugins: { legend: { position: 'bottom' } } }
      });
    }

    // Companies table
    async function loadCompanies(){
      const companies = await api('api/superadmin.php?action=list_companies');
      const tbody = container.querySelector('#companiesTableBody');
      tbody.innerHTML = (companies||[]).map(c=>{
        const r = c.roles || {}; // {slug: count}
        const roles = (slug)=> (r[slug] ?? 0);
        const name = c.name || ('Company #' + c.id);
        return `<tr>
          <td>${name}</td>
          <td class="text-end">${c.users_count ?? 0}</td>
          <td>${roles('admin')}</td>
          <td>${roles('hr_head')}</td>
          <td>${roles('hr_officer')}</td>
          <td>${roles('manager')}</td>
          <td>${roles('employee')}</td>
        </tr>`;
      }).join('');
    }
    await loadCompanies();
    container.querySelector('#btnRefreshCompanies')?.addEventListener('click', loadCompanies);
  }

  // Page 2: Companies management
  async function renderCompanies(container){
    container.innerHTML = `
      <div class="row g-3">
        <div class="col-lg-5">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Create Company</h6>
            </div>
            <div class="card-body">
              <div class="mb-2">
                <label class="form-label">Name</label>
                <input id="coName" class="form-control" placeholder="Acme Ltd" />
              </div>
              <div class="mb-2"><label class="form-label">Email</label><input id="coEmail" class="form-control" placeholder="info@acme.com" /></div>
              <div class="mb-2"><label class="form-label">Phone</label><input id="coPhone" class="form-control" placeholder="+233..." /></div>
              <div class="mb-2"><label class="form-label">Website</label><input id="coWebsite" class="form-control" placeholder="https://..." /></div>
              <button class="btn btn-primary" id="btnCreateCompany">Create Company</button>
            </div>
          </div>
        </div>
        <div class="col-lg-7">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Companies</h6>
              <div>
                <button class="btn btn-sm btn-outline-primary" id="btnRefreshCo">Refresh</button>
              </div>
            </div>
            <div class="card-body">
              <div class="row g-3">
                <div class="col-12">
                  <select class="form-select" id="companiesSelect"></select>
                </div>
                <div class="col-12">
                  <div class="table-responsive" style="max-height: 240px;">
                    <table class="table table-sm align-middle">
                      <thead><tr><th>User</th><th>Email</th><th>Status</th><th>Roles</th><th>Created</th></tr></thead>
                      <tbody id="companyUsersBody"></tbody>
                    </table>
                  </div>
                </div>
                <div class="col-12">
                  <div class="card">
                    <div class="card-header"><h6 class="mb-0">Create Company User</h6></div>
                    <div class="card-body row g-2">
                      <div class="col-md-4"><input id="cuUsername" class="form-control" placeholder="username" /></div>
                      <div class="col-md-4"><input id="cuFirst" class="form-control" placeholder="First name" /></div>
                      <div class="col-md-4"><input id="cuLast" class="form-control" placeholder="Last name" /></div>
                      <div class="col-md-6"><input id="cuEmail" class="form-control" placeholder="email@example.com" /></div>
                      <div class="col-md-6"><input id="cuPassword" class="form-control" placeholder="password (optional)" type="password" /></div>
                      <div class="col-md-6">
                        <select id="cuRole" class="form-select">
                          <option value="admin">Admin</option>
                          <option value="hr_head">HR Head/Lead</option>
                          <option value="hr_officer">HR Officer</option>
                          <option value="manager">Manager</option>
                          <option value="employee">Employee</option>
                        </select>
                      </div>
                      <div class="col-md-6 text-end">
                        <button class="btn btn-primary" id="btnCreateCompanyUser">Create User</button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>`;

    async function loadCompaniesIntoSelect(selectedId){
      const companies = await api('api/superadmin.php?action=list_companies');
      const sel = container.querySelector('#companiesSelect');
      sel.innerHTML = companies.map(c=> `<option value="${c.id}">${c.name || ('Company #'+c.id)} (${c.users_count||0} users)</option>`).join('');
      if (selectedId) sel.value = String(selectedId);
      await loadUsers();
    }

    async function loadUsers(){
      const sel = container.querySelector('#companiesSelect');
      const cid = parseInt(sel.value || '0', 10) || 0;
      if (!cid) { container.querySelector('#companyUsersBody').innerHTML = ''; return; }
      const users = await api('api/superadmin.php?action=list_users&company_id='+cid);
      const tbody = container.querySelector('#companyUsersBody');
      tbody.innerHTML = (users||[]).map(u=> `<tr>
        <td>${u.username} <div class="text-muted small">${(u.first_name||'')+' '+(u.last_name||'')}</div></td>
        <td>${u.email||''}</td>
        <td><span class="badge bg-${u.status==='active'?'success':'secondary'}">${u.status}</span></td>
        <td>${(u.roles||[]).join(', ')}</td>
        <td class="text-nowrap">${(u.created_at||'').replace('T',' ').replace('Z','')}</td>
      </tr>`).join('');
    }

    container.querySelector('#btnRefreshCo')?.addEventListener('click', ()=> loadCompaniesIntoSelect());
    container.querySelector('#companiesSelect')?.addEventListener('change', loadUsers);

    container.querySelector('#btnCreateCompany')?.addEventListener('click', async ()=>{
      const payload = {
        name: container.querySelector('#coName').value.trim(),
        email: container.querySelector('#coEmail').value.trim(),
        phone: container.querySelector('#coPhone').value.trim(),
        website: container.querySelector('#coWebsite').value.trim()
      };
      try{
        await api('api/superadmin.php?action=create_company', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        window.auth?.showNotification?.('Company created','success');
        container.querySelector('#coName').value='';
        container.querySelector('#coEmail').value='';
        container.querySelector('#coPhone').value='';
        container.querySelector('#coWebsite').value='';
        await loadCompaniesIntoSelect();
      }catch(e){ window.auth?.showNotification?.(String(e.message||e),'error'); }
    });

    container.querySelector('#btnCreateCompanyUser')?.addEventListener('click', async ()=>{
      const cid = parseInt(container.querySelector('#companiesSelect').value||'0',10)||0;
      const payload = {
        company_id: cid,
        username: container.querySelector('#cuUsername').value.trim(),
        first_name: container.querySelector('#cuFirst').value.trim(),
        last_name: container.querySelector('#cuLast').value.trim(),
        email: container.querySelector('#cuEmail').value.trim(),
        password: container.querySelector('#cuPassword').value,
        role_slug: container.querySelector('#cuRole').value
      };
      try{
        await api('api/superadmin.php?action=create_company_user', { method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) });
        window.auth?.showNotification?.('User created','success');
        ['#cuUsername','#cuFirst','#cuLast','#cuEmail','#cuPassword'].forEach(s=> container.querySelector(s).value='');
        await loadUsers();
      }catch(e){ window.auth?.showNotification?.(String(e.message||e),'error'); }
    });

    // init
    await loadCompaniesIntoSelect();
  }

  window.pages.superadmin_dashboard = { render: renderSuperAdminDashboard };
  window.pages.companies = { render: renderCompanies };
})();
