// Positions management page module
(function () {
  window.pages = window.pages || {};

  function canManage() {
    const role = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head','hr_officer'].includes(role);
  }

  function canDeactivatePositions() {
    const role = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head'].includes(role);
  }

  // Caches
  let _posCache = null;
  let _empCache = null;
  let _deptCache = null;
  const _state = { search: '', department: '', status: '' };

  async function ensureCaches() {
    if (!_posCache) await cachePositions();
    if (!_empCache) await cacheEmployees();
    if (!_deptCache) await cacheDepartments();
  }

  async function cachePositions() {
    try {
      const res = await fetch('api/organization.php?entity=positions', { credentials: 'same-origin' });
      const data = await res.json();
      _posCache = data?.success && Array.isArray(data.data) ? data.data : [];
    } catch (_) { _posCache = []; }
  }

  async function cacheEmployees() {
    try {
      const res = await fetch('api/employees.php', { credentials: 'same-origin' });
      const data = await res.json();
      _empCache = data?.success && Array.isArray(data.data) ? data.data : [];
    } catch (_) { _empCache = []; }
  }

  async function cacheDepartments() {
    try {
      const res = await fetch('api/organization.php?entity=departments', { credentials: 'same-origin' });
      const data = await res.json();
      _deptCache = data?.success && Array.isArray(data.data) ? data.data : [];
    } catch (_) { _deptCache = []; }
  }

  function escapeHtml(s){ return String(s ?? '').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }
  function statusBadge(status){ const map={active:'status-badge status-active', inactive:'status-badge status-inactive'}; return `<span class="${map[status]||'status-badge'}">${escapeHtml(status||'')}</span>`; }

  async function render(container) {
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Positions</h5>
          ${canManage() ? `<button class="btn btn-sm btn-primary" id="toggleCreate"><i class="fas fa-plus"></i> New Position</button>` : ''}
        </div>
        <div class="card-body">
          <div class="d-flex flex-wrap gap-2 mb-3 align-items-end">
            <div class="me-2">
              <label class="form-label mb-1">Search</label>
              <input id="posSearch" class="form-control form-control-sm" placeholder="Title or Code..." />
            </div>
            <div class="me-2">
              <label class="form-label mb-1">Department</label>
              <select id="posDeptFilter" class="form-select form-select-sm"><option value="">All</option></select>
            </div>
            <div class="me-2">
              <label class="form-label mb-1">Status</label>
              <select id="posStatusFilter" class="form-select form-select-sm">
                <option value="">All</option>
                <option value="active">active</option>
                <option value="inactive">inactive</option>
              </select>
            </div>
            <div class="ms-auto">
              <button class="btn btn-sm btn-outline-secondary" id="posExportBtn"><i class="fas fa-file-csv"></i> Export CSV</button>
            </div>
          </div>
          <ul class="nav nav-tabs" role="tablist">
            <li class="nav-item" role="presentation"><button class="nav-link active" data-bs-toggle="tab" data-bs-target="#tab-pos-list" type="button" role="tab"><i class="fas fa-briefcase"></i> Positions</button></li>
            ${canManage() ? `<li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-map" type="button" role="tab"><i class="fas fa-user-tag"></i> Map Employees</button></li>` : ''}
            <li class="nav-item" role="presentation"><button class="nav-link" data-bs-toggle="tab" data-bs-target="#tab-dist" type="button" role="tab"><i class="fas fa-diagram-project"></i> Distribution</button></li>
          </ul>
          <div class="tab-content mt-3">
            <div class="tab-pane fade show active" id="tab-pos-list" role="tabpanel">
              ${canManage() ? `
              <div id="createBox" class="mb-4" style="display:none;">
                <form id="createPosForm" class="row g-3">
                  <div class="col-md-3">
                    <label class="form-label">Title</label>
                    <input type="text" name="title" class="form-control" required />
                  </div>
                  <div class="col-md-2">
                    <label class="form-label">Code</label>
                    <input type="text" name="code" class="form-control" required />
                  </div>
                  <div class="col-md-3">
                    <label class="form-label">Department</label>
                    <select name="department_id" id="posDeptSelect" class="form-select"><option value=""></option></select>
                  </div>
                  <div class="col-md-2">
                    <label class="form-label">Level</label>
                    <select name="level" class="form-select">
                      <option value="entry">entry</option>
                      <option value="mid">mid</option>
                      <option value="senior">senior</option>
                      <option value="lead">lead</option>
                    </select>
                  </div>
                  <div class="col-md-2">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                      <option value="active">active</option>
                      <option value="inactive">inactive</option>
                    </select>
                  </div>
                  <div class="col-md-6">
                    <label class="form-label">Requirements</label>
                    <input type="text" name="requirements" class="form-control" />
                  </div>
                  <div class="col-md-6">
                    <label class="form-label">Responsibilities</label>
                    <input type="text" name="responsibilities" class="form-control" />
                  </div>
                  <div class="col-md-4">
                    <label class="form-label">Min Salary</label>
                    <input type="number" name="min_salary" class="form-control" step="0.01" />
                  </div>
                  <div class="col-md-4">
                    <label class="form-label">Max Salary</label>
                    <input type="number" name="max_salary" class="form-control" step="0.01" />
                  </div>
                  <div class="col-md-4">
                    <label class="form-label">Description</label>
                    <input type="text" name="description" class="form-control" />
                  </div>
                  <div class="col-12 d-flex justify-content-end">
                    <button type="button" class="btn btn-light me-2" id="cancelCreate">Cancel</button>
                    <button type="submit" class="btn btn-primary">Create</button>
                  </div>
                </form>
                <hr/>
              </div>` : ''}

              <div class="table-responsive">
                <table class="table mb-0">
                  <thead>
                    <tr>
                      <th>Title</th>
                      <th>Code</th>
                      <th>Department</th>
                      <th>Level</th>
                      <th>Status</th>
                      <th style="width: 160px;">Actions</th>
                    </tr>
                  </thead>
                  <tbody id="posTableBody"><tr><td colspan="6">Loading...</td></tr></tbody>
                </table>
              </div>
            </div>

            <div class="tab-pane fade" id="tab-map" role="tabpanel">
              ${canManage() ? `
              <div id="mapWrap">
                <div class="row g-3 align-items-end">
                  <div class="col-md-4">
                    <label class="form-label">Target Position</label>
                    <select id="mapPosSelect" class="form-select"></select>
                  </div>
                  <div class="col-md-8 text-end">
                    <button class="btn btn-sm btn-secondary me-2" id="mapReload"><i class="fas fa-rotate"></i> Reload</button>
                    <button class="btn btn-sm btn-primary" id="assignSelected"><i class="fas fa-user-check"></i> Assign Selected</button>
                  </div>
                  <div class="col-12">
                    <input id="mapSearch" class="form-control" placeholder="Search employees by name, email or #..." />
                  </div>
                  <div class="col-12">
                    <div class="table-responsive">
                      <table class="table table-sm align-middle">
                        <thead>
                          <tr>
                            <th style="width:28px"><input type="checkbox" id="mapSelectAll" /></th>
                            <th>#</th>
                            <th>Name</th>
                            <th>Position</th>
                            <th>Department</th>
                            <th>Status</th>
                          </tr>
                        </thead>
                        <tbody id="mapEmpTableBody"><tr><td colspan="6">Loading...</td></tr></tbody>
                      </table>
                    </div>
                  </div>
                </div>
              </div>` : '<div class="text-muted">You do not have permission to map employees.</div>'}
            </div>

            <div class="tab-pane fade" id="tab-dist" role="tabpanel">
              <div class="d-flex justify-content-between align-items-center mb-2">
                <div class="form-check">
                  <input class="form-check-input" type="checkbox" id="showUnassigned" />
                  <label class="form-check-label" for="showUnassigned">Show unassigned</label>
                </div>
                <button class="btn btn-sm btn-outline-secondary" id="distReload"><i class="fas fa-rotate"></i> Reload</button>
              </div>
              <div id="distContent">Loading...</div>
            </div>
          </div>
        </div>
      </div>

      <!-- Edit Position Modal -->
      <div class="modal fade" id="posEditModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Edit Position</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <form id="editPosForm" class="row g-3">
                <div class="col-md-3">
                  <label class="form-label">Title</label>
                  <input type="text" name="title" class="form-control" required />
                </div>
                <div class="col-md-2">
                  <label class="form-label">Code</label>
                  <input type="text" name="code" class="form-control" required />
                </div>
                <div class="col-md-3">
                  <label class="form-label">Department</label>
                  <select name="department_id" id="editPosDeptSelect" class="form-select"><option value=""></option></select>
                </div>
                <div class="col-md-2">
                  <label class="form-label">Level</label>
                  <select name="level" class="form-select">
                    <option value="entry">entry</option>
                    <option value="mid">mid</option>
                    <option value="senior">senior</option>
                    <option value="lead">lead</option>
                  </select>
                </div>
                <div class="col-md-2">
                  <label class="form-label">Status</label>
                  <select name="status" class="form-select">
                    <option value="active">active</option>
                    <option value="inactive">inactive</option>
                  </select>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Requirements</label>
                  <input type="text" name="requirements" class="form-control" />
                </div>
                <div class="col-md-6">
                  <label class="form-label">Responsibilities</label>
                  <input type="text" name="responsibilities" class="form-control" />
                </div>
                <div class="col-md-4">
                  <label class="form-label">Min Salary</label>
                  <input type="number" name="min_salary" class="form-control" step="0.01" />
                </div>
                <div class="col-md-4">
                  <label class="form-label">Max Salary</label>
                  <input type="number" name="max_salary" class="form-control" step="0.01" />
                </div>
                <div class="col-md-4">
                  <label class="form-label">Description</label>
                  <input type="text" name="description" class="form-control" />
                </div>
              </form>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Close</button>
              <button type="button" class="btn btn-primary" id="savePosBtn">Save</button>
            </div>
          </div>
        </div>
      </div>
    `;

    if (canManage()) {
      const toggleBtn = container.querySelector('#toggleCreate');
      const box = container.querySelector('#createBox');
      const cancelBtn = container.querySelector('#cancelCreate');
      toggleBtn?.addEventListener('click', async () => {
        box.style.display = box.style.display === 'none' ? 'block' : 'none';
        if (box.style.display === 'block') {
          try { await ensureCaches(); populateDeptSelect(container.querySelector('#posDeptSelect')); } catch (_) {}
        }
      });
      cancelBtn?.addEventListener('click', () => { box.style.display = 'none'; });

      const form = container.querySelector('#createPosForm');
      form?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const body = formToJSON(new FormData(form));
        // Client-side duplicate code check (case-insensitive)
        try { await ensureCaches(); } catch(_){ }
        const codeVal = String(body.code||'').trim().toLowerCase();
        if (codeVal && Array.isArray(_posCache) && _posCache.some(p=>String(p.code||'').trim().toLowerCase()===codeVal)){
          window.auth?.showNotification?.('Position code already exists', 'warning');
          return;
        }
        try {
          const res = await fetch('api/organization.php?entity=positions', {
            method: 'POST', headers: { 'Content-Type': 'application/json' }, credentials: 'same-origin', body: JSON.stringify(body)
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Position created', 'success');
            form.reset(); box.style.display = 'none'; await loadPositions(container); try { await cachePositions(); } catch(_){ }
          } else { window.auth?.showNotification?.(data?.message || 'Failed to create position', 'error'); }
        } catch (_) { window.auth?.showNotification?.('Failed to create position', 'error'); }
      });
    }

    // Delegate actions (edit / soft delete)
    container.addEventListener('click', async (e) => {
      const btn = e.target.closest('[data-action]'); if (!btn) return;
      const action = btn.getAttribute('data-action'); const id = btn.getAttribute('data-id');
      if (action === 'deactivate' && id) {
        if (!confirm('Deactivate this position?')) return;
        try {
          const res = await fetch(`api/organization.php?entity=positions&id=${encodeURIComponent(id)}`, { method: 'DELETE', credentials: 'same-origin' });
          const data = await res.json();
          if (data?.success) { window.auth?.showNotification?.('Position deactivated', 'success'); await loadPositions(container); try { await cachePositions(); } catch(_){} }
          else { window.auth?.showNotification?.(data?.message || 'Action failed', 'error'); }
        } catch (_) { window.auth?.showNotification?.('Action failed', 'error'); }
      } else if (action === 'edit' && id) {
        try { await ensureCaches(); await openEditPosition(container, parseInt(id,10)); } catch (_) {}
      } else if (action === 'activate' && id) {
        try {
          const r = await fetch(`api/organization.php?entity=positions&id=${encodeURIComponent(id)}`, { method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ status: 'active' }) });
          const out = await r.json();
          if (out?.success){ window.auth?.showNotification?.('Position reactivated', 'success'); await loadPositions(container); try { await cachePositions(); } catch(_){} }
          else { window.auth?.showNotification?.(out?.message || 'Action failed', 'error'); }
        } catch(_) { window.auth?.showNotification?.('Action failed', 'error'); }
      }
    });

    // Filters and export
    const searchEl = container.querySelector('#posSearch');
    const deptFilterEl = container.querySelector('#posDeptFilter');
    const statusFilterEl = container.querySelector('#posStatusFilter');
    const exportBtn = container.querySelector('#posExportBtn');
    try { await ensureCaches(); populateDeptFilter(deptFilterEl); } catch(_){ }
    searchEl?.addEventListener('input', ()=>{ _state.search = searchEl.value.trim().toLowerCase(); renderPositionsTable(container); });
    deptFilterEl?.addEventListener('change', ()=>{ _state.department = deptFilterEl.value || ''; renderPositionsTable(container); });
    statusFilterEl?.addEventListener('change', ()=>{ _state.status = statusFilterEl.value || ''; renderPositionsTable(container); });
    exportBtn?.addEventListener('click', ()=>{ exportPositionsCSV(); });

    // Initial loads
    await loadPositions(container);
    try { await ensureCaches(); } catch(_){ }
    initMapTab(container);
    initDistributionTab(container);
  }

  function formToJSON(fd){ const obj={}; fd.forEach((v,k)=>{ obj[k]=v; }); return obj; }

  async function loadPositions(container){
    const tbody = container.querySelector('#posTableBody');
    if (tbody) tbody.innerHTML = `<tr><td colspan="6">Loading...</td></tr>`;
    try {
      const res = await fetch('api/organization.php?entity=positions', { credentials: 'same-origin' });
      const data = await res.json();
      _posCache = data?.success && Array.isArray(data.data) ? data.data : [];
      renderPositionsTable(container);
    } catch (_) {
      if (tbody) tbody.innerHTML = `<tr><td colspan="6" class="text-danger">Failed to load positions</td></tr>`;
    }
  }

  function renderPositionsTable(container){
    const tbody = container.querySelector('#posTableBody'); if (!tbody) return;
    const filtered = getFilteredPositionsList();
    if (!filtered.length) { tbody.innerHTML = `<tr><td colspan="6" class="text-muted text-center">No positions</td></tr>`; return; }
    tbody.innerHTML = filtered.map(p=>posRow(p)).join('');
  }

  function getFilteredPositionsList(){
    const list = Array.isArray(_posCache) ? _posCache : [];
    const term = _state.search;
    const dep = _state.department;
    const st = _state.status;
    return list.filter(p=>{
      const matchTerm = !term || (String(p.title||'').toLowerCase().includes(term) || String(p.code||'').toLowerCase().includes(term));
      const matchDep = !dep || String(p.department_id||'') === String(dep);
      const matchSt = !st || String(p.status||'') === st;
      return matchTerm && matchDep && matchSt;
    });
  }

  function posRow(p){
    const status = p.status || 'active';
    const editBtn = `<button class="btn btn-sm btn-outline-primary" data-action="edit" data-id="${p.id}"><i class="fas fa-pen"></i></button>`;
    const deactivateBtn = (status === 'active' && canDeactivatePositions()) ? `<button class="btn btn-sm btn-danger" data-action="deactivate" data-id="${p.id}"><i class="fas fa-ban"></i></button>` : '';
    const activateBtn = (status === 'inactive' && canDeactivatePositions()) ? `<button class="btn btn-sm btn-success" data-action="activate" data-id="${p.id}"><i class="fas fa-rotate"></i></button>` : '';
    const controls = canManage() ? `<div class="btn-group">${editBtn}${deactivateBtn}${activateBtn}</div>` : '';
    return `
      <tr>
        <td>${escapeHtml(p.title || '')}</td>
        <td>${escapeHtml(p.code || '')}</td>
        <td>${escapeHtml(p.department_name || '-')}</td>
        <td>${escapeHtml(p.level || '')}</td>
        <td>${statusBadge(status)}</td>
        <td>${controls}</td>
      </tr>`;
  }

  function exportPositionsCSV(){
    const rows = getFilteredPositionsList();
    const header = ['id','title','code','department_id','department_name','level','status'];
    const csv = [header.join(',')]
      .concat(rows.map(p=>[
        p.id,
        safeCsv(p.title),
        safeCsv(p.code),
        p.department_id ?? '',
        safeCsv(p.department_name),
        safeCsv(p.level),
        safeCsv(p.status||'active')
      ].join(',')))
      .join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a'); a.href = url; a.download = 'positions.csv'; a.click();
    setTimeout(()=>URL.revokeObjectURL(url), 1000);
  }

  function safeCsv(v){
    const s = String(v ?? '');
    if (/[",\n]/.test(s)) return '"'+s.replace(/"/g,'""')+'"';
    return s;
  }

  function populateDeptSelect(selectEl){
    if (!selectEl) return;
    const opts = ['<option value=""></option>']
      .concat((_deptCache||[]).map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`));
    selectEl.innerHTML = opts.join('');
  }

  function populateDeptFilter(selectEl){
    if (!selectEl) return;
    const opts = ['<option value="">All</option>']
      .concat((_deptCache||[]).map(d=>`<option value="${d.id}">${escapeHtml(d.name)}</option>`));
    selectEl.innerHTML = opts.join('');
  }

  async function openEditPosition(container, id){
    const res = await fetch(`api/organization.php?entity=positions&id=${encodeURIComponent(id)}`, { credentials:'same-origin' });
    const data = await res.json();
    if (!data?.success) { window.auth?.showNotification?.('Position not found', 'error'); return; }
    const p = data.data;
    const form = container.querySelector('#editPosForm'); if (!form) return;
    // Populate
    form.querySelector('[name="title"]').value = p.title || '';
    form.querySelector('[name="code"]').value = p.code || '';
    populateDeptSelect(container.querySelector('#editPosDeptSelect'));
    const depSel = container.querySelector('#editPosDeptSelect'); if (depSel) depSel.value = p.department_id ? String(p.department_id) : '';
    form.querySelector('[name="level"]').value = p.level || 'entry';
    form.querySelector('[name="status"]').value = p.status || 'active';
    form.querySelector('[name="requirements"]').value = p.requirements || '';
    form.querySelector('[name="responsibilities"]').value = p.responsibilities || '';
    form.querySelector('[name="min_salary"]').value = p.min_salary || '';
    form.querySelector('[name="max_salary"]').value = p.max_salary || '';
    form.querySelector('[name="description"]').value = p.description || '';
    const saveBtn = container.querySelector('#savePosBtn'); if (saveBtn) saveBtn.dataset.id = String(id);
    const modal = bootstrap.Modal.getOrCreateInstance(container.querySelector('#posEditModal')); modal.show();
    if (!saveBtn._wired){
      saveBtn._wired = true;
      saveBtn.addEventListener('click', async ()=>{
        const payload = formToJSON(new FormData(form));
        try {
          const r = await fetch(`api/organization.php?entity=positions&id=${encodeURIComponent(saveBtn.dataset.id)}`, {
            method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload)
          });
          const out = await r.json();
          if (out?.success){ window.auth?.showNotification?.('Position updated', 'success'); modal.hide(); await loadPositions(container); try { await cachePositions(); } catch(_){ } }
          else { window.auth?.showNotification?.(out?.message || 'Update failed', 'error'); }
        } catch(_){ window.auth?.showNotification?.('Update failed', 'error'); }
      });
    }
  }

  function initMapTab(container){
    const mapPosSel = container.querySelector('#mapPosSelect');
    const mapBody = container.querySelector('#mapEmpTableBody');
    const mapSearch = container.querySelector('#mapSearch');
    const selectAll = container.querySelector('#mapSelectAll');
    const reloadBtn = container.querySelector('#mapReload');
    const assignBtn = container.querySelector('#assignSelected');
    if (!mapPosSel || !mapBody) return;
    const render = ()=>{
      mapPosSel.innerHTML = (_posCache||[]).map(p=>`<option value="${p.id}">${escapeHtml(p.title||('#'+p.id))}</option>`).join('');
      const term = (mapSearch?.value||'').toLowerCase();
      const rows = (_empCache||[])
        .filter(e=>{ const t = `${e.employee_number||''} ${(e.full_name||((e.first_name||'')+' '+(e.last_name||''))).toLowerCase()} ${(e.email||'').toLowerCase()}`.toLowerCase(); return !term || t.includes(term); })
        .map(e=>{
          const pos = e.position_title || '-';
          const dep = e.department_name || '-';
          return `<tr>
            <td><input type="checkbox" class="mapRowChk" data-id="${e.id}"/></td>
            <td>${escapeHtml(e.employee_number||'')}</td>
            <td>${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())}</td>
            <td>${escapeHtml(pos)}</td>
            <td>${escapeHtml(dep)}</td>
            <td>${statusBadge(e.employment_status||'active')}</td>
          </tr>`;
        }).join('');
      mapBody.innerHTML = rows || '<tr><td colspan="6" class="text-muted">No employees</td></tr>';
    };
    const refresh = async ()=>{ await ensureCaches(); render(); };
    mapSearch?.addEventListener('input', render);
    selectAll?.addEventListener('change', ()=>{ mapBody.querySelectorAll('.mapRowChk').forEach(ch=>{ ch.checked = !!selectAll.checked; }); });
    reloadBtn?.addEventListener('click', refresh);
    assignBtn?.addEventListener('click', async ()=>{
      const targetId = mapPosSel.value; if (!targetId) { window.auth?.showNotification?.('Select a target position','warning'); return; }
      const ids = Array.from(mapBody.querySelectorAll('.mapRowChk:checked')).map(ch=>ch.getAttribute('data-id'));
      if (!ids.length) { window.auth?.showNotification?.('Select at least one employee','warning'); return; }
      assignBtn.disabled = true;
      try {
        for (let i=0;i<ids.length;i++){
          const r = await fetch(`api/employees.php?id=${encodeURIComponent(ids[i])}`, { method:'PUT', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ position_id: targetId }) });
          await r.json();
        }
        window.auth?.showNotification?.('Employees assigned','success'); await cacheEmployees(); render(); await rebuildDistribution(container);
      } catch(_) { window.auth?.showNotification?.('Assignment failed','error'); } finally { assignBtn.disabled = false; }
    });
    // initial
    refresh();
  }

  function initDistributionTab(container){
    const reloadBtn = container.querySelector('#distReload');
    const showUn = container.querySelector('#showUnassigned');
    const content = container.querySelector('#distContent');
    const render = ()=>{
      const empByPos = new Map();
      (_posCache||[]).forEach(p=>empByPos.set(String(p.id), []));
      const unassigned = [];
      (_empCache||[]).forEach(e=>{ const key = e.position_id ? String(e.position_id) : null; if (key && empByPos.has(key)) empByPos.get(key).push(e); else unassigned.push(e); });
      let html = '';
      (_posCache||[]).forEach(p=>{
        const list = empByPos.get(String(p.id));
        html += `<div class="card mb-3">
          <div class="card-header d-flex justify-content-between align-items-center">
            <strong>${escapeHtml(p.title||('#'+p.id))}</strong>
            <span class="badge bg-primary">${list.length}</span>
          </div>
          ${list.length ? `<ul class="list-group list-group-flush">${list.map(e=>`<li class=\"list-group-item p-2\">${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())} <span class=\"text-muted\">${escapeHtml(e.employee_number||'')}</span></li>`).join('')}</ul>` : '<div class="p-3 text-muted">No employees</div>'}
        </div>`;
      });
      if (showUn?.checked) {
        html += `<div class="card mb-3 border-warning">
          <div class="card-header d-flex justify-content-between align-items-center">
            <strong>Unassigned</strong>
            <span class="badge bg-warning text-dark">${unassigned.length}</span>
          </div>
          ${unassigned.length ? `<ul class="list-group list-group-flush">${unassigned.map(e=>`<li class=\"list-group-item p-2\">${escapeHtml(e.full_name||(`${e.first_name||''} ${e.last_name||''}`).trim())}</li>`).join('')}</ul>` : '<div class="p-3 text-muted">None</div>'}
        </div>`;
      }
      content.innerHTML = html || '<div class="text-muted">No positions</div>';
    };
    const refresh = async ()=>{ await ensureCaches(); render(); };
    reloadBtn?.addEventListener('click', refresh);
    showUn?.addEventListener('change', render);
    refresh();
  }

  async function rebuildDistribution(container){
    try { await ensureCaches(); } catch(_){ }
    const content = container.querySelector('#distContent'); if (content) { const reloadBtn = container.querySelector('#distReload'); reloadBtn?.click(); if (!reloadBtn) { const evt = new Event('change'); container.querySelector('#showUnassigned')?.dispatchEvent(evt); } }
  }

  window.pages.positions = { render };
})();
