// Lightweight client-side logger with fetch instrumentation
(function () {
  const LOG_ENDPOINT = 'api/logs.php';
  const SKIP_LOG_URLS = [LOG_ENDPOINT];

  function nowMs() { return performance && performance.now ? performance.now() : Date.now(); }

  async function postLog(payload) {
    try {
      await window._origFetch(LOG_ENDPOINT, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        credentials: 'same-origin',
        body: JSON.stringify(payload)
      });
    } catch (_) {
      // Swallow logging failures
    }
  }

  // Public logger
  window.logEvent = function (level, message, context = {}) {
    const payload = {
      level, message,
      page: window.app?.currentPage || null,
      url: window.location.href,
      method: context.method || null,
      status: context.status || null,
      duration_ms: context.duration_ms || null,
      error: context.error || null,
      context,
      request_id: context.request_id || null,
    };
    postLog(payload);
  };

  // Preserve original fetch
  if (!window._origFetch) window._origFetch = window.fetch.bind(window);

  // Wrap fetch for auto-logging
  window.fetch = async function(input, init = {}) {
    try {
      const url = typeof input === 'string' ? input : (input?.url || '');
      const shouldSkip = SKIP_LOG_URLS.some(u => url.includes(u));
      const start = nowMs();
      let res;
      try {
        res = await window._origFetch(input, init);
        return res;
      } finally {
        const end = nowMs();
        if (!shouldSkip) {
          const rid = res?.headers?.get?.('X-Request-ID') || null;
          window.logEvent('info', 'api_call', {
            method: (init && init.method) || 'GET',
            status: res?.status || null,
            duration_ms: Math.round(end - start),
            request_id: rid,
            endpoint: url
          });
        }
      }
    } catch (e) {
      // Log critical fetch error
      try {
        const url = typeof input === 'string' ? input : (input?.url || '');
        if (!SKIP_LOG_URLS.some(u => url.includes(u))) {
          window.logEvent('error', 'api_call_failed', {
            method: (init && init.method) || 'GET',
            error: String(e?.message || e),
            endpoint: url
          });
        }
      } catch(_){ }
      throw e;
    }
  };

  // Global error handlers
  window.addEventListener('error', function (e) {
    try {
      const err = e?.error || {};
      window.logEvent('error', 'window_error', {
        error: String(err?.message || e?.message || 'Unknown error'),
        stack: String(err?.stack || ''),
        file: e?.filename || null,
        line: e?.lineno || null,
        column: e?.colno || null,
      });
    } catch(_) { }
  });

  window.addEventListener('unhandledrejection', function (e) {
    try {
      const reason = e?.reason || {};
      window.logEvent('error', 'unhandled_promise_rejection', {
        error: String(reason?.message || reason || 'Unknown rejection'),
        stack: String(reason?.stack || ''),
      });
    } catch(_) { }
  });
})();
