// HR Dashboard module
(function(){
  window.pages = window.pages || {};

  function escapeHtml(s){
    return String(s ?? '').replace(/[&<>"']/g, c=>({"&":"&amp;","<":"&lt;",">":"&gt;","\"":"&quot;","'":"&#39;"}[c]));
  }

  // ===== HR Analytics (KPIs + Charts) =====
  let hrCharts = { gender:null, dept:null, vac:null, leave:null, ret:null };
  function destroyChart(inst){ try{ inst && typeof inst.destroy === 'function' && inst.destroy(); }catch(_){} }
  function palette(n){
    const base = ['#1A3E73','#2CA58D','#F2C14E','#F78154','#7B6D8D','#4ECDC4','#EF476F','#06D6A0','#118AB2','#073B4C'];
    const out=[]; for(let i=0;i<n;i++){ out.push(base[i%base.length]); } return out;
  }
  async function loadHrStats(){
    const year = new Date().getFullYear();
    try{
      const r = await fetch(`api/analytics.php?action=hr_dashboard_stats&year=${year}`,{credentials:'same-origin'});
      const j = await r.json(); if (!j?.success) throw new Error('bad');
      const d = j.data || {};
      // KPIs
      const emp = d.employees||{}; const vac = d.vacancies||{};
      const setText = (id, val)=>{ const el=document.getElementById(id); if(el) el.textContent = String(val); };
      setText('hrKpiEmpTotal', emp.total ?? '-');
      setText('hrKpiEmpActive', emp.active ?? '-');
      setText('hrKpiOpenVac', vac.open_current ?? '-');
      setText('hrKpiFilledYear', vac.filled_this_year ?? '-');

      // Charts: Gender
      if (window.Chart && document.getElementById('hrChartGender')){
        const labels = (d.gender_distribution||[]).map(x=> (x.g||'unspecified').toString());
        const values = (d.gender_distribution||[]).map(x=> Number(x.c||0));
        const ctx = document.getElementById('hrChartGender').getContext('2d');
        destroyChart(hrCharts.gender);
        hrCharts.gender = new Chart(ctx, { type:'doughnut', data:{ labels, datasets:[{ data: values, backgroundColor: palette(values.length) }] }, options:{ plugins:{ legend:{ position:'bottom' } } } });
      }

      // Chart: Employee Retention (last 12 months)
      if (window.Chart && document.getElementById('hrChartRetention')){
        const r = d.retention || {labels:[], headcount_start:[], terminations:[], retention_rate:[]};
        const labels = r.labels || [];
        const ctx = document.getElementById('hrChartRetention').getContext('2d');
        destroyChart(hrCharts.ret);
        hrCharts.ret = new Chart(ctx, {
          type: 'bar',
          data: {
            labels,
            datasets: [
              { type: 'bar', label: 'Terminations', data: (r.terminations||[]).map(Number), backgroundColor: '#F78154', yAxisID: 'y' },
              { type: 'line', label: 'Retention %', data: (r.retention_rate||[]).map(Number), borderColor: '#1A3E73', backgroundColor: 'rgba(26,62,115,0.15)', tension: 0.2, yAxisID: 'y1' }
            ]
          },
          options: {
            responsive: true,
            plugins: { legend: { position: 'bottom' } },
            scales: {
              y: { beginAtZero: true, title: { display:true, text: 'Terminations' } },
              y1: { beginAtZero: true, max: 100, position: 'right', grid: { drawOnChartArea: false }, title: { display:true, text: 'Retention %' } }
            }
          }
        });
        // exits pending badge (if present)
        const badge = document.getElementById('hrExitPending');
        if (badge) badge.textContent = String(Number(d.exits_pending||0));
      }

      // Charts: Department (Top 10)
      if (window.Chart && document.getElementById('hrChartDept')){
        const sorted = (d.department_distribution||[]).slice().sort((a,b)=> Number(b.count||0)-Number(a.count||0));
        const top = sorted.slice(0,10);
        const labels = top.map(x=> x.name||'');
        const values = top.map(x=> Number(x.count||0));
        const ctx = document.getElementById('hrChartDept').getContext('2d');
        destroyChart(hrCharts.dept);
        hrCharts.dept = new Chart(ctx, { type:'bar', data:{ labels, datasets:[{ label:'Employees', data: values, backgroundColor: '#1A3E73' }] }, options:{ indexAxis:'y', plugins:{ legend:{ display:false } }, scales:{ x:{ beginAtZero:true } } } });
      }

      // Charts: Vacancies
      if (window.Chart && document.getElementById('hrChartVac')){
        const labels = ['Opened this year','Filled this year'];
        const values = [Number(vac.opened_this_year||0), Number(vac.filled_this_year||0)];
        const ctx = document.getElementById('hrChartVac').getContext('2d');
        destroyChart(hrCharts.vac);
        hrCharts.vac = new Chart(ctx, { type:'bar', data:{ labels, datasets:[{ label:'Count', data: values, backgroundColor: ['#F2C14E','#2CA58D'] }] }, options:{ plugins:{ legend:{ display:false } }, scales:{ y:{ beginAtZero:true, precision:0 } } } });
      }

      // Charts: Leave Types (Allowed vs Used vs Pending)
      if (window.Chart && document.getElementById('hrChartLeave')){
        const rows = Array.isArray(d.leave_types)? d.leave_types : [];
        // Limit to top 10 by allowed_total_days
        const top = rows.slice().sort((a,b)=> Number(b.allowed_total_days||0)-Number(a.allowed_total_days||0)).slice(0,10);
        const labels = top.map(x=> x.name||x.code||`Type ${x.leave_type_id}`);
        const allowed = top.map(x=> Number(x.allowed_total_days||0));
        const used = top.map(x=> Number(x.used_days||0));
        const pending = top.map(x=> Number(x.pending_days||0));
        // available = allowed - used - pending (not below 0)
        const available = allowed.map((a,i)=> Math.max(0, a - used[i] - pending[i]));
        const ctx = document.getElementById('hrChartLeave').getContext('2d');
        destroyChart(hrCharts.leave);
        hrCharts.leave = new Chart(ctx, { type:'bar', data:{ labels, datasets:[
          { label:'Used', data: used, backgroundColor:'#F78154' },
          { label:'Pending', data: pending, backgroundColor:'#F2C14E' },
          { label:'Available', data: available, backgroundColor:'#2CA58D' }
        ] }, options:{ responsive:true, plugins:{ legend:{ position:'bottom' } }, scales:{ x:{ stacked:true }, y:{ stacked:true, beginAtZero:true } } } });
      }

    }catch(_){ /* swallow to avoid breaking dashboard */ }
  }
  function formatDate(s){ try{ return new Date(String(s).replace(' ','T')).toLocaleDateString(); }catch(_){ return s||''; } }
  function formatDateTime(s){ try{ return new Date(String(s).replace(' ','T')).toLocaleString(); }catch(_){ return s||''; } }
  function greeting(){ const h=(new Date()).getHours(); if (h<12) return 'Good Morning'; if (h<17) return 'Good Afternoon'; return 'Good Evening'; }

  async function render(container){
    const user = window.auth?.currentUser || {};
    const firstName = user.first_name || 'HR';
    const year = new Date().getFullYear();

    container.innerHTML = `
      <div class="row">
        <div class="col-12">
          <div class="card mb-3">
            <div class="card-header d-flex justify-content-between align-items-center">
              <div>
                <h5 class="mb-0"><span id="greetText">${greeting()}</span>, ${escapeHtml(firstName)}</h5>
                <small class="text-muted" id="clockNow"></small>
              </div>
              <button class="btn btn-sm btn-outline-primary" id="dashHrRefresh"><i class="fas fa-rotate"></i> Refresh</button>
            </div>
            <div class="card-body py-2">
              <div id="annTicker" class="small text-nowrap overflow-hidden" style="white-space:nowrap;"></div>
            </div>
          </div>
        </div>
      </div>

      <div class="row g-3">
        <div class="col-xxl-3 col-md-6">
          <div class="card h-100"><div class="card-body">
            <div class="text-muted small">Employees (Total)</div>
            <div class="display-6" id="hrKpiEmpTotal">-</div>
          </div></div>
        </div>
        <div class="col-xxl-3 col-md-6">
          <div class="card h-100"><div class="card-body">
            <div class="text-muted small">Employees (Active)</div>
            <div class="display-6" id="hrKpiEmpActive">-</div>
          </div></div>
        </div>
        <div class="col-xxl-3 col-md-6">
          <div class="card h-100"><div class="card-body">
            <div class="text-muted small">Open Vacancies (Now)</div>
            <div class="display-6" id="hrKpiOpenVac">-</div>
          </div></div>
        </div>
        <div class="col-xxl-3 col-md-6">
          <div class="card h-100"><div class="card-body">
            <div class="text-muted small">Filled Vacancies (${year})</div>
            <div class="display-6" id="hrKpiFilledYear">-</div>
          </div></div>
        </div>
      </div>

      <div class="row g-3 mt-1">
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Gender Distribution</h6></div>
            <div class="card-body"><canvas id="hrChartGender" height="220"></canvas></div>
          </div>
        </div>
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Employees by Department</h6></div>
            <div class="card-body"><canvas id="hrChartDept" height="220"></canvas></div>
          </div>
        </div>
      </div>

      <div class="row g-3 mt-1">
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Vacancies (${year})</h6></div>
            <div class="card-body"><canvas id="hrChartVac" height="220"></canvas><div class="small text-muted mt-2">Open now shown as KPI above</div></div>
          </div>
        </div>
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Leave Types: Allowed vs Used (${year})</h6></div>
            <div class="card-body"><canvas id="hrChartLeave" height="220"></canvas></div>
          </div>
        </div>
      </div>

      <div class="row g-3 mt-1">
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Employee Retention (12 months)</h6>
              <span class="badge bg-warning text-dark" title="Pending Exit Requests">Pending exits: <span id="hrExitPending">0</span></span>
            </div>
            <div class="card-body"><canvas id="hrChartRetention" height="220"></canvas></div>
          </div>
        </div>
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Pending Exit Requests</h6></div>
            <div class="card-body p-0"><ul class="list-group list-group-flush" id="hrExitList"><li class="list-group-item text-muted">Loading...</li></ul></div>
          </div>
        </div>
      </div>

      <div class="row g-3">
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header"><h6 class="mb-0">My Profile</h6></div>
            <div class="card-body" id="hrProfile">Loading...</div>
          </div>
        </div>
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Recent Documents (All-staff)</h6>
              <a href="#/documents" onclick="app.showPage('documents');return false;" class="small">View all</a>
            </div>
            <div class="card-body p-0">
              <ul class="list-group list-group-flush" id="hrDocsList"><li class="list-group-item text-muted">Loading...</li></ul>
            </div>
          </div>
        </div>
        <div class="col-lg-4">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">My Messages</h6><a href="#/messages" onclick="app.showPage('messages');return false;" class="small">Open</a></div>
            <div class="card-body p-0"><ul class="list-group list-group-flush" id="hrMsgList"><li class="list-group-item text-muted">Loading...</li></ul></div>
          </div>
        </div>
      </div>

      <div class="row g-3 mt-1">
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Upcoming Birthdays (3 days)</h6><small id="hrBdRange" class="text-muted"></small></div>
            <div class="card-body p-0"><ul class="list-group list-group-flush" id="hrBdList"><li class="list-group-item text-muted">Loading...</li></ul></div>
          </div>
        </div>
        <div class="col-lg-6">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Upcoming Work Anniversaries (3 days)</h6><small id="hrAnRange" class="text-muted"></small></div>
            <div class="card-body p-0"><ul class="list-group list-group-flush" id="hrAnList"><li class="list-group-item text-muted">Loading...</li></ul></div>
          </div>
        </div>
      </div>

      <div class="row g-3 mt-1">
        <div class="col-lg-5">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center"><h6 class="mb-0">Attendance Stats (${year})</h6><small class="text-muted" id="hrAttScope"></small></div>
            <div class="card-body" id="hrAttStats">Loading...</div>
          </div>
        </div>
        <div class="col-lg-7">
          <div class="card h-100">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h6 class="mb-0">Company Calendar</h6>
              <div>
                <button class="btn btn-sm btn-outline-secondary" id="hrCalPrev"><i class="fas fa-chevron-left"></i></button>
                <span class="mx-2" id="hrCalTitle"></span>
                <button class="btn btn-sm btn-outline-secondary" id="hrCalNext"><i class="fas fa-chevron-right"></i></button>
              </div>
            </div>
            <div class="card-body" id="hrHolidayCalendarWrap"></div>
          </div>
        </div>
      </div>
    `;

    // wire
    document.getElementById('dashHrRefresh')?.addEventListener('click', ()=>reloadAll());
    startClock();
    await reloadAll();
    initCalendar();
  }

  function startClock(){
    const elClock = document.getElementById('clockNow');
    const elGreet = document.getElementById('greetText');
    function tick(){ if (elClock) elClock.textContent = new Date().toLocaleString(); if (elGreet) elGreet.textContent = greeting(); }
    tick(); setInterval(tick, 1000);
  }

  async function reloadAll(){
    await Promise.all([
      loadAnnouncementsTicker(),
      loadProfile(),
      loadRecentDocs(),
      loadMessages(),
      loadUpcomingPeople(),
      loadAttendanceStats(),
      loadHrStats(),
      loadPendingExits()
    ]);
  }

  async function loadPendingExits(){
    const el = document.getElementById('hrExitList');
    if (!el) return;
    try{
      const r = await fetch('api/exits.php?status=pending', { credentials:'same-origin' });
      const j = await r.json();
      const rows = j?.success ? (j.data||[]) : [];
      if (!rows.length){ el.innerHTML = '<li class="list-group-item text-muted">None</li>'; return; }
      const canAct = !!window.auth?.canApprove?.('exit');
      el.innerHTML = rows.slice(0,8).map(x=>{
        const when = x.proposed_last_day ? new Date(String(x.proposed_last_day)).toLocaleDateString() : '-';
        const act = canAct ? `<div class="ms-auto">
            <button class="btn btn-sm btn-success me-1" data-exit-approve="${x.id}">Approve</button>
            <button class="btn btn-sm btn-outline-danger" data-exit-reject="${x.id}">Reject</button>
          </div>` : '';
        return `<li class="list-group-item d-flex align-items-center">
          <div>
            <div class="fw-semibold">${escapeHtml(x.employee_name||('#'+x.employee_id))}</div>
            <div class="small text-muted">Dept: ${escapeHtml(x.department_name||'-')} • Last day: ${escapeHtml(when)}</div>
          </div>
          ${act}
        </li>`;
      }).join('');

      // wire actions once
      if (!el._wired){
        el._wired = true;
        el.addEventListener('click', async (ev)=>{
          const a = ev.target.closest('[data-exit-approve]');
          const rj = ev.target.closest('[data-exit-reject]');
          if (a){
            const id = Number(a.getAttribute('data-exit-approve'));
            const notes = prompt('Approval notes (optional):','');
            try{
              const resp = await fetch(`api/exits.php?action=approve&id=${id}`, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ notes }) });
              const jj = await resp.json();
              if (jj?.success){ window.auth?.showNotification?.('Exit approved','success'); loadPendingExits(); }
              else { window.auth?.showNotification?.(jj?.message||'Failed','error'); }
            }catch(_){ window.auth?.showNotification?.('Failed','error'); }
          } else if (rj){
            const id = Number(rj.getAttribute('data-exit-reject'));
            const reason = prompt('Rejection reason (required):','');
            if (!reason) return;
            try{
              const resp = await fetch(`api/exits.php?action=reject&id=${id}`, { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ rejection_reason: reason }) });
              const jj = await resp.json();
              if (jj?.success){ window.auth?.showNotification?.('Exit rejected','info'); loadPendingExits(); }
              else { window.auth?.showNotification?.(jj?.message||'Failed','error'); }
            }catch(_){ window.auth?.showNotification?.('Failed','error'); }
          }
        });
      }
    }catch(_){ if (el) el.innerHTML = '<li class="list-group-item text-danger">Failed to load</li>'; }
  }

  async function loadAnnouncementsTicker(){
    const wrap = document.getElementById('annTicker'); if (!wrap) return;
    wrap.textContent = 'Loading announcements...';
    try{
      const res = await fetch('api/announcements.php?action=latest&limit=10',{credentials:'same-origin'});
      const js = await res.json();
      const items = js?.success ? (js.data||[]) : [];
      if (!items.length){ wrap.innerHTML = '<span class="text-muted">No announcements</span>'; return; }
      const text = items.map(a=>`<span class="me-4"><strong>${escapeHtml(a.title||'')}</strong>: ${escapeHtml(a.content||'')}</span>`).join('');
      wrap.innerHTML = `<div id="annMarquee" style="display:inline-block; animation: scrollLeft 30s linear infinite;">${text}</div>`;
      injectMarqueeStyle();
    }catch(_){ wrap.innerHTML = '<span class="text-danger">Failed to load</span>'; }
  }
  function injectMarqueeStyle(){
    if (document.getElementById('mq-style')) return;
    const s = document.createElement('style'); s.id='mq-style'; s.textContent = `@keyframes scrollLeft {0%{transform:translateX(0)}100%{transform:translateX(-50%)}}`;
    document.head.appendChild(s);
  }

  async function loadProfile(){
    const el = document.getElementById('hrProfile'); if (!el) return;
    try{
      const r = await fetch('api/analytics.php?action=people',{credentials:'same-origin'});
      const j = await r.json();
      if (!j?.success){ el.innerHTML = '<div class="text-danger">Failed</div>'; return; }
      const p = j.data?.profile||{}; const roles = Array.isArray(p.roles)? p.roles.map(r=>r.name).join(', '):'';
      el.innerHTML = `
        <div class="d-flex align-items-center gap-3">
          <img class="rounded-circle" src="${escapeHtml(window.auth?.currentUser?.avatar||'assets/img/avatar-default.png')}" width="56" height="56" alt="avatar">
          <div>
            <div class="fw-bold">${escapeHtml((p.first_name||'')+' '+(p.last_name||''))}</div>
            <div class="text-muted small">${escapeHtml(roles||'')}</div>
            <div class="text-muted small">Department: ${escapeHtml(p.department_name||'--')}</div>
          </div>
        </div>`;
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed</div>'; }
  }

  async function loadRecentDocs(){
    const el = document.getElementById('hrDocsList'); if (!el) return;
    try{
      const r = await fetch('api/documents.php?action=recent&days=7&all_only=1',{credentials:'same-origin'});
      const j = await r.json(); const rows = j?.success ? (j.data||[]) : [];
      el.innerHTML = rows.slice(0,8).map(d=>`<li class="list-group-item d-flex justify-content-between align-items-center">
        <div><div class="fw-semibold">${escapeHtml(d.title||'')}</div><div class="small text-muted">${escapeHtml(d.category||'')}${d.ack_required? ' • Ack required':''}</div></div>
        <div class="text-end small">
          <div>${escapeHtml(formatDateTime(d.created_at))}</div>
          <a href="api/documents.php?action=download&id=${d.id}" class="btn btn-sm btn-outline-primary"><i class="fas fa-download"></i></a>
        </div>
      </li>`).join('') || '<li class="list-group-item text-muted">No recent documents</li>';
    }catch(_){ el.innerHTML = '<li class="list-group-item text-danger">Failed to load</li>'; }
  }

  async function loadMessages(){
    const el = document.getElementById('hrMsgList'); if (!el) return;
    try{
      const r = await fetch('api/messages.php?action=inbox',{credentials:'same-origin'});
      const j = await r.json(); const rows = j?.success ? (j.data||[]) : [];
      el.innerHTML = rows.slice(0,5).map(m=>`<li class="list-group-item">
        <div class="fw-semibold">${escapeHtml(m.subject||'Message')}</div>
        <div class="small text-muted">from ${escapeHtml((m.sender_first_name||'')+' '+(m.sender_last_name||m.sender_username||''))} • ${escapeHtml(formatDateTime(m.created_at))}</div>
        <div class="small">${escapeHtml(String(m.content||'').slice(0,120))}${String(m.content||'').length>120?'...':''}</div>
      </li>`).join('') || '<li class="list-group-item text-muted">No messages</li>';
    }catch(_){ el.innerHTML = '<li class="list-group-item text-danger">Failed to load</li>'; }
  }

  async function loadUpcomingPeople(){
    const bdEl = document.getElementById('hrBdList'); const anEl = document.getElementById('hrAnList');
    try{
      const r = await fetch('api/analytics.php?action=people_window&days=3',{credentials:'same-origin'});
      const j = await r.json(); if (!j?.success){ throw new Error('bad'); }
      const b = Array.isArray(j.data?.birthdays)? j.data.birthdays:[]; const a = Array.isArray(j.data?.anniversaries)? j.data.anniversaries:[];
      const end = j.data?.end; if (document.getElementById('hrBdRange')) document.getElementById('hrBdRange').textContent = `until ${formatDate(end)}`;
      if (document.getElementById('hrAnRange')) document.getElementById('hrAnRange').textContent = `until ${formatDate(end)}`;
      bdEl.innerHTML = b.length? b.map(x=>`<li class="list-group-item d-flex justify-content-between align-items-center">
        <div><div class="fw-semibold">${escapeHtml((x.first_name||'')+' '+(x.last_name||''))}</div><div class="small text-muted">${escapeHtml(x.department_name||'')} • ${formatDate(x.occurs_on)}</div></div>
        ${x.user_id? `<button class="btn btn-sm btn-outline-primary" data-wish="${x.user_id}" data-name="${escapeHtml((x.first_name||'')+' '+(x.last_name||''))}" data-type="birthday">Send wish</button>`:''}
      </li>`).join('') : '<li class="list-group-item text-muted">None</li>';
      anEl.innerHTML = a.length? a.map(x=>`<li class="list-group-item d-flex justify-content-between align-items-center">
        <div><div class="fw-semibold">${escapeHtml((x.first_name||'')+' '+(x.last_name||''))}</div><div class="small text-muted">${escapeHtml(x.department_name||'')} • ${formatDate(x.occurs_on)} • ${Number(x.years||0)} yr</div></div>
        ${x.user_id? `<button class="btn btn-sm btn-outline-primary" data-wish="${x.user_id}" data-name="${escapeHtml((x.first_name||'')+' '+(x.last_name||''))}" data-type="anniversary">Send note</button>`:''}
      </li>`).join('') : '<li class="list-group-item text-muted">None</li>';
      // wire wish senders
      document.querySelectorAll('[data-wish]')?.forEach(btn=>{
        if (btn._wishBound) return; btn._wishBound = true;
        btn.addEventListener('click', async ()=>{
          const uid = Number(btn.getAttribute('data-wish'));
          const type = btn.getAttribute('data-type');
          const name = btn.getAttribute('data-name');
          const placeholder = type==='birthday' ? `Write a birthday wish to ${name}` : `Write an anniversary message to ${name}`;
          const msg = prompt(placeholder, type==='birthday' ? 'Happy Birthday!' : 'Happy Work Anniversary!');
          if (!msg) return;
          try{
            const r = await fetch('api/messages.php?action=send',{method:'POST',headers:{'Content-Type':'application/json'},credentials:'same-origin',body:JSON.stringify({recipient_user_id:uid, subject:type==='birthday'?'Happy Birthday':'Happy Work Anniversary', content:msg})});
            const j = await r.json();
            if (j?.success) window.auth?.showNotification?.('Message sent','success'); else window.auth?.showNotification?.(j?.message||'Failed','error');
          }catch(_){ window.auth?.showNotification?.('Failed','error'); }
        });
      });
    }catch(_){ if (bdEl) bdEl.innerHTML='<li class="list-group-item text-danger">Failed to load</li>'; if (anEl) anEl.innerHTML='<li class="list-group-item text-danger">Failed to load</li>'; }
  }

  async function loadAttendanceStats(){
    const el = document.getElementById('hrAttStats'); const scopeEl = document.getElementById('hrAttScope'); if (!el) return;
    try{
      const r = await fetch('api/attendance.php?action=stats_year',{credentials:'same-origin'});
      const j = await r.json(); if (!j?.success){ throw new Error('bad'); }
      const d = j.data||{}; if (scopeEl) scopeEl.textContent = (d.scope||'').toUpperCase();
      el.innerHTML = `
        <div class="row text-center">
          <div class="col-4"><div class="h4">${Number(d.clock_in_total ?? d.clock_in ?? 0)}</div><div class="text-muted small">Clock-ins</div></div>
          <div class="col-4"><div class="h4">${Number(d.clock_out_total ?? d.clock_out ?? 0)}</div><div class="text-muted small">Clock-outs</div></div>
          <div class="col-4"><div class="h4">${Number(d.records||0)}</div><div class="text-muted small">Records</div></div>
        </div>`;
    }catch(_){ el.innerHTML = '<div class="text-danger">Failed</div>'; }
  }

  // Calendar copied & adapted from main dashboard
  let hrCalState = null;
  function initCalendar(){
    const wrap = document.getElementById('hrHolidayCalendarWrap'); if (!wrap) return;
    const now = new Date(); hrCalState = {year: now.getFullYear(), month: now.getMonth()};
    document.getElementById('hrCalPrev')?.addEventListener('click', ()=>shift(-1));
    document.getElementById('hrCalNext')?.addEventListener('click', ()=>shift(1));
    renderCal();
  }
  function shift(d){ if (!hrCalState) return; let y=hrCalState.year, m=hrCalState.month+d; while(m<0){m+=12;y--;} while(m>11){m-=12;y++;} hrCalState.year=y; hrCalState.month=m; renderCal(); }
  async function renderCal(){
    const wrap = document.getElementById('hrHolidayCalendarWrap'); const title = document.getElementById('hrCalTitle'); if (!wrap||!title||!hrCalState) return;
    const y=hrCalState.year, m=hrCalState.month; const first = new Date(y,m,1); const last = new Date(y,m+1,0);
    title.textContent = first.toLocaleString(undefined,{month:'long',year:'numeric'});
    const from = `${y}-${String(m+1).padStart(2,'0')}-01`; const to = `${y}-${String(m+1).padStart(2,'0')}-${String(last.getDate()).padStart(2,'0')}`;
    let holidays = {}; let eventsByDate = {};
    try{
      const rh = await fetch(`api/holidays.php?action=range&from=${from}&to=${to}`,{credentials:'same-origin'}); const jh = await rh.json(); if (jh?.success) jh.data.forEach(r=>{ holidays[r.holiday_date]=r.name||'Holiday'; });
      const re = await fetch(`api/calendar.php?action=range&from=${from}&to=${to}`,{credentials:'same-origin'}); const je = await re.json(); if (je?.success) je.data.forEach(ev=>{ const d = new Date(String(ev.start_at||'').replace(' ','T')); if (!isNaN(d)){ const key = `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,'0')}-${String(d.getDate()).padStart(2,'0')}`; (eventsByDate[key]=eventsByDate[key]||[]).push({title:ev.title||'Event'}); } });
    }catch(_){}
    const startDow = (first.getDay()+6)%7; const days=last.getDate(); const cells=[]; for(let i=0;i<startDow;i++) cells.push(null); for(let d=1; d<=days; d++) cells.push(new Date(y,m,d)); while(cells.length%7!==0) cells.push(null);
    const head=['Mon','Tue','Wed','Thu','Fri','Sat','Sun']; let html='<div class="table-responsive"><table class="table table-bordered table-sm mb-0"><thead><tr>'+head.map(h=>`<th class="text-center">${h}</th>`).join('')+'</tr></thead><tbody>';
    for(let i=0;i<cells.length;i+=7){ html+='<tr>'; for(let j=0;j<7;j++){ const dt=cells[i+j]; if(!dt){ html+='<td class="bg-light"></td>'; continue; } const key=`${dt.getFullYear()}-${String(dt.getMonth()+1).padStart(2,'0')}-${String(dt.getDate()).padStart(2,'0')}`; const isWeekend = dt.getDay()===0||dt.getDay()===6; const badgeHol = holidays[key]? `<div class="badge bg-danger w-100">${escapeHtml(holidays[key])}</div>` : (isWeekend? '<div class="badge bg-secondary w-100">Weekend</div>' : ''); const evs = eventsByDate[key]||[]; let evHtml=''; if (evs.length){ const max=2; evs.slice(0,max).forEach(e=>{ evHtml+=`<div class="badge bg-primary w-100 text-truncate" title="${escapeHtml(e.title)}">${escapeHtml(e.title)}</div>`; }); if (evs.length>max){ evHtml+=`<div class="small text-primary">+${evs.length-max} more</div>`; } } html+=`<td style="vertical-align:top;min-width:116px"><div class="small fw-bold">${dt.getDate()}</div>${badgeHol}${evHtml}</td>`; } html+='</tr>'; }
    html+='</tbody></table></div>'; wrap.innerHTML = html;
  }

  window.pages.dashboard_hr = { render };
})();
