// Company Settings page (Admin only)
(function(){
  window.pages = window.pages || {};

  function isAdmin(){
    const u = window.auth?.currentUser; const r = u?.role_slug || u?.role; return r === 'admin' || r === 'super_admin';
  }

  async function render(container){
    if (!isAdmin()){
      container.innerHTML = '<div class="alert alert-warning">Access denied. Company Settings are available to Admins only.</div>';
      return;
    }

    container.innerHTML = `
      <div class="row g-3">
        <div class="col-lg-7">
          <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
              <h5 class="mb-0">Company Profile</h5>
              <small class="text-muted">SmartQuantumHR</small>
            </div>
            <div class="card-body">
              <div class="row g-3">
                <div class="col-md-6">
                  <label class="form-label">Company Name</label>
                  <input class="form-control" id="companyName" placeholder="e.g. SmartQuantumHR Ltd" />
                </div>
                <div class="col-md-6">
                  <label class="form-label">Email</label>
                  <input class="form-control" id="companyEmail" placeholder="e.g. info@company.com" />
                </div>
                <div class="col-12">
                  <label class="form-label">Address</label>
                  <textarea class="form-control" id="companyAddress" rows="3" placeholder="Street, City, Region"></textarea>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Phone</label>
                  <input class="form-control" id="companyPhone" placeholder="e.g. +233 .." />
                </div>
                <div class="col-md-6">
                  <label class="form-label">Website</label>
                  <input class="form-control" id="companyWebsite" placeholder="https://..." />
                </div>
              </div>
              <div class="mt-3 d-flex gap-2">
                <button class="btn btn-primary" id="btnSaveCompany">Save Changes</button>
                <button class="btn btn-outline-secondary" id="btnResetCompany">Reset</button>
              </div>
            </div>
          </div>
        </div>
        <div class="col-lg-5">
          <div class="card h-100">
            <div class="card-header"><h6 class="mb-0">Branding</h6></div>
            <div class="card-body">
              <div class="mb-3">
                <label class="form-label">Current Logo</label>
                <div class="d-flex align-items-center gap-3">
                  <img id="companyLogoPreview" src="assets/img/smartquantumhr-logo.svg" alt="Company Logo" style="height:56px;border-radius:6px;border:1px solid #e5e7eb;background:#fff;padding:6px" />
                  <button class="btn btn-sm btn-outline-danger" id="btnClearLogo" title="Remove Logo">Remove</button>
                </div>
              </div>
              <div class="mb-3">
                <label class="form-label">Upload New Logo</label>
                <div class="input-group">
                  <input type="file" class="form-control" id="companyLogoFile" accept="image/*" />
                  <button class="btn btn-outline-primary" id="btnUploadLogo">Upload</button>
                </div>
                <div class="form-text">Supported: PNG, JPG, GIF, WEBP, SVG. Max 5MB.</div>
              </div>
              <div class="mb-3">
                <label class="form-label">Or set Logo by URL</label>
                <input class="form-control" id="companyLogoUrl" placeholder="https://... or /path or data:image/..." />
                <div class="mt-2 d-flex gap-2">
                  <button class="btn btn-primary btn-sm" id="btnSaveLogoUrl">Save URL</button>
                  <button class="btn btn-outline-secondary btn-sm" id="btnPreviewLogoUrl">Preview</button>
                </div>
                <div class="form-text">You can use absolute URLs, root-relative paths, app-relative paths, or data URIs.</div>
              </div>
              <hr/>
              <div>
                <label class="form-label">Primary Color</label>
                <input type="color" class="form-control form-control-color" value="#1A3E73" disabled />
                <div class="form-text">Brand color is managed via assets/css/brand.css</div>
              </div>
            </div>
          </div>
        </div>
      </div>`;

    const saveBtn = container.querySelector('#btnSaveCompany');
    const resetBtn = container.querySelector('#btnResetCompany');

    async function loadCompany(){
      try {
        const res = await fetch('api/settings.php?action=company', { credentials:'same-origin' });
        const json = await res.json();
        if (json?.success && json.data){
          const d = json.data;
          container.querySelector('#companyName').value = d.name ?? '';
          container.querySelector('#companyEmail').value = d.email ?? '';
          container.querySelector('#companyAddress').value = d.address ?? '';
          container.querySelector('#companyPhone').value = d.phone ?? '';
          container.querySelector('#companyWebsite').value = d.website ?? '';
          const logo = (d.logo ?? '').trim();
          setLogoPreview(logo || 'assets/img/smartquantumhr-logo.svg');
          const urlField = container.querySelector('#companyLogoUrl');
          if (urlField) urlField.value = logo || '';
        }
      } catch(_) { /* ignore */ }
    }

    async function saveCompany(){
      const payload = {
        name: container.querySelector('#companyName').value.trim(),
        email: container.querySelector('#companyEmail').value.trim(),
        address: container.querySelector('#companyAddress').value.trim(),
        phone: container.querySelector('#companyPhone').value.trim(),
        website: container.querySelector('#companyWebsite').value.trim(),
      };
      try {
        const res = await fetch('api/settings.php?action=company', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload) });
        const json = await res.json();
        if (json?.success){ window.auth?.showNotification?.('Company updated','success'); }
        else { window.auth?.showNotification?.(json?.message || 'Failed to save','error'); }
      } catch(e){ window.auth?.showNotification?.('Failed to save','error'); }
    }

    function setLogoPreview(url){
      try {
        const base = window.location.href.split('#')[0];
        const resolved = url ? new URL(url, base).href : 'assets/img/smartquantumhr-logo.svg';
        const img = container.querySelector('#companyLogoPreview');
        if (img) img.src = resolved;
      } catch(_) {
        const img = container.querySelector('#companyLogoPreview');
        if (img) img.src = url || 'assets/img/smartquantumhr-logo.svg';
      }
    }

    async function uploadLogo(){
      const fileInput = container.querySelector('#companyLogoFile');
      const f = fileInput?.files && fileInput.files[0];
      if (!f) { window.auth?.showNotification?.('Choose a logo file first','warning'); return; }
      const fd = new FormData();
      fd.append('logo', f);
      try {
        const res = await fetch('api/settings.php?action=company_logo', { method:'POST', body: fd, credentials:'same-origin' });
        const json = await res.json();
        if (json?.success){
          const newLogo = json.data?.logo || '';
          setLogoPreview(newLogo || 'assets/img/smartquantumhr-logo.svg');
          const urlField = container.querySelector('#companyLogoUrl');
          if (urlField) urlField.value = newLogo || '';
          fileInput.value = '';
          window.auth?.showNotification?.('Logo updated','success');
        } else {
          window.auth?.showNotification?.(json?.message || 'Upload failed','error');
        }
      } catch(e){ window.auth?.showNotification?.('Upload failed','error'); }
    }

    async function clearLogo(){
      if (!confirm('Remove current logo?')) return;
      try {
        const res = await fetch('api/settings.php?action=company_logo_clear', { method:'POST', credentials:'same-origin' });
        const json = await res.json();
        if (json?.success){
          setLogoPreview('assets/img/smartquantumhr-logo.svg');
          const urlField = container.querySelector('#companyLogoUrl');
          if (urlField) urlField.value = '';
          window.auth?.showNotification?.('Logo removed','info');
        } else {
          window.auth?.showNotification?.(json?.message || 'Failed to remove logo','error');
        }
      } catch(e){ window.auth?.showNotification?.('Failed to remove logo','error'); }
    }

    async function saveLogoUrl(){
      const url = (container.querySelector('#companyLogoUrl')?.value || '').trim();
      if (!url){ window.auth?.showNotification?.('Enter a URL or use Upload','warning'); return; }
      try {
        const res = await fetch('api/settings.php?action=company', { method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify({ logo: url }) });
        const json = await res.json();
        if (json?.success){ setLogoPreview(url); window.auth?.showNotification?.('Logo URL saved','success'); }
        else { window.auth?.showNotification?.(json?.message || 'Failed to save logo URL','error'); }
      } catch(e){ window.auth?.showNotification?.('Failed to save logo URL','error'); }
    }

    function previewLogoUrl(){
      const url = (container.querySelector('#companyLogoUrl')?.value || '').trim();
      setLogoPreview(url || 'assets/img/smartquantumhr-logo.svg');
    }

    if (saveBtn) saveBtn.addEventListener('click', saveCompany);
    if (resetBtn) resetBtn.addEventListener('click', loadCompany);
    container.querySelector('#btnUploadLogo')?.addEventListener('click', uploadLogo);
    container.querySelector('#btnClearLogo')?.addEventListener('click', clearLogo);
    container.querySelector('#btnSaveLogoUrl')?.addEventListener('click', saveLogoUrl);
    container.querySelector('#btnPreviewLogoUrl')?.addEventListener('click', previewLogoUrl);

    // initial load
    loadCompany();
  }

  window.pages.company_settings = { render };
})();
