// Appraisal Templates & KPIs management (HR/Admin)
(function(){
  window.pages = window.pages || {};

  function canManage(){
    const role = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head','hr_officer'].includes(role);
  }

  async function render(container){
    if (!canManage()){
      container.innerHTML = `<div class="card"><div class="card-body"><div class="text-muted">You do not have permission to manage appraisal templates.</div></div></div>`;
      return;
    }

    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Appraisal Templates</h5>
          <div class="d-flex gap-2">
            <button class="btn btn-sm btn-outline-secondary" id="tplReload"><i class="fas fa-rotate"></i> Reload</button>
          </div>
        </div>
        <div class="card-body">
          <div class="row g-3 align-items-end">
            <div class="col-md-4">
              <label class="form-label">Template Name</label>
              <input type="text" id="tplName" class="form-control" placeholder="e.g., Sales Annual Review" />
            </div>
            <div class="col-md-4">
              <label class="form-label">Department (optional)</label>
              <select id="tplDept" class="form-select"><option value="">-- Any --</option></select>
              <div class="form-text">Leave empty to make template available to all departments.</div>
            </div>
            <div class="col-md-4 text-end">
              <button class="btn btn-primary" id="tplCreate"><i class="fas fa-plus"></i> Create Template</button>
            </div>
          </div>
          <hr/>
          <div id="tplList">Loading...</div>
        </div>
      </div>

      <div class="modal fade" id="kpiEditModal" tabindex="-1">
        <div class="modal-dialog">
          <div class="modal-content">
            <div class="modal-header"><h5 class="modal-title">Edit KPI</h5><button class="btn-close" data-bs-dismiss="modal"></button></div>
            <div class="modal-body">
              <form id="kpiEditForm" class="row g-3">
                <div class="col-8"><label class="form-label">Title</label><input class="form-control" name="title" required /></div>
                <div class="col-4"><label class="form-label">Weight</label><input type="number" step="0.01" min="0" max="100" class="form-control" name="weight" required /></div>
              </form>
            </div>
            <div class="modal-footer">
              <button class="btn btn-light" data-bs-dismiss="modal">Close</button>
              <button class="btn btn-primary" id="kpiSaveBtn" data-id="">Save</button>
            </div>
          </div>
        </div>
      </div>
    `;

    const deptSel = container.querySelector('#tplDept');
    try {
      // Reuse organization API like Departments page
      const res = await fetch('api/organization.php?entity=departments', { credentials:'same-origin' });
      const js = await res.json();
      const opts = ['<option value="">-- Any --</option>']
        .concat((js?.data||[]).map(d=> `<option value="${d.id}">${esc(d.name)}</option>`));
      deptSel.innerHTML = opts.join('');
    } catch(_) {
      // keep default option
    }

    container.querySelector('#tplCreate').addEventListener('click', async ()=>{
      const name = container.querySelector('#tplName').value.trim();
      const dept = container.querySelector('#tplDept').value;
      if (!name){ note('Name is required','warning'); return; }
      const payload = { name, department_id: dept || '' };
      const res = await fetch('api/appraisal.php?action=template_create', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (res?.success){ note('Template created','success'); container.querySelector('#tplName').value=''; await loadTemplates(); }
      else { note(res?.message||'Create failed','error'); }
    });

    container.querySelector('#tplReload').addEventListener('click', loadTemplates);

    async function loadTemplates(){
      const wrap = container.querySelector('#tplList');
      wrap.innerHTML = 'Loading...';
      try{
        const res = await fetch('api/appraisal.php?action=templates', { credentials:'same-origin' });
        const js = await res.json();
        if (!js?.success){ wrap.innerHTML = '<div class="text-danger">Failed to load</div>'; return; }
        const list = js.data || [];
        if (!list.length){ wrap.innerHTML = '<div class="text-muted">No templates</div>'; return; }
        wrap.innerHTML = list.map(tpl=> templateCard(tpl)).join('');
        // wire actions
        wrap.querySelectorAll('[data-action]').forEach(btn=>{
          btn.addEventListener('click', async ()=>{
            const act = btn.getAttribute('data-action');
            const tid = btn.getAttribute('data-id');
            if (act==='delete'){ if(!confirm('Delete this template?')) return; await doTemplateDelete(tid); await loadTemplates(); return; }
            if (act==='edit-name'){
              const name = prompt('New name', btn.getAttribute('data-name')||''); if (!name) return;
              await doTemplateUpdate(tid, { name }); await loadTemplates(); return;
            }
            if (act==='edit-dept'){
              const dept = prompt('Department ID (blank for Any)', btn.getAttribute('data-dept')||'');
              await doTemplateUpdate(tid, { department_id: dept }); await loadTemplates(); return;
            }
            if (act==='kpi-add'){
              const title = prompt('KPI title'); if (!title) return;
              const weight = prompt('Weight (0-100)', '0') || '0';
              await doKpiCreate(tid, { title, weight }); await loadTemplates(); return;
            }
            if (act==='kpi-edit'){
              const id = btn.getAttribute('data-kpi');
              openKpiEdit(id, btn.getAttribute('data-title')||'', btn.getAttribute('data-weight')||'');
              return;
            }
            if (act==='kpi-delete'){
              const id = btn.getAttribute('data-kpi');
              if (!confirm('Delete KPI?')) return;
              await doKpiDelete(id); await loadTemplates(); return;
            }
          });
        });
      }catch(_){ wrap.innerHTML = '<div class="text-danger">Failed to load</div>'; }
    }

    function templateCard(t){
      const kpis = (t.kpis||[]);
      const rows = kpis.map(k=>`
        <tr>
          <td>${esc(k.title||'')}</td>
          <td>${esc(k.weight??'')}</td>
          <td class="text-end">
            <div class="btn-group btn-group-sm">
              <button class="btn btn-outline-primary" data-action="kpi-edit" data-id="${t.id}" data-kpi="${k.id}" data-title="${escAttr(k.title||'')}" data-weight="${escAttr(k.weight??'')}"><i class="fas fa-pen"></i></button>
              <button class="btn btn-outline-danger" data-action="kpi-delete" data-id="${t.id}" data-kpi="${k.id}"><i class="fas fa-trash"></i></button>
            </div>
          </td>
        </tr>
      `).join('');
      return `
        <div class="card mb-3">
          <div class="card-header d-flex justify-content-between align-items-center">
            <div>
              <strong>${esc(t.name)}</strong>
              <span class="text-muted ms-2">${t.department_name? esc(t.department_name):'Any'}</span>
            </div>
            <div class="btn-group btn-group-sm">
              <button class="btn btn-outline-primary" data-action="edit-name" data-id="${t.id}" data-name="${escAttr(t.name)}"><i class="fas fa-pen"></i> Name</button>
              <button class="btn btn-outline-primary" data-action="edit-dept" data-id="${t.id}" data-dept="${escAttr(t.department_id||'')}"><i class="fas fa-building"></i> Dept</button>
              <button class="btn btn-outline-danger" data-action="delete" data-id="${t.id}"><i class="fas fa-trash"></i></button>
            </div>
          </div>
          <div class="card-body">
            <div class="d-flex justify-content-between align-items-center mb-2">
              <h6 class="mb-0">KPIs</h6>
              <button class="btn btn-sm btn-outline-primary" data-action="kpi-add" data-id="${t.id}"><i class="fas fa-plus"></i> Add KPI</button>
            </div>
            <div class="table-responsive">
              <table class="table table-sm align-middle">
                <thead><tr><th>Title</th><th>Weight</th><th class="text-end">Actions</th></tr></thead>
                <tbody>${rows || '<tr><td colspan="3" class="text-muted">No KPIs yet</td></tr>'}</tbody>
              </table>
            </div>
          </div>
        </div>
      `;
    }

    async function doTemplateUpdate(id, patch){
      const payload = Object.assign({ id: Number(id) }, patch);
      const res = await fetch('api/appraisal.php?action=template_update', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Update failed','error'); else note('Updated','success');
    }
    async function doTemplateDelete(id){
      const res = await fetch('api/appraisal.php?action=template_delete', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ id:Number(id) }) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Delete failed','error'); else note('Deleted','success');
    }
    async function doKpiCreate(template_id, body){
      const payload = { template_id:Number(template_id), title: String(body.title||''), weight: Number(body.weight||0) };
      const res = await fetch('api/appraisal.php?action=kpi_create', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Create KPI failed','error'); else note('KPI created','success');
    }
    async function doKpiUpdate(id, body){
      const payload = { id:Number(id) };
      if (body.title!=null) payload.title = String(body.title);
      if (body.weight!=null) payload.weight = Number(body.weight);
      const res = await fetch('api/appraisal.php?action=kpi_update', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify(payload) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Update KPI failed','error'); else note('KPI updated','success');
    }
    async function doKpiDelete(id){
      const res = await fetch('api/appraisal.php?action=kpi_delete', { method:'POST', credentials:'same-origin', headers:{'Content-Type':'application/json'}, body: JSON.stringify({ id:Number(id) }) }).then(r=>r.json()).catch(()=>null);
      if (!res?.success) note(res?.message||'Delete KPI failed','error'); else note('KPI deleted','success');
    }

    function openKpiEdit(id, title, weight){
      const modalEl = document.getElementById('kpiEditModal');
      const modal = bootstrap.Modal.getOrCreateInstance(modalEl);
      const form = document.getElementById('kpiEditForm');
      form.querySelector('[name="title"]').value = title || '';
      form.querySelector('[name="weight"]').value = weight || '';
      const btn = document.getElementById('kpiSaveBtn');
      btn.dataset.id = String(id);
      if (!btn._wired){
        btn._wired = true;
        btn.addEventListener('click', async ()=>{
          const fd = new FormData(form);
          await doKpiUpdate(btn.dataset.id, { title: fd.get('title'), weight: fd.get('weight') });
          modal.hide();
          await loadTemplates();
        });
      }
      modal.show();
    }

    function note(msg, type){ window.auth?.showNotification?.(msg, type||'info'); }
    function esc(s){ return String(s??'').replace(/[&<>"']/g, c=>({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c])); }
    function escAttr(s){ return esc(s).replace(/"/g,'&quot;'); }

    // initial
    await loadTemplates();
  }

  window.pages.appraisal_templates = { render };
})();
