// Appraisals page module
(function () {
  window.pages = window.pages || {};

  function currentRole() {
    return window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
  }
  function canCreateAppraisal() {
    const role = currentRole();
    // HR can create for anyone; Employees/Managers can create for themselves (self-setup)
    return ['super_admin', 'admin', 'hr_head', 'hr_officer', 'manager', 'employee'].includes(role);
  }
  function isSelfCreator() {
    const role = currentRole();
    return ['manager','employee'].includes(role);
  }

  async function render(container) {
    const selfCreate = isSelfCreator();
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Appraisals</h5>
          <div>
            ${canCreateAppraisal() ? `<button class="btn btn-sm btn-primary" id="showCreateAppraisal"><i class="fas fa-plus"></i> New Appraisal</button>` : ''}
          </div>
        </div>
        <div class="card-body">
          ${canCreateAppraisal() ? `
          <div id="createAppraisalContainer" class="mb-4" style="display:none;">
            <form id="createAppraisalForm" class="row g-3">
              ${selfCreate ? `
              <div class="col-12">
                <div class="alert alert-info">You are creating your own appraisal. The reviewer will be set to your line manager or your department head.</div>
              </div>
              ` : `
              <div class="col-md-4">
                <label class="form-label">Employee</label>
                <select name="employee_id" class="form-control" id="appEmpSelect" required></select>
              </div>
              <div class="col-md-4">
                <label class="form-label">Reviewer (Line Manager)</label>
                <select name="reviewer_id" class="form-control" id="appReviewerSelect" required></select>
              </div>
              `}
              <div class="col-md-3">
                <label class="form-label">Period Start</label>
                <input type="date" name="period_start" class="form-control" required />
              </div>
              <div class="col-md-3">
                <label class="form-label">Period End</label>
                <input type="date" name="period_end" class="form-control" required />
              </div>
              <div class="col-md-6">
                <label class="form-label">Template (optional)</label>
                <select name="template_id" id="appTplSelect" class="form-select"></select>
              </div>
              <div class="col-md-6">
                <label class="form-label">Comments (optional)</label>
                <input type="text" name="comments" class="form-control" placeholder="Any initial comment" />
              </div>
              <div class="col-12 d-flex justify-content-end">
                <button type="button" class="btn btn-light me-2" id="cancelCreateAppraisal">Cancel</button>
                <button type="submit" class="btn btn-primary">Create</button>
              </div>
            </form>
            <hr/>
          </div>` : ''}

          <div class="table-responsive">
            <table class="table mb-0">
              <thead>
                <tr>
                  <th>Employee</th>
                  <th>Reviewer</th>
                  <th>Period</th>
                  <th>Status</th>
                  <th style="width: 220px;">Actions</th>
                </tr>
              </thead>
              <tbody id="appraisalTableBody">
                <tr><td colspan="5">Loading...</td></tr>
              </tbody>
            </table>
          </div>
        </div>
      </div>

      <!-- Submit Appraisal Modal (employee agree with comments + KPI scores) -->
      <div class="modal fade" id="appSubmitModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Submit Appraisal</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <form id="appSubmitForm" class="row g-3">
                <div class="col-md-6">
                  <label class="form-label">Goals & Achievement</label>
                  <textarea name="goals_achievement" class="form-control" rows="2"></textarea>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Strengths</label>
                  <textarea name="strengths" class="form-control" rows="2"></textarea>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Areas for Improvement</label>
                  <textarea name="areas_for_improvement" class="form-control" rows="2"></textarea>
                </div>
                <div class="col-md-6">
                  <label class="form-label">Development Plan</label>
                  <textarea name="development_plan" class="form-control" rows="2"></textarea>
                </div>
                <div class="col-md-8">
                  <label class="form-label">Comments</label>
                  <textarea name="comments" class="form-control" rows="2"></textarea>
                </div>
                <div class="col-md-4">
                  <label class="form-label">Overall Rating</label>
                  <select name="overall_rating" class="form-select">
                    <option value="">-- Select --</option>
                    <option>Excellent</option>
                    <option>Very Good</option>
                    <option>Good</option>
                    <option>Average</option>
                    <option>Poor</option>
                  </select>
                </div>
                <div class="col-12">
                  <label class="form-label">KPI Scores</label>
                  <div class="table-responsive">
                    <table class="table table-sm">
                      <thead><tr><th>KPI</th><th>Weight</th><th>Your Score</th></tr></thead>
                      <tbody id="kpiRows"><tr><td colspan="3" class="text-muted">No KPIs</td></tr></tbody>
                    </table>
                  </div>
                </div>
              </form>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
              <button type="button" class="btn btn-primary" id="doSubmitAppraisalBtn" data-id="">Submit</button>
            </div>
          </div>
        </div>
      </div>

      <!-- Review Appraisal Modal (manager scores) -->
      <div class="modal fade" id="appReviewModal" tabindex="-1">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
          <div class="modal-content">
            <div class="modal-header">
              <h5 class="modal-title">Review Appraisal</h5>
              <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
              <div class="table-responsive">
                <table class="table table-sm">
                  <thead><tr><th>KPI</th><th>Weight</th><th>Employee Score</th><th>Your Score</th></tr></thead>
                  <tbody id="kpiReviewRows"><tr><td colspan="4" class="text-muted">No KPIs</td></tr></tbody>
                </table>
              </div>
            </div>
            <div class="modal-footer">
              <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
              <button type="button" class="btn btn-primary" id="doReviewAppraisalBtn" data-id="">Mark Reviewed</button>
            </div>
          </div>
        </div>
      </div>
    `;

    if (canCreateAppraisal()) {
      const btn = container.querySelector('#showCreateAppraisal');
      const box = container.querySelector('#createAppraisalContainer');
      const cancelBtn = container.querySelector('#cancelCreateAppraisal');
      if (btn && box) btn.addEventListener('click', () => box.style.display = box.style.display === 'none' ? 'block' : 'none');
      if (cancelBtn && box) cancelBtn.addEventListener('click', () => box.style.display = 'none');

      // Load employees and templates
      const empSelect = container.querySelector('#appEmpSelect');
      const revSelect = container.querySelector('#appReviewerSelect');
      if (empSelect || revSelect) {
        loadEmployeesForSelect(empSelect, revSelect).catch(() => {});
      }
      const tplSelect = container.querySelector('#appTplSelect');
      if (tplSelect) { loadTemplates(tplSelect).catch(()=>{}); }

      const form = container.querySelector('#createAppraisalForm');
      form?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const body = formToJSON(new FormData(form));
        // For self creators, send only period and optional comments; backend resolves employee/reviewer
        if (isSelfCreator()) {
          Object.keys(body).forEach(k=>{ if(!['period_start','period_end','comments','template_id'].includes(k)) delete body[k]; });
        }
        try {
          const res = await fetch('api/appraisal.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify(body)
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Appraisal created', 'success');
            form.reset();
            box.style.display = 'none';
            await loadAppraisals(container);
          } else {
            window.auth?.showNotification?.(data?.message || 'Failed to create appraisal', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Failed to create appraisal', 'error');
        }
      });
    }

    // Delegate actions
    container.addEventListener('click', async (e) => {
      const btn = e.target.closest('[data-action]');
      if (!btn) return;
      const action = btn.getAttribute('data-action');
      const id = btn.getAttribute('data-id');
      if (!id) return;
      try {
        if (action === 'submit-appraisal') {
          openSubmitModal(id, container);
          return;
        }
        if (action === 'review-appraisal') {
          openReviewModal(id, container);
          return;
        }
        let url = '';
        if (action === 'approve-appraisal') url = `api/appraisal.php?action=approve&id=${encodeURIComponent(id)}`;
        if (!url) return;
        const res = await fetch(url, { method: 'POST', credentials: 'same-origin' });
        const data = await res.json();
        if (data?.success) {
          window.auth?.showNotification?.('Action completed', 'success');
          await loadAppraisals(container);
        } else {
          window.auth?.showNotification?.(data?.message || 'Action failed', 'error');
        }
      } catch (_) {
        window.auth?.showNotification?.('Action failed', 'error');
      }
    });

    await loadAppraisals(container);
  }

  async function loadEmployeesForSelect(empSelect, revSelect) {
    try {
      const res = await fetch('api/employees.php', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && Array.isArray(data.data)) {
        const opts = data.data.map((e) => `<option value="${e.id}">${escapeHtml(e.full_name || `${e.first_name || ''} ${e.last_name || ''}`.trim())}</option>`).join('');
        if (empSelect) empSelect.innerHTML = `<option value="">Select Employee</option>` + opts;
        if (revSelect) revSelect.innerHTML = `<option value="">Select Reviewer</option>` + opts;
      }
    } catch (_) {}
  }

  async function loadAppraisals(container) {
    const tbody = container.querySelector('#appraisalTableBody');
    if (!tbody) return;
    tbody.innerHTML = `<tr><td colspan="5">Loading...</td></tr>`;
    try {
      const res = await fetch('api/appraisal.php', { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && Array.isArray(data.data)) {
        if (data.data.length === 0) {
          tbody.innerHTML = `<tr><td colspan="5" class="text-center text-muted">No appraisals found</td></tr>`;
          return;
        }
        tbody.innerHTML = data.data.map(row => appraisalRow(row)).join('');
      } else {
        tbody.innerHTML = `<tr><td colspan="5" class="text-danger">Failed to load appraisals</td></tr>`;
      }
    } catch (e) {
      tbody.innerHTML = `<tr><td colspan="5" class="text-danger">Failed to load appraisals</td></tr>`;
    }
  }

  function appraisalRow(a) {
    const period = `${escapeHtml(a.period_start || '')} - ${escapeHtml(a.period_end || '')}`;
    const canApprove = !!window.auth?.canApprove?.('appraisal');
    const role = currentRole();
    const actions = [];
    if (a.status === 'draft') {
      actions.push(`<button class="btn btn-sm btn-warning me-1" data-action="submit-appraisal" data-id="${a.id}"><i class="fas fa-paper-plane"></i> Submit</button>`);
    }
    if (a.status === 'submitted' && ['manager','hr_head','admin','super_admin'].includes(role)) {
      actions.push(`<button class="btn btn-sm btn-info me-1" data-action="review-appraisal" data-id="${a.id}"><i class="fas fa-check-double"></i> Review</button>`);
    }
    if (a.status === 'reviewed' && canApprove) {
      actions.push(`<button class="btn btn-sm btn-success" data-action="approve-appraisal" data-id="${a.id}"><i class="fas fa-check"></i> Approve</button>`);
    }

    return `
      <tr>
        <td>${escapeHtml(a.employee_name || '-')}</td>
        <td>${escapeHtml(a.reviewer_name || '-')}</td>
        <td>${period}</td>
        <td>${statusBadge(a.status)}</td>
        <td>${actions.join('')}</td>
      </tr>
    `;
  }

  function statusBadge(status) {
    const map = {
      'draft': 'status-badge status-pending',
      'submitted': 'status-badge status-approved',
      'reviewed': 'status-badge status-info',
      'approved': 'status-badge status-active',
      'rejected': 'status-badge status-rejected'
    };
    const cls = map[status] || 'status-badge';
    return `<span class="${cls}">${escapeHtml(status || '')}</span>`;
  }

  function formToJSON(fd) {
    const obj = {};
    fd.forEach((v, k) => { obj[k] = v; });
    return obj;
  }

  function escapeHtml(s) {
    return String(s ?? '').replace(/[&<>"']/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c]));
  }

  window.pages.appraisal = { render };

  function openSubmitModal(id, container){
    const btn = document.getElementById('doSubmitAppraisalBtn');
    if (btn) btn.dataset.id = String(id);
    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById('appSubmitModal'));
    // load KPIs for this appraisal
    const tbody = document.getElementById('kpiRows');
    if (tbody){ tbody.innerHTML = '<tr><td colspan="3">Loading...</td></tr>'; }
    fetch(`api/appraisal.php?action=kpis&id=${encodeURIComponent(id)}`, { credentials:'same-origin' })
      .then(r=>r.json()).then(js=>{
        if (!tbody) return;
        if (!js?.success || !Array.isArray(js.data) || js.data.length===0){ tbody.innerHTML = '<tr><td colspan="3" class="text-muted">No KPIs</td></tr>'; return; }
        tbody.innerHTML = js.data.map(k=>
          `<tr>
            <td>${escapeHtml(k.title||'')}</td>
            <td>${escapeHtml(k.weight??'')}</td>
            <td style="max-width:140px"><input type="number" step="0.01" min="0" max="100" class="form-control form-control-sm" name="kpi_scores[${k.kpi_id}]" value="${k.employee_score??''}"/></td>
          </tr>`
        ).join('');
      }).catch(()=>{ if (tbody) tbody.innerHTML = '<tr><td colspan="3" class="text-danger">Failed to load KPIs</td></tr>'; });
    modal.show();
    if (btn && !btn._wired){
      btn._wired = true;
      btn.addEventListener('click', async ()=>{
        const form = document.getElementById('appSubmitForm');
        const payload = formToJSON(new FormData(form));
        try{
          const res = await fetch(`api/appraisal.php?action=submit&id=${encodeURIComponent(btn.dataset.id)}`, {
            method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload)
          });
          const js = await res.json();
          if (js?.success){
            window.auth?.showNotification?.('Appraisal submitted', 'success');
            modal.hide();
            await loadAppraisals(container);
          } else {
            window.auth?.showNotification?.(js?.message||'Submit failed', 'error');
          }
        }catch(_){ window.auth?.showNotification?.('Submit failed', 'error'); }
      });
    }
  }

  function openReviewModal(id, container){
    const btn = document.getElementById('doReviewAppraisalBtn');
    if (btn) btn.dataset.id = String(id);
    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById('appReviewModal'));
    const tbody = document.getElementById('kpiReviewRows');
    if (tbody){ tbody.innerHTML = '<tr><td colspan="4">Loading...</td></tr>'; }
    fetch(`api/appraisal.php?action=kpis&id=${encodeURIComponent(id)}`, { credentials:'same-origin' })
      .then(r=>r.json()).then(js=>{
        if (!tbody) return;
        if (!js?.success || !Array.isArray(js.data) || js.data.length===0){ tbody.innerHTML = '<tr><td colspan="4" class="text-muted">No KPIs</td></tr>'; return; }
        tbody.innerHTML = js.data.map(k=>
          `<tr>
            <td>${escapeHtml(k.title||'')}</td>
            <td>${escapeHtml(k.weight??'')}</td>
            <td>${escapeHtml(k.employee_score??'')}</td>
            <td style="max-width:140px"><input type="number" step="0.01" min="0" max="100" class="form-control form-control-sm" name="manager_scores[${k.kpi_id}]" value="${k.manager_score??''}"/></td>
          </tr>`
        ).join('');
      }).catch(()=>{ if (tbody) tbody.innerHTML = '<tr><td colspan="4" class="text-danger">Failed to load KPIs</td></tr>'; });
    modal.show();
    if (btn && !btn._wired){
      btn._wired = true;
      btn.addEventListener('click', async ()=>{
        const formData = new FormData();
        // collect manager scores
        document.querySelectorAll('#kpiReviewRows input[name^="manager_scores"]').forEach(inp=>{
          formData.append(inp.name, inp.value);
        });
        const payload = formToJSON(formData);
        try{
          const res = await fetch(`api/appraisal.php?action=review&id=${encodeURIComponent(btn.dataset.id)}`, {
            method:'POST', headers:{'Content-Type':'application/json'}, credentials:'same-origin', body: JSON.stringify(payload)
          });
          const js = await res.json();
          if (js?.success){
            window.auth?.showNotification?.('Appraisal reviewed', 'success');
            modal.hide();
            await loadAppraisals(container);
          } else {
            window.auth?.showNotification?.(js?.message||'Review failed', 'error');
          }
        }catch(_){ window.auth?.showNotification?.('Review failed', 'error'); }
      });
    }
  }

  async function loadTemplates(selectEl){
    try{
      const res = await fetch('api/appraisal.php?action=templates', { credentials:'same-origin' });
      const js = await res.json();
      if (!js?.success){ selectEl.innerHTML = '<option value="">-- None --</option>'; return; }
      const opts = ['<option value="">-- None --</option>'].concat((js.data||[]).map(t=> `<option value="${t.id}">${escapeHtml(t.name)}${t.department_name? ' ('+escapeHtml(t.department_name)+')':''}</option>`));
      selectEl.innerHTML = opts.join('');
    }catch(_){ selectEl.innerHTML = '<option value="">-- None --</option>'; }
  }
})();
