// Announcements page module
(function () {
  window.pages = window.pages || {};

  function canManage() {
    const role = window.auth?.currentUser?.role_slug || window.auth?.currentUser?.role;
    return ['super_admin','admin','hr_head','hr_officer'].includes(role);
  }

  async function render(container) {
    container.innerHTML = `
      <div class="card">
        <div class="card-header d-flex justify-content-between align-items-center">
          <h5 class="card-title mb-0">Announcements</h5>
          ${canManage() ? `<button class="btn btn-sm btn-primary" id="showCreateAnn"><i class="fas fa-plus"></i> New</button>` : ''}
        </div>
        <div class="card-body">
          ${canManage() ? `
          <div id="createAnnBox" class="mb-4" style="display:none;">
            <form id="createAnnForm" class="row g-3">
              <div class="col-12">
                <label class="form-label">Title</label>
                <input type="text" name="title" class="form-control" required />
              </div>
              <div class="col-12">
                <label class="form-label">Content</label>
                <textarea name="content" class="form-control" rows="4" required></textarea>
              </div>
              <div class="col-md-4">
                <label class="form-label">Priority</label>
                <select name="priority" class="form-control">
                  <option value="low">Low</option>
                  <option value="medium" selected>Medium</option>
                  <option value="high">High</option>
                  <option value="urgent">Urgent</option>
                </select>
              </div>
              <div class="col-12 d-flex justify-content-end">
                <button type="button" class="btn btn-light me-2" id="cancelAnnCreate">Cancel</button>
                <button type="submit" class="btn btn-primary">Create</button>
              </div>
            </form>
            <hr/>
          </div>` : ''}

          <div class="d-flex align-items-center mb-3">
            <div class="form-check form-switch">
              <input class="form-check-input" type="checkbox" id="toggleAllAnnouncements">
              <label class="form-check-label" for="toggleAllAnnouncements">Show all (including drafts)</label>
            </div>
          </div>

          <div id="annList">Loading...</div>
        </div>
      </div>
    `;

    // Create form handlers
    if (canManage()) {
      const toggleBtn = container.querySelector('#showCreateAnn');
      const box = container.querySelector('#createAnnBox');
      const cancelBtn = container.querySelector('#cancelAnnCreate');
      if (toggleBtn && box) toggleBtn.addEventListener('click', () => box.style.display = box.style.display === 'none' ? 'block' : 'none');
      if (cancelBtn && box) cancelBtn.addEventListener('click', () => box.style.display = 'none');

      const form = container.querySelector('#createAnnForm');
      form?.addEventListener('submit', async (e) => {
        e.preventDefault();
        const body = formToJSON(new FormData(form));
        try {
          const res = await fetch('api/announcements.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            credentials: 'same-origin',
            body: JSON.stringify(body)
          });
          const data = await res.json();
          if (data?.success) {
            window.auth?.showNotification?.('Announcement created', 'success');
            form.reset();
            box.style.display = 'none';
            await loadAnnouncements(container);
          } else {
            window.auth?.showNotification?.(data?.message || 'Failed to create announcement', 'error');
          }
        } catch (_) {
          window.auth?.showNotification?.('Failed to create announcement', 'error');
        }
      });
    }

    const toggleAll = container.querySelector('#toggleAllAnnouncements');
    if (toggleAll) {
      toggleAll.addEventListener('change', () => loadAnnouncements(container, toggleAll.checked));
    }

    // Delegate actions for publish/archive/delete
    container.addEventListener('click', async (e) => {
      const btn = e.target.closest('[data-action]');
      if (!btn) return;
      const action = btn.getAttribute('data-action');
      const id = btn.getAttribute('data-id');
      if (!id) return;
      try {
        let url = '';
        let opts = { method: 'POST', credentials: 'same-origin' };
        if (action === 'publish') url = `api/announcements.php?action=publish&id=${encodeURIComponent(id)}`;
        if (action === 'archive') url = `api/announcements.php?action=archive&id=${encodeURIComponent(id)}`;
        if (action === 'delete') { url = `api/announcements.php?id=${encodeURIComponent(id)}`; opts.method = 'DELETE'; }
        if (!url) return;
        const res = await fetch(url, opts);
        const data = await res.json();
        if (data?.success) {
          window.auth?.showNotification?.('Action completed', 'success');
          await loadAnnouncements(container, toggleAll?.checked);
        } else {
          window.auth?.showNotification?.(data?.message || 'Action failed', 'error');
        }
      } catch (_) {
        window.auth?.showNotification?.('Action failed', 'error');
      }
    });

    await loadAnnouncements(container);
  }

  async function loadAnnouncements(container, showAll = false) {
    const list = container.querySelector('#annList');
    list.innerHTML = '<div>Loading...</div>';
    try {
      let url = 'api/announcements.php';
      if (!showAll) url += '?action=latest&limit=10';
      const res = await fetch(url, { credentials: 'same-origin' });
      const data = await res.json();
      if (data?.success && Array.isArray(data.data)) {
        if (data.data.length === 0) {
          list.innerHTML = '<div class="text-muted">No announcements found</div>';
          return;
        }
        list.innerHTML = data.data.map(a => announcementCard(a)).join('');
      } else {
        list.innerHTML = '<div class="text-danger">Failed to load announcements</div>';
      }
    } catch (_) {
      list.innerHTML = '<div class="text-danger">Failed to load announcements</div>';
    }
  }

  function announcementCard(a) {
    const controls = canManage() ? `
      <div class="d-flex gap-2">
        ${a.status !== 'published' ? `<button class="btn btn-sm btn-success" data-action="publish" data-id="${a.id}"><i class="fas fa-upload"></i> Publish</button>` : ''}
        ${a.status === 'published' ? `<button class="btn btn-sm btn-warning" data-action="archive" data-id="${a.id}"><i class="fas fa-box-archive"></i> Archive</button>` : ''}
        <button class="btn btn-sm btn-danger" data-action="delete" data-id="${a.id}"><i class="fas fa-trash"></i></button>
      </div>
    ` : '';

    const badge = statusBadge(a.status);

    return `
      <div class="card mb-3">
        <div class="card-header d-flex justify-content-between align-items-center">
          <div class="d-flex align-items-center gap-2">
            <strong>${escapeHtml(a.title || 'Untitled')}</strong>
            ${badge}
          </div>
          ${controls}
        </div>
        <div class="card-body">
          <div class="mb-2">${escapeHtml(a.content || '')}</div>
          <small class="text-muted">By ${escapeHtml(a.author_name || 'Unknown')} • ${formatDateTime(a.published_at || a.created_at)}</small>
        </div>
      </div>
    `;
  }

  function statusBadge(status) {
    const map = {
      'draft': 'status-badge status-pending',
      'published': 'status-badge status-approved',
      'archived': 'status-badge status-inactive'
    };
    const cls = map[status] || 'status-badge';
    return `<span class="${cls}">${escapeHtml(status || '')}</span>`;
  }

  function formToJSON(fd) {
    const obj = {};
    fd.forEach((v, k) => { obj[k] = v; });
    return obj;
  }

  function formatDateTime(s) {
    if (!s) return '';
    try { return new Date(s).toLocaleString(); } catch { return s; }
  }

  function escapeHtml(s) {
    return String(s ?? '').replace(/[&<>"']/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#39;'}[c]));
  }

  window.pages.announcements = { render };
})();
