<?php
/**
 * Simple file upload endpoint for SuperbHR
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ApiResponse::success(null, 'ok');
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    ApiResponse::error('Method not allowed', 405);
}

if (!isset($_FILES['file'])) {
    ApiResponse::error('No file provided', 400);
}

$file = $_FILES['file'];
if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
    $err = $file['error'] ?? 'unknown';
    $map = [
        UPLOAD_ERR_INI_SIZE => 'File exceeds server limit (upload_max_filesize).',
        UPLOAD_ERR_FORM_SIZE => 'File exceeds form limit (MAX_FILE_SIZE).',
        UPLOAD_ERR_PARTIAL => 'The uploaded file was only partially uploaded.',
        UPLOAD_ERR_NO_FILE => 'No file was uploaded.',
        UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder.',
        UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
        UPLOAD_ERR_EXTENSION => 'A PHP extension stopped the file upload.'
    ];
    $msg = is_int($err) && isset($map[$err]) ? $map[$err] : ('Upload error code: ' . $err);
    ApiResponse::error($msg, 400);
}

// Enforce a max file size (bytes)
$MAX_BYTES = 15 * 1024 * 1024; // 15 MB
if (!isset($file['size']) || $file['size'] > $MAX_BYTES) {
    ApiResponse::error('File too large. Maximum allowed is 15 MB.', 413);
}

// Validate MIME type
$allowed = [
    'image/jpeg','image/png','image/gif','image/webp','image/bmp','image/x-ms-bmp',
    'application/pdf'
];
try {
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
} catch (Throwable $e) {
    $mime = 'application/octet-stream';
}

if (!in_array($mime, $allowed, true)) {
    ApiResponse::error('Unsupported file type: ' . $mime, 415);
}

$extMap = [
    'image/jpeg' => '.jpg',
    'image/png'  => '.png',
    'image/gif'  => '.gif',
    'image/webp' => '.webp',
    'image/bmp'  => '.bmp',
    'image/x-ms-bmp' => '.bmp',
    'application/pdf' => '.pdf',
];
$ext = $extMap[$mime] ?? ('.' . strtolower(pathinfo($file['name'], PATHINFO_EXTENSION)));

$subDir = date('Y') . '/' . date('m');
$baseDir = __DIR__ . '/../storage/uploads/' . $subDir;
if (!is_dir($baseDir)) {
    @mkdir($baseDir, 0777, true);
}
$basename = bin2hex(random_bytes(8)) . $ext;
$dest = rtrim($baseDir, '/\\') . '/' . $basename;

if (!move_uploaded_file($file['tmp_name'], $dest)) {
    ApiResponse::error('Failed to move uploaded file', 500);
}

$relativePath = 'storage/uploads/' . $subDir . '/' . $basename;

ApiResponse::success([
    'path' => $relativePath,
    'size' => @filesize($dest),
    'mime_type' => $mime,
    'original_name' => $file['name'] ?? null,
]);
