<?php
/**
 * Training Management API with Approval Workflow
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

switch ($method) {
    case 'GET':
        if ($action === 'participants') {
            getTrainingParticipants($db, $id);
        } elseif ($id) {
            getTraining($db, $id);
        } else {
            getTrainings($db);
        }
        break;
    
    case 'POST':
        if ($action === 'enroll') {
            enrollInTraining($db);
        } elseif ($action === 'approve') {
            approveTrainingEnrollment($db);
        } elseif ($action === 'complete') {
            completeTraining($db);
        } else {
            createTraining($db);
        }
        break;
    
    case 'PUT':
        updateTraining($db, $id);
        break;
    
    case 'DELETE':
        deleteTraining($db, $id);
        break;
    
    default:
        ApiResponse::error('Method not allowed', 405);
}

function getTrainings($db) {
    $user = getCurrentUser();
    $role = $user['role_slug'];
    
    $query = "SELECT 
                t.*,
                CONCAT(instructor.first_name, ' ', instructor.last_name) as instructor_name,
                COUNT(tp.id) as enrolled_count,
                COUNT(CASE WHEN tp.status = 'completed' THEN 1 END) as completed_count
              FROM trainings t
              LEFT JOIN employees instructor ON t.instructor_employee_id = instructor.id
              LEFT JOIN training_participants tp ON t.id = tp.training_id
              WHERE t.company_id = :company_id";
    
    $params = [':company_id' => $user['company_id']];
    
    // Apply filters
    if (isset($_GET['status']) && !empty($_GET['status'])) {
        $query .= " AND t.status = :status";
        $params[':status'] = $_GET['status'];
    }
    
    if (isset($_GET['category']) && !empty($_GET['category'])) {
        $query .= " AND t.category = :category";
        $params[':category'] = $_GET['category'];
    }
    
    $query .= " GROUP BY t.id ORDER BY t.start_date DESC";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    $trainings = $stmt->fetchAll();
    
    // Add enrollment status for current user if employee
    if ($role === 'employee') {
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $employee = $empStmt->fetch();
            $employeeId = $employee['id'];
            
            foreach ($trainings as &$training) {
                $enrollQuery = "SELECT status FROM training_participants 
                               WHERE training_id = :training_id AND employee_id = :employee_id";
                $enrollStmt = $db->prepare($enrollQuery);
                $enrollStmt->bindParam(':training_id', $training['id']);
                $enrollStmt->bindParam(':employee_id', $employeeId);
                $enrollStmt->execute();
                
                $training['enrollment_status'] = $enrollStmt->rowCount() > 0 ? 
                    $enrollStmt->fetch()['status'] : 'not_enrolled';
            }
        }
    }
    
    ApiResponse::success($trainings);
}

function createTraining($db) {
    $user = getCurrentUser();
    
    // Check permissions - only HR and admin can create trainings
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Validate required fields
    $required = ['title', 'duration_hours', 'delivery_method'];
    foreach ($required as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            ApiResponse::error("Field '$field' is required");
        }
    }
    
    try {
        $query = "INSERT INTO trainings (
                    company_id, title, description, category, level, delivery_method,
                    duration_hours, start_date, end_date, location, max_participants,
                    instructor_type, instructor_employee_id, external_instructor,
                    objectives, prerequisites, materials_required, cost_per_participant,
                    status, created_by, created_at
                  ) VALUES (
                    :company_id, :title, :description, :category, :level, :delivery_method,
                    :duration_hours, :start_date, :end_date, :location, :max_participants,
                    :instructor_type, :instructor_employee_id, :external_instructor,
                    :objectives, :prerequisites, :materials_required, :cost_per_participant,
                    :status, :created_by, NOW()
                  )";
        
        $stmt = $db->prepare($query);
        
        $stmt->bindParam(':company_id', $user['company_id']);
        $stmt->bindParam(':title', $input['title']);
        $stmt->bindParam(':description', $input['description'] ?? null);
        $stmt->bindParam(':category', $input['category'] ?? null);
        $stmt->bindParam(':level', $input['level'] ?? 'beginner');
        $stmt->bindParam(':delivery_method', $input['delivery_method']);
        $stmt->bindParam(':duration_hours', $input['duration_hours']);
        $stmt->bindParam(':start_date', $input['start_date'] ?? null);
        $stmt->bindParam(':end_date', $input['end_date'] ?? null);
        $stmt->bindParam(':location', $input['location'] ?? null);
        $stmt->bindParam(':max_participants', $input['max_participants'] ?? null);
        $stmt->bindParam(':instructor_type', $input['instructor_type'] ?? 'internal');
        $stmt->bindParam(':instructor_employee_id', $input['instructor_employee_id'] ?? null);
        $stmt->bindParam(':external_instructor', $input['external_instructor'] ?? null);
        $stmt->bindParam(':objectives', $input['objectives'] ?? null);
        $stmt->bindParam(':prerequisites', $input['prerequisites'] ?? null);
        $stmt->bindParam(':materials_required', $input['materials_required'] ?? null);
        $stmt->bindParam(':cost_per_participant', $input['cost_per_participant'] ?? 0);
        $stmt->bindParam(':status', $input['status'] ?? 'draft');
        $stmt->bindParam(':created_by', $user['id']);
        
        $stmt->execute();
        $trainingId = $db->lastInsertId();
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'training_created', ['training_id' => $trainingId]);
        
        ApiResponse::success(['id' => $trainingId], 'Training created successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to create training: ' . $e->getMessage());
    }
}

function enrollInTraining($db) {
    $user = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['training_id'])) {
        ApiResponse::error('Training ID is required');
    }
    
    // Get employee ID
    $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
    $empStmt = $db->prepare($empQuery);
    $empStmt->bindParam(':user_id', $user['id']);
    $empStmt->execute();
    
    if ($empStmt->rowCount() === 0) {
        ApiResponse::error('Employee record not found');
    }
    
    $employee = $empStmt->fetch();
    $employeeId = $employee['id'];
    
    try {
        // Check if training exists and is active
        $trainingQuery = "SELECT * FROM trainings WHERE id = :training_id AND status = 'active'";
        $trainingStmt = $db->prepare($trainingQuery);
        $trainingStmt->bindParam(':training_id', $input['training_id']);
        $trainingStmt->execute();
        
        if ($trainingStmt->rowCount() === 0) {
            ApiResponse::error('Training not found or not available for enrollment');
        }
        
        $training = $trainingStmt->fetch();
        
        // Check if already enrolled
        $checkQuery = "SELECT id FROM training_participants 
                       WHERE training_id = :training_id AND employee_id = :employee_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':training_id', $input['training_id']);
        $checkStmt->bindParam(':employee_id', $employeeId);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() > 0) {
            ApiResponse::error('Already enrolled in this training');
        }
        
        // Check capacity
        if ($training['max_participants']) {
            $countQuery = "SELECT COUNT(*) as enrolled FROM training_participants 
                          WHERE training_id = :training_id AND status IN ('enrolled', 'in_progress', 'completed')";
            $countStmt = $db->prepare($countQuery);
            $countStmt->bindParam(':training_id', $input['training_id']);
            $countStmt->execute();
            
            $count = $countStmt->fetch();
            if ($count['enrolled'] >= $training['max_participants']) {
                ApiResponse::error('Training is full');
            }
        }
        
        // Enroll participant
        $enrollQuery = "INSERT INTO training_participants (
                          training_id, employee_id, enrollment_date, status, created_at
                        ) VALUES (
                          :training_id, :employee_id, CURDATE(), 'enrolled', NOW()
                        )";
        
        $enrollStmt = $db->prepare($enrollQuery);
        $enrollStmt->bindParam(':training_id', $input['training_id']);
        $enrollStmt->bindParam(':employee_id', $employeeId);
        $enrollStmt->execute();
        
        $participantId = $db->lastInsertId();
        
        // Create notification for manager/HR to approve enrollment
        createTrainingNotification($db, $participantId, 'training_enrollment_request');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'training_enrollment', [
            'training_id' => $input['training_id'],
            'participant_id' => $participantId
        ]);
        
        ApiResponse::success(['participant_id' => $participantId], 'Successfully enrolled in training');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to enroll in training: ' . $e->getMessage());
    }
}

function approveTrainingEnrollment($db) {
    $user = getCurrentUser();
    
    // Check if user can approve training enrollments
    if (!canApprove('training')) {
        ApiResponse::forbidden('Insufficient permissions to approve training enrollments');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['participant_id'])) {
        ApiResponse::error('Participant ID is required');
    }
    
    try {
        // Check if participant exists and is pending approval
        $checkQuery = "SELECT tp.*, t.title as training_title, e.company_id
                       FROM training_participants tp
                       JOIN trainings t ON tp.training_id = t.id
                       JOIN employees e ON tp.employee_id = e.id
                       WHERE tp.id = :participant_id AND tp.status = 'enrolled' AND e.company_id = :company_id";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':participant_id', $input['participant_id']);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Training enrollment not found or already processed');
        }
        
        // Update enrollment status
        $updateQuery = "UPDATE training_participants SET 
                          status = 'in_progress',
                          updated_at = NOW()
                        WHERE id = :participant_id";
        
        $updateStmt = $db->prepare($updateQuery);
        $updateStmt->bindParam(':participant_id', $input['participant_id']);
        $updateStmt->execute();
        
        // Create notification for employee
        createTrainingNotification($db, $input['participant_id'], 'training_enrollment_approved');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'training_enrollment_approved', [
            'participant_id' => $input['participant_id']
        ]);
        
        ApiResponse::success(null, 'Training enrollment approved');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to approve training enrollment: ' . $e->getMessage());
    }
}

function completeTraining($db) {
    $user = getCurrentUser();
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Check permissions - only HR and instructors can mark training as complete
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    if (!isset($input['participant_id'])) {
        ApiResponse::error('Participant ID is required');
    }
    
    try {
        // Check if participant exists and is in progress
        $checkQuery = "SELECT tp.*, t.title as training_title, e.company_id
                       FROM training_participants tp
                       JOIN trainings t ON tp.training_id = t.id
                       JOIN employees e ON tp.employee_id = e.id
                       WHERE tp.id = :participant_id AND tp.status = 'in_progress' AND e.company_id = :company_id";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':participant_id', $input['participant_id']);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Training participant not found or not in progress');
        }
        
        // Update completion status
        $updateQuery = "UPDATE training_participants SET 
                          status = 'completed',
                          completion_date = CURDATE(),
                          score = :score,
                          feedback = :feedback,
                          certificate_issued = :certificate_issued,
                          updated_at = NOW()
                        WHERE id = :participant_id";
        
        $updateStmt = $db->prepare($updateQuery);
        $updateStmt->bindParam(':participant_id', $input['participant_id']);
        $updateStmt->bindParam(':score', $input['score'] ?? null);
        $updateStmt->bindParam(':feedback', $input['feedback'] ?? null);
        $updateStmt->bindParam(':certificate_issued', isset($input['certificate_issued']) ? 1 : 0);
        $updateStmt->execute();
        
        // Create notification for employee
        createTrainingNotification($db, $input['participant_id'], 'training_completed');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'training_completed', [
            'participant_id' => $input['participant_id']
        ]);
        
        ApiResponse::success(null, 'Training marked as completed');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to complete training: ' . $e->getMessage());
    }
}

function createTrainingNotification($db, $participantId, $type) {
    // Get participant details
    $query = "SELECT tp.*, t.title as training_title,
                     e.user_id as employee_user_id, e.manager_id,
                     CONCAT(e.first_name, ' ', e.last_name) as employee_name
              FROM training_participants tp
              JOIN trainings t ON tp.training_id = t.id
              JOIN employees e ON tp.employee_id = e.id
              WHERE tp.id = :participant_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':participant_id', $participantId);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $participant = $stmt->fetch();
        
        $notifications = [];
        
        switch ($type) {
            case 'training_enrollment_request':
                // Notify manager and HR
                if ($participant['manager_id']) {
                    $managerQuery = "SELECT user_id FROM employees WHERE id = :manager_id";
                    $managerStmt = $db->prepare($managerQuery);
                    $managerStmt->bindParam(':manager_id', $participant['manager_id']);
                    $managerStmt->execute();
                    
                    if ($managerStmt->rowCount() > 0) {
                        $manager = $managerStmt->fetch();
                        $notifications[] = [
                            'user_id' => $manager['user_id'],
                            'title' => 'Training Enrollment Request',
                            'content' => "{$participant['employee_name']} has requested to enroll in '{$participant['training_title']}'."
                        ];
                    }
                }
                break;
                
            case 'training_enrollment_approved':
            case 'training_completed':
                // Notify employee
                if ($participant['employee_user_id']) {
                    $title = $type === 'training_enrollment_approved' ? 
                        'Training Enrollment Approved' : 'Training Completed';
                    $content = $type === 'training_enrollment_approved' ?
                        "Your enrollment in '{$participant['training_title']}' has been approved." :
                        "You have successfully completed '{$participant['training_title']}' training.";
                    
                    $notifications[] = [
                        'user_id' => $participant['employee_user_id'],
                        'title' => $title,
                        'content' => $content
                    ];
                }
                break;
        }
        
        // Insert notifications
        foreach ($notifications as $notification) {
            $notifQuery = "INSERT INTO notifications (user_id, type, title, content, created_at) 
                           VALUES (:user_id, :type, :title, :content, NOW())";
            $notifStmt = $db->prepare($notifQuery);
            $notifStmt->bindParam(':user_id', $notification['user_id']);
            $notifStmt->bindParam(':type', $type);
            $notifStmt->bindParam(':title', $notification['title']);
            $notifStmt->bindParam(':content', $notification['content']);
            $notifStmt->execute();
        }
    }
}
?>
