<?php
/**
 * Generic Reporting API (whitelisted entities/columns)
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$database = new Database();
$db = $database->getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$entity = $_GET['entity'] ?? '';

$user = getCurrentUser();
$role = $user['role_slug'] ?? 'employee';
$companyId = (int)($user['company_id'] ?? 0);

// Whitelisted entities and columns
$ENTITIES = [
  'employees' => [
    'from' => 'employees e',
    'joins' => ' LEFT JOIN departments d ON e.department_id = d.id'
              . ' LEFT JOIN positions p ON e.position_id = p.id'
              . ' LEFT JOIN branches b ON e.branch_id = b.id'
              . ' LEFT JOIN employees m ON e.manager_id = m.id',
    'company_field' => 'e.company_id',
    'date_field' => 'e.hire_date',
    'columns' => [
      'id' => 'e.id',
      'employee_number' => 'e.employee_number',
      'first_name' => 'e.first_name',
      'last_name' => 'e.last_name',
      'full_name' => "CONCAT(e.first_name, ' ', e.last_name)",
      'email' => 'e.email',
      'phone' => 'e.phone',
      'hire_date' => 'e.hire_date',
      'employment_status' => 'e.employment_status',
      'department_name' => 'd.name',
      'position_title' => 'p.title',
      'branch_name' => 'b.name',
      'manager_name' => "CONCAT(m.first_name, ' ', m.last_name)",
    ],
    'searchable' => ['e.employee_number','e.first_name','e.last_name','e.email','d.name','p.title','b.name'],
    'default_columns' => ['employee_number','full_name','department_name','position_title','hire_date','employment_status'],
  ],
  'leave_requests' => [
    'from' => 'leave_requests lr',
    'joins' => ' JOIN employees e ON lr.employee_id = e.id'
              . ' LEFT JOIN leave_types lt ON lr.leave_type_id = lt.id',
    'company_field' => 'e.company_id',
    'date_field' => 'lr.start_date',
    'columns' => [
      'id' => 'lr.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'employee_number' => 'e.employee_number',
      'leave_type' => 'lt.name',
      'start_date' => 'lr.start_date',
      'end_date' => 'lr.end_date',
      'days_requested' => 'lr.days_requested',
      'status' => 'lr.status',
      'reason' => 'lr.reason',
    ],
    'searchable' => ['e.first_name','e.last_name','e.employee_number','lt.name','lr.status'],
    'default_columns' => ['employee_number','employee_name','leave_type','start_date','end_date','days_requested','status'],
  ],
  'appraisals' => [
    'from' => 'appraisals a',
    'joins' => ' JOIN employees e ON a.employee_id = e.id'
              . ' LEFT JOIN employees r ON a.reviewer_id = r.id',
    'company_field' => 'e.company_id',
    'date_field' => 'a.created_at',
    'columns' => [
      'id' => 'a.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'reviewer_name' => "CONCAT(r.first_name,' ',r.last_name)",
      'period_start' => 'a.period_start',
      'period_end' => 'a.period_end',
      'status' => 'a.status',
      'overall_rating' => 'a.overall_rating',
      'submitted_at' => 'a.submitted_at',
      'reviewed_at' => 'a.reviewed_at',
      'approved_at' => 'a.approved_at',
    ],
    'searchable' => ['e.first_name','e.last_name','r.first_name','r.last_name','a.status'],
    'default_columns' => ['employee_name','reviewer_name','period_start','period_end','status','overall_rating'],
  ],
  'expenses' => [
    'from' => 'expenses x',
    'joins' => ' LEFT JOIN employees e ON x.employee_id = e.id',
    'company_field' => 'x.company_id',
    'date_field' => 'x.created_at',
    'columns' => [
      'id' => 'x.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'type' => 'x.type',
      'category' => 'x.category',
      'amount' => 'x.amount',
      'currency' => 'x.currency',
      'status' => 'x.status',
      'created_at' => 'x.created_at',
    ],
    'searchable' => ['x.type','x.category','x.status','e.first_name','e.last_name'],
    'default_columns' => ['employee_name','type','category','amount','currency','status','created_at'],
  ],
  'trainings' => [
    'from' => 'trainings t',
    'joins' => '',
    'company_field' => 't.company_id',
    'date_field' => 't.start_date',
    'columns' => [
      'id' => 't.id',
      'title' => 't.title',
      'category' => 't.category',
      'level' => 't.level',
      'delivery_method' => 't.delivery_method',
      'start_date' => 't.start_date',
      'end_date' => 't.end_date',
      'location' => 't.location',
      'status' => 't.status',
    ],
    'searchable' => ['t.title','t.category','t.level','t.status'],
    'default_columns' => ['title','category','level','start_date','end_date','status'],
  ],
  'payrolls' => [
    'from' => 'payrolls pr',
    'joins' => ' JOIN employees e ON pr.employee_id = e.id',
    'company_field' => 'pr.company_id',
    'date_field' => 'pr.pay_period_start',
    'columns' => [
      'id' => 'pr.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'pay_period_start' => 'pr.pay_period_start',
      'pay_period_end' => 'pr.pay_period_end',
      'basic_salary' => 'pr.basic_salary',
      'gross_salary' => 'pr.gross_salary',
      'net_salary' => 'pr.net_salary',
      'status' => 'pr.status',
    ],
    'searchable' => ['e.first_name','e.last_name','pr.status'],
    'default_columns' => ['employee_name','pay_period_start','pay_period_end','gross_salary','net_salary','status'],
  ],
  // Additional coverage
  'attendance' => [
    'from' => 'attendance a',
    'joins' => ' JOIN employees e ON a.employee_id = e.id',
    'company_field' => 'e.company_id',
    'date_field' => 'a.date',
    'columns' => [
      'id' => 'a.id',
      'date' => 'a.date',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'employee_number' => 'e.employee_number',
      'status' => 'a.status',
      'location' => 'a.location',
      'notes' => 'a.notes',
    ],
    'searchable' => ['e.first_name','e.last_name','e.employee_number','a.status'],
    'default_columns' => ['date','employee_number','employee_name','status'],
  ],
  'training_participants' => [
    'from' => 'training_participants tp',
    'joins' => ' JOIN trainings t ON tp.training_id = t.id JOIN employees e ON tp.employee_id = e.id',
    'company_field' => 't.company_id',
    'date_field' => 't.start_date',
    'columns' => [
      'id' => 'tp.id',
      'training_title' => 't.title',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'status' => 'tp.status',
      'score' => 'tp.score',
      'completed_at' => 'tp.completed_at',
    ],
    'searchable' => ['t.title','e.first_name','e.last_name','tp.status'],
    'default_columns' => ['training_title','employee_name','status','score'],
  ],
  'grades' => [
    'from' => 'grades g',
    'joins' => '',
    'company_field' => 'g.company_id',
    'date_field' => 'g.created_at',
    'columns' => [
      'id' => 'g.id',
      'name' => 'g.name',
      'code' => 'g.code',
      'base_salary' => 'g.base_salary',
      'created_at' => 'g.created_at',
    ],
    'searchable' => ['g.name','g.code'],
    'default_columns' => ['name','code','base_salary'],
  ],
  'leave_types' => [
    'from' => 'leave_types lt',
    'joins' => '',
    'company_field' => 'lt.company_id',
    'date_field' => 'lt.created_at',
    'columns' => [
      'id' => 'lt.id',
      'name' => 'lt.name',
      'code' => 'lt.code',
      'days_per_year' => 'lt.days_per_year',
    ],
    'searchable' => ['lt.name','lt.code'],
    'default_columns' => ['name','code','days_per_year'],
  ],
  'leave_allocations' => [
    'from' => 'leave_allocations la',
    'joins' => ' LEFT JOIN leave_types lt ON la.leave_type_id = lt.id',
    'company_field' => 'la.company_id',
    'date_field' => 'la.created_at',
    'columns' => [
      'id' => 'la.id',
      'grade' => 'la.grade',
      'leave_type' => 'lt.name',
      'days_per_year' => 'la.days_per_year',
    ],
    'searchable' => ['la.grade','lt.name'],
    'default_columns' => ['grade','leave_type','days_per_year'],
  ],
  'job_postings' => [
    'from' => 'job_postings jp',
    'joins' => '',
    'company_field' => 'jp.company_id',
    'date_field' => 'jp.created_at',
    'columns' => [
      'id' => 'jp.id',
      'title' => 'jp.title',
      'status' => 'jp.status',
      'employment_type' => 'jp.employment_type',
      'deadline' => 'jp.deadline',
    ],
    'searchable' => ['jp.title','jp.status','jp.employment_type'],
    'default_columns' => ['title','employment_type','status','deadline'],
  ],
  'candidates' => [
    'from' => 'candidates c',
    'joins' => '',
    'company_field' => 'c.company_id',
    'date_field' => 'c.created_at',
    'columns' => [
      'id' => 'c.id',
      'full_name' => "CONCAT(c.first_name,' ',c.last_name)",
      'email' => 'c.email',
      'phone' => 'c.phone',
      'status' => 'c.status',
      'source' => 'c.source',
    ],
    'searchable' => ['c.first_name','c.last_name','c.email','c.status','c.source'],
    'default_columns' => ['full_name','email','phone','status','source'],
  ],
  'job_applications' => [
    'from' => 'job_applications ja',
    'joins' => ' JOIN job_postings jp ON ja.job_posting_id = jp.id JOIN candidates c ON ja.candidate_id = c.id',
    'company_field' => 'jp.company_id',
    'date_field' => 'ja.application_date',
    'columns' => [
      'id' => 'ja.id',
      'job_title' => 'jp.title',
      'candidate_name' => "CONCAT(c.first_name,' ',c.last_name)",
      'status' => 'ja.status',
      'application_date' => 'ja.application_date',
    ],
    'searchable' => ['jp.title','c.first_name','c.last_name','ja.status'],
    'default_columns' => ['job_title','candidate_name','status','application_date'],
  ],
  'benefits' => [
    'from' => 'benefits b',
    'joins' => '',
    'company_field' => 'b.company_id',
    'date_field' => 'b.created_at',
    'columns' => [
      'id' => 'b.id',
      'name' => 'b.name',
      'type' => 'b.type',
      'provider' => 'b.provider',
      'status' => 'b.status',
    ],
    'searchable' => ['b.name','b.type','b.provider','b.status'],
    'default_columns' => ['name','type','provider','status'],
  ],
  'allowance_types' => [
    'from' => 'allowance_types atp',
    'joins' => '',
    'company_field' => 'atp.company_id',
    'date_field' => 'atp.created_at',
    'columns' => [
      'id' => 'atp.id',
      'name' => 'atp.name',
      'code' => 'atp.code',
      'is_taxable' => 'atp.is_taxable',
    ],
    'searchable' => ['atp.name','atp.code'],
    'default_columns' => ['name','code','is_taxable'],
  ],
  'deduction_types' => [
    'from' => 'deduction_types dtp',
    'joins' => '',
    'company_field' => 'dtp.company_id',
    'date_field' => 'dtp.created_at',
    'columns' => [
      'id' => 'dtp.id',
      'name' => 'dtp.name',
      'code' => 'dtp.code',
      'is_percentage' => 'dtp.is_percentage',
    ],
    'searchable' => ['dtp.name','dtp.code'],
    'default_columns' => ['name','code','is_percentage'],
  ],
  'employee_certificates' => [
    'from' => 'employee_certificates ec',
    'joins' => ' JOIN employees e ON ec.employee_id = e.id JOIN certificates c ON ec.certificate_id = c.id',
    'company_field' => 'c.company_id',
    'date_field' => 'ec.issue_date',
    'columns' => [
      'id' => 'ec.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'certificate' => 'c.name',
      'issue_date' => 'ec.issue_date',
      'expiry_date' => 'ec.expiry_date',
      'status' => 'ec.status',
    ],
    'searchable' => ['c.name','e.first_name','e.last_name','ec.status'],
    'default_columns' => ['employee_name','certificate','issue_date','expiry_date','status'],
  ],
  'employee_skills' => [
    'from' => 'employee_skills es',
    'joins' => ' JOIN employees e ON es.employee_id = e.id JOIN skills s ON es.skill_id = s.id',
    'company_field' => 's.company_id',
    'date_field' => 'es.created_at',
    'columns' => [
      'id' => 'es.id',
      'employee_name' => "CONCAT(e.first_name,' ',e.last_name)",
      'skill' => 's.name',
      'proficiency' => 'es.proficiency',
      'years_of_experience' => 'es.years_of_experience',
    ],
    'searchable' => ['s.name','e.first_name','e.last_name','es.proficiency'],
    'default_columns' => ['employee_name','skill','proficiency','years_of_experience'],
  ],
];

// Saved reports & scheduling endpoints
if ($method === 'GET' && $action === 'saved') { ensureReportTables($db); listSavedReports($db, $companyId, $user); }

// Row-level security (self/team) for employee/manager roles
if (in_array($role, ['employee','manager'])) {
  // resolve current employee id if exists
  $empId = null;
  try {
    $q = $db->prepare('SELECT id FROM employees WHERE user_id = :uid');
    $q->bindValue(':uid', $user['id'], PDO::PARAM_INT); $q->execute();
    $row = $q->fetch(); if ($row) $empId = (int)$row['id'];
  } catch (Throwable $e) { $empId = null; }
  if ($empId) {
    if ($entity === 'employees') {
      if ($role === 'employee') { $where[] = 'e.id = :r_emp'; $params[':r_emp'] = $empId; }
      else { $where[] = '(e.manager_id = :r_mgr OR e.id = :r_emp)'; $params[':r_mgr'] = $empId; $params[':r_emp'] = $empId; }
    } elseif (in_array($entity, ['leave_requests','expenses','payrolls','attendance','training_participants'])) {
      // All use employees alias e in joins above
      if ($role === 'employee') { $where[] = 'e.id = :r_emp'; $params[':r_emp'] = $empId; }
      else { $where[] = '(e.manager_id = :r_mgr OR e.id = :r_emp)'; $params[':r_mgr'] = $empId; $params[':r_emp'] = $empId; }
    } elseif ($entity === 'appraisals') {
      if ($role === 'employee') { $where[] = 'e.id = :r_emp'; $params[':r_emp'] = $empId; }
      else { $where[] = '(e.manager_id = :r_mgr OR a.reviewer_id = :r_mgr OR e.id = :r_emp)'; $params[':r_mgr'] = $empId; $params[':r_emp'] = $empId; }
    } else {
      // other entities remain company-scoped
    }
  }
}
if ($method === 'GET' && $action === 'saved_show') { ensureReportTables($db); showSavedReport($db, $companyId, $user); }
if ($method === 'GET' && $action === 'schedules') { ensureReportTables($db); listReportSchedules($db, $companyId, $user); }
if ($method === 'POST' && $action === 'save') { ensureReportTables($db); saveReport($db, $companyId, $user); }
if ($method === 'POST' && $action === 'delete') { ensureReportTables($db); deleteReport($db, $companyId, $user); }
if ($method === 'POST' && $action === 'email') { ensureReportTables($db); emailReport($db, $ENTITIES, $companyId, $user); }
if ($method === 'POST' && $action === 'schedule') { ensureReportTables($db); saveSchedule($db, $companyId, $user); }
if ($method === 'POST' && $action === 'run_due') { ensureReportTables($db); runDueSchedules($db, $ENTITIES, $companyId, $user); }

// Fall through to normal GET runner below
if ($method !== 'GET') { ApiResponse::error('Method not allowed', 405); }

if ($action === 'schema') {
  if ($entity && isset($ENTITIES[$entity])) {
    $cfg = $ENTITIES[$entity];
    ApiResponse::success([
      'entity' => $entity,
      'columns' => array_keys($cfg['columns']),
      'default_columns' => $cfg['default_columns'],
      'date_field' => $cfg['date_field'],
    ]);
  } else {
    $out = [];
    foreach ($ENTITIES as $k => $v) {
      $out[$k] = [
        'columns' => array_keys($v['columns']),
        'default_columns' => $v['default_columns'],
        'date_field' => $v['date_field'],
      ];
    }
    ApiResponse::success($out);
  }
}

if (!$entity || !isset($ENTITIES[$entity])) {
  ApiResponse::error('Unknown or missing entity');
}

$cfg = $ENTITIES[$entity];

$columnsParam = isset($_GET['columns']) ? explode(',', $_GET['columns']) : $cfg['default_columns'];
$columns = [];
foreach ($columnsParam as $c) {
  $c = trim($c);
  if ($c !== '' && isset($cfg['columns'][$c])) $columns[$c] = $cfg['columns'][$c];
}
if (empty($columns)) {
  $columns = array_intersect_key($cfg['columns'], array_flip($cfg['default_columns']));
}

$select = [];
foreach ($columns as $alias => $expr) { $select[] = "$expr AS `$alias`"; }
$selectSql = implode(', ', $select);

$sql = 'SELECT ' . $selectSql . ' FROM ' . $cfg['from'] . ' ' . ($cfg['joins'] ?? '');
$where = [];
$params = [];

// company scope
if (!empty($cfg['company_field'])) {
  $where[] = $cfg['company_field'] . ' = :cid';
  $params[':cid'] = $companyId;
}

// date range
$dateField = $cfg['date_field'];
if (!empty($_GET['date_from'])) { $where[] = "$dateField >= :dfrom"; $params[':dfrom'] = $_GET['date_from']; }
if (!empty($_GET['date_to'])) { $where[] = "$dateField <= :dto"; $params[':dto'] = $_GET['date_to']; }

// search
$search = trim((string)($_GET['search'] ?? ''));
if ($search !== '' && !empty($cfg['searchable'])) {
  $ors = [];
  foreach ($cfg['searchable'] as $scol) { $ors[] = "$scol LIKE :search"; }
  if ($ors) { $where[] = '(' . implode(' OR ', $ors) . ')'; $params[':search'] = '%' . $search . '%'; }
}

// filters (JSON array of {field, op, value})
$filtersJson = $_GET['filters'] ?? '';
if ($filtersJson) {
  $filters = json_decode($filtersJson, true);
  if (is_array($filters)) {
    foreach ($filters as $i => $f) {
      $field = $f['field'] ?? '';
      $op = strtolower($f['op'] ?? 'eq');
      $value = $f['value'] ?? null;
      if (!isset($cfg['columns'][$field])) continue; // allow only whitelisted columns
      $param = ':f' . $i;
      switch ($op) {
        case 'eq': $where[] = $cfg['columns'][$field] . " = $param"; $params[$param] = $value; break;
        case 'ne': $where[] = $cfg['columns'][$field] . " <> $param"; $params[$param] = $value; break;
        case 'like': $where[] = $cfg['columns'][$field] . " LIKE $param"; $params[$param] = "%$value%"; break;
        case 'gt': $where[] = $cfg['columns'][$field] . " > $param"; $params[$param] = $value; break;
        case 'lt': $where[] = $cfg['columns'][$field] . " < $param"; $params[$param] = $value; break;
        case 'gte': $where[] = $cfg['columns'][$field] . " >= $param"; $params[$param] = $value; break;
        case 'lte': $where[] = $cfg['columns'][$field] . " <= $param"; $params[$param] = $value; break;
        case 'between':
          $v1 = is_array($value) ? ($value[0] ?? null) : null;
          $v2 = is_array($value) ? ($value[1] ?? null) : null;
          if ($v1 !== null && $v2 !== null) { $where[] = $cfg['columns'][$field] . " BETWEEN $param" . 'a' . " AND $param" . 'b'; $params[$param.'a'] = $v1; $params[$param.'b'] = $v2; }
          break;
        case 'in':
          if (is_array($value) && $value) {
            $phs = [];
            foreach ($value as $k => $v) { $p = $param . '_' . $k; $phs[] = $p; $params[$p] = $v; }
            $where[] = $cfg['columns'][$field] . ' IN (' . implode(',', $phs) . ')';
          }
          break;
      }
    }
  }
}

if ($where) { $sql .= ' WHERE ' . implode(' AND ', $where); }

// sorting
$sort = $_GET['sort'] ?? '';
if ($sort) {
  // format: field:asc or field:desc
  $parts = explode(':', $sort);
  $sf = trim($parts[0]); $dir = strtolower(trim($parts[1] ?? 'asc')); $dir = $dir === 'desc' ? 'DESC' : 'ASC';
  if (isset($cfg['columns'][$sf])) { $sql .= ' ORDER BY ' . $cfg['columns'][$sf] . ' ' . $dir; }
}

// pagination
$page = max(1, (int)($_GET['page'] ?? 1));
$pageSize = min(500, max(10, (int)($_GET['page_size'] ?? 50)));
$offset = ($page - 1) * $pageSize;
$sql .= ' LIMIT ' . $pageSize . ' OFFSET ' . $offset;

$stmt = $db->prepare($sql);
foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
$stmt->execute();
$rows = $stmt->fetchAll();

ApiResponse::success([
  'rows' => $rows,
  'page' => $page,
  'page_size' => $pageSize,
  'columns' => array_keys($columns),
]);

// ================= Helpers for Saved Reports & Email =================

function ensureReportTables(PDO $db){
  $db->exec("CREATE TABLE IF NOT EXISTS saved_reports (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    owner_user_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    entity VARCHAR(64) NOT NULL,
    columns_json TEXT,
    filters_json TEXT,
    date_from DATE NULL,
    date_to DATE NULL,
    sort VARCHAR(128) NULL,
    shared_with JSON NULL,
    created_at DATETIME NOT NULL,
    updated_at DATETIME NULL,
    INDEX (company_id), INDEX (owner_user_id), INDEX (entity)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS report_schedules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    saved_report_id INT NOT NULL,
    frequency ENUM('once','daily','weekly','monthly','quarterly') NOT NULL DEFAULT 'monthly',
    day_of_week TINYINT NULL,
    day_of_month TINYINT NULL,
    time_of_day VARCHAR(5) NULL,
    recipients_json TEXT,
    format ENUM('csv','xlsx','pdf') NOT NULL DEFAULT 'csv',
    active TINYINT(1) NOT NULL DEFAULT 1,
    last_run_at DATETIME NULL,
    next_run_at DATETIME NULL,
    created_at DATETIME NOT NULL,
    updated_at DATETIME NULL,
    INDEX (company_id), INDEX (saved_report_id), INDEX (active), INDEX (next_run_at)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

function listSavedReports(PDO $db, int $companyId, array $user){
  $st = $db->prepare("SELECT id, name, entity, created_at, owner_user_id FROM saved_reports WHERE company_id = :cid ORDER BY created_at DESC");
  $st->bindValue(':cid', $companyId, PDO::PARAM_INT); $st->execute();
  ApiResponse::success($st->fetchAll());
}

function showSavedReport(PDO $db, int $companyId, array $user){
  $id = (int)($_GET['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $st = $db->prepare('SELECT * FROM saved_reports WHERE id = :id AND company_id = :cid');
  $st->bindValue(':id',$id,PDO::PARAM_INT); $st->bindValue(':cid',$companyId,PDO::PARAM_INT); $st->execute();
  if ($st->rowCount()===0) ApiResponse::notFound('Report not found');
  $row = $st->fetch();
  $row['columns'] = json_decode($row['columns_json'], true) ?: [];
  $row['filters'] = json_decode($row['filters_json'], true) ?: [];
  unset($row['columns_json'], $row['filters_json']);
  ApiResponse::success($row);
}

function listReportSchedules(PDO $db, int $companyId, array $user){
  $st = $db->prepare("SELECT rs.*, sr.name FROM report_schedules rs JOIN saved_reports sr ON rs.saved_report_id = sr.id WHERE rs.company_id = :cid ORDER BY rs.next_run_at ASC, rs.created_at DESC");
  $st->bindValue(':cid', $companyId, PDO::PARAM_INT); $st->execute();
  ApiResponse::success($st->fetchAll());
}

function saveReport(PDO $db, int $companyId, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $name = trim((string)($in['name'] ?? ''));
  $entity = trim((string)($in['entity'] ?? ''));
  if ($name === '' || $entity === '') ApiResponse::error('Name and entity are required');
  $columns = isset($in['columns']) ? (array)$in['columns'] : [];
  $filters = isset($in['filters']) ? $in['filters'] : [];
  $date_from = $in['date_from'] ?? null; $date_to = $in['date_to'] ?? null; $sort = $in['sort'] ?? null;
  $shared_with = isset($in['shared_with']) ? json_encode($in['shared_with']) : null;
  $st = $db->prepare("INSERT INTO saved_reports (company_id, owner_user_id, name, entity, columns_json, filters_json, date_from, date_to, sort, shared_with, created_at)
                      VALUES (:cid, :uid, :name, :entity, :cols, :fil, :df, :dt, :sort, :shared, NOW())");
  $st->bindValue(':cid',$companyId,PDO::PARAM_INT);
  $st->bindValue(':uid',$user['id'],PDO::PARAM_INT);
  $st->bindValue(':name',$name);
  $st->bindValue(':entity',$entity);
  $st->bindValue(':cols', json_encode(array_values($columns)));
  $st->bindValue(':fil', is_string($filters)? $filters : json_encode($filters));
  $st->bindValue(':df', $date_from);
  $st->bindValue(':dt', $date_to);
  $st->bindValue(':sort', $sort);
  $st->bindValue(':shared', $shared_with);
  $st->execute();
  ApiResponse::success(['id' => $db->lastInsertId()], 'Report saved');
}

function deleteReport(PDO $db, int $companyId, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  // allow owner or admin/hr_head
  $st = $db->prepare('SELECT owner_user_id FROM saved_reports WHERE id = :id AND company_id = :cid');
  $st->bindValue(':id',$id,PDO::PARAM_INT); $st->bindValue(':cid',$companyId,PDO::PARAM_INT); $st->execute();
  if ($st->rowCount()===0) ApiResponse::notFound('Not found');
  $row = $st->fetch();
  if ($row['owner_user_id'] != $user['id'] && !in_array($user['role_slug'], ['super_admin','admin','hr_head'])) ApiResponse::forbidden('Not allowed');
  $db->prepare('DELETE FROM saved_reports WHERE id = :id')->execute([':id'=>$id]);
  ApiResponse::success(null,'Deleted');
}

function emailReport(PDO $db, array $ENTITIES, int $companyId, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $savedId = isset($in['saved_report_id']) ? (int)$in['saved_report_id'] : 0;
  $entity = $in['entity'] ?? '';
  $columns = $in['columns'] ?? [];
  $filters = $in['filters'] ?? [];
  $date_from = $in['date_from'] ?? null; $date_to = $in['date_to'] ?? null; $sort = $in['sort'] ?? null;
  $recipients = $in['recipients'] ?? '';
  $subject = trim((string)($in['subject'] ?? 'SmartQuantumHR Report'));
  $format = strtolower((string)($in['format'] ?? 'csv'));

  if ($savedId>0){
    $st = $db->prepare('SELECT * FROM saved_reports WHERE id = :id AND company_id = :cid');
    $st->execute([':id'=>$savedId, ':cid'=>$companyId]);
    if ($st->rowCount()===0) ApiResponse::notFound('Saved report not found');
    $sr = $st->fetch();
    $entity = $sr['entity'];
    $columns = json_decode($sr['columns_json'], true) ?: [];
    $filters = json_decode($sr['filters_json'], true) ?: [];
    $date_from = $sr['date_from']; $date_to = $sr['date_to']; $sort = $sr['sort'];
    if ($subject==='') $subject = 'Report: '.$sr['name'];
  }
  if (!$entity || !isset($ENTITIES[$entity])) ApiResponse::error('Invalid entity');
  $rowsAndCols = runReportQuery($db, $ENTITIES, $companyId, $entity, $columns, $filters, $date_from, $date_to, $sort, 1, 100000);
  $rows = $rowsAndCols['rows']; $cols = $rowsAndCols['columns'];

  // Build CSV
  $csv = implode(',', array_map('csvCell',$cols))."\n";
  foreach ($rows as $r){
    $line = [];
    foreach ($cols as $c){ $line[] = csvCell($r[$c] ?? ''); }
    $csv .= implode(',', $line)."\n";
  }
  $filename = ($entity.'-report-'.date('Ymd_His').'.csv');
  $toList = normalizeRecipients($recipients);
  if (!$toList) ApiResponse::error('No recipients');
  $okAll = true; $errors = [];
  foreach ($toList as $to){
    $ok = mailWithAttachment($to, $subject, 'Please find the report attached.', $filename, $csv, 'text/csv');
    if (!$ok) { $okAll = false; $errors[] = $to; }
  }
  if ($okAll) ApiResponse::success(['sent_to'=>$toList],'Email sent');
  ApiResponse::error('Failed to send to: '.implode(', ',$errors));
}

function runReportQuery(PDO $db, array $ENTITIES, int $companyId, string $entity, array $columns, $filters, $date_from, $date_to, $sort, int $page, int $pageSize){
  $cfg = $ENTITIES[$entity];
  $columnsParam = $columns && is_array($columns) ? $columns : $cfg['default_columns'];
  $sel = [];$colmap=[]; foreach ($columnsParam as $c){ if (isset($cfg['columns'][$c])) { $sel[] = $cfg['columns'][$c]." AS `".$c."`"; $colmap[$c]=1; } }
  if (!$sel){ foreach ($cfg['default_columns'] as $c){ $sel[]=$cfg['columns'][$c]." AS `".$c."`"; $colmap[$c]=1; } }
  $sql = 'SELECT '.implode(', ',$sel).' FROM '.$cfg['from'].' '.($cfg['joins']??'');
  $where=[]; $params=[];
  if (!empty($cfg['company_field'])) { $where[] = $cfg['company_field'].' = :cid'; $params[':cid']=$companyId; }
  $dateField = $cfg['date_field'];
  if ($date_from) { $where[] = "$dateField >= :dfrom"; $params[':dfrom']=$date_from; }
  if ($date_to) { $where[] = "$dateField <= :dto"; $params[':dto']=$date_to; }
  if ($filters){ if (is_string($filters)) $filters = json_decode($filters,true); if (is_array($filters)){ $i=0; foreach ($filters as $f){ $fld=$f['field']??''; if (!isset($cfg['columns'][$fld])) continue; $op=strtolower($f['op']??'eq'); $param=':pf'.$i++; switch($op){ case 'eq': $where[]=$cfg['columns'][$fld]." = $param"; $params[$param]=$f['value']; break; case 'ne': $where[]=$cfg['columns'][$fld]." <> $param"; $params[$param]=$f['value']; break; case 'like': $where[]=$cfg['columns'][$fld]." LIKE $param"; $params[$param]='%'.$f['value'].'%'; break; case 'gt': $where[]=$cfg['columns'][$fld]." > $param"; $params[$param]=$f['value']; break; case 'lt': $where[]=$cfg['columns'][$fld]." < $param"; $params[$param]=$f['value']; break; case 'gte': $where[]=$cfg['columns'][$fld]." >= $param"; $params[$param]=$f['value']; break; case 'lte': $where[]=$cfg['columns'][$fld]." <= $param"; $params[$param]=$f['value']; break; case 'between': $a=$param.'a'; $b=$param.'b'; $where[]=$cfg['columns'][$fld]." BETWEEN $a AND $b"; $params[$a]=$f['value'][0]??null; $params[$b]=$f['value'][1]??null; break; case 'in': if (is_array($f['value'])){ $ph=[]; foreach($f['value'] as $k=>$v){ $p=$param.'_'.$k; $ph[]=$p; $params[$p]=$v; } $where[]=$cfg['columns'][$fld].' IN ('.implode(',',$ph).')'; } break; } } } }
  if ($where) $sql .= ' WHERE '.implode(' AND ',$where);
  if ($sort){ $parts=explode(':',$sort); $sf=trim($parts[0]); $dir=strtolower($parts[1]??'asc'); $dir=$dir==='desc'?'DESC':'ASC'; if (isset($cfg['columns'][$sf])) $sql.=' ORDER BY '.$cfg['columns'][$sf].' '.$dir; }
  $offset = ($page-1)*$pageSize; $sql .= ' LIMIT '.$pageSize.' OFFSET '.$offset;
  $st=$db->prepare($sql); foreach ($params as $k=>$v) $st->bindValue($k,$v); $st->execute(); $rows=$st->fetchAll();
  return ['rows'=>$rows,'columns'=>array_keys($colmap)];
}

function normalizeRecipients($r){
  if (is_array($r)) return array_values(array_filter(array_map('trim',$r)));
  $r = trim((string)$r); if ($r==='') return [];
  $parts = preg_split('/[;,]+/',$r); return array_values(array_filter(array_map('trim',$parts)));
}
function csvCell($v){
  $s = (string)($v ?? '');
  if (strpos($s,'"')!==false || strpos($s,',')!==false || strpos($s,"\n")!==false){ $s = '"'.str_replace('"','""',$s).'"'; }
  return $s;
}
function mailWithAttachment($to, $subject, $message, $filename, $data, $mime){
  // Get from settings
  $fromName = 'SmartQuantumHR'; $fromEmail = 'no-reply@localhost';
  try {
    $db = (new Database())->getConnection();
    $st = $db->prepare('SELECT email_from_name, email_from_email FROM companies WHERE id = :id');
    $st->bindValue(':id', getCurrentUser()['company_id'], PDO::PARAM_INT); $st->execute(); $row=$st->fetch()?:[];
    if (!empty($row['email_from_name'])) $fromName = $row['email_from_name'];
    if (!empty($row['email_from_email'])) $fromEmail = $row['email_from_email'];
  } catch (Throwable $e) { /* fallback defaults */ }
  $boundary = 'bnd_'.md5(uniqid('',true));
  $headers = "MIME-Version: 1.0\r\n".
             "From: ".$fromName." <".$fromEmail.">\r\n".
             "Content-Type: multipart/mixed; boundary=\"$boundary\"\r\n";
  $body  = "--$boundary\r\n";
  $body .= "Content-Type: text/plain; charset=UTF-8\r\n\r\n".$message."\r\n";
  $body .= "--$boundary\r\n";
  $body .= "Content-Type: $mime; name=\"$filename\"\r\n";
  $body .= "Content-Transfer-Encoding: base64\r\n";
  $body .= "Content-Disposition: attachment; filename=\"$filename\"\r\n\r\n";
  $body .= chunk_split(base64_encode($data))."\r\n";
  $body .= "--$boundary--";
  return @mail($to, $subject, $body, $headers);
}

function saveSchedule(PDO $db, int $companyId, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $savedId = (int)($in['saved_report_id'] ?? 0); if ($savedId<=0) ApiResponse::error('saved_report_id required');
  $freq = $in['frequency'] ?? 'monthly';
  $dow = isset($in['day_of_week']) ? (int)$in['day_of_week'] : null;
  $dom = isset($in['day_of_month']) ? (int)$in['day_of_month'] : null;
  $tod = $in['time_of_day'] ?? '08:00';
  $rec = is_array($in['recipients']) ? json_encode($in['recipients']) : json_encode(normalizeRecipients($in['recipients'] ?? ''));
  $fmt = strtolower($in['format'] ?? 'csv'); if (!in_array($fmt,['csv','xlsx','pdf'])) $fmt='csv';
  $next = date('Y-m-d H:i:s', strtotime('+1 hour')); // simple placeholder; can be refined
  $st = $db->prepare("INSERT INTO report_schedules (company_id, saved_report_id, frequency, day_of_week, day_of_month, time_of_day, recipients_json, format, active, next_run_at, created_at)
                      VALUES (:cid,:rid,:f,:dow,:dom,:tod,:rec,:fmt,1,:next,NOW())");
  foreach ([[':cid',$companyId,PDO::PARAM_INT],[':rid',$savedId,PDO::PARAM_INT]] as $p){ $st->bindValue($p[0],$p[1],$p[2]); }
  $st->bindValue(':f',$freq); $st->bindValue(':dow',$dow); $st->bindValue(':dom',$dom); $st->bindValue(':tod',$tod); $st->bindValue(':rec',$rec); $st->bindValue(':fmt',$fmt); $st->bindValue(':next',$next);
  $st->execute();
  ApiResponse::success(null,'Schedule saved');
}

function runDueSchedules(PDO $db, array $ENTITIES, int $companyId, array $user){
  // Only admin/hr_head
  if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) ApiResponse::forbidden('Not allowed');
  $now = date('Y-m-d H:i:s');
  $st = $db->prepare('SELECT rs.*, sr.entity, sr.columns_json, sr.filters_json, sr.date_from, sr.date_to, sr.sort, sr.name FROM report_schedules rs JOIN saved_reports sr ON rs.saved_report_id = sr.id WHERE rs.company_id = :cid AND rs.active = 1 AND (rs.next_run_at IS NULL OR rs.next_run_at <= :now)');
  $st->bindValue(':cid',$companyId,PDO::PARAM_INT); $st->bindValue(':now',$now); $st->execute();
  $count=0; while ($row=$st->fetch()){
    $rowsAndCols = runReportQuery($db, $ENTITIES, $companyId, $row['entity'], json_decode($row['columns_json'],true)?:[], json_decode($row['filters_json'],true)?:[], $row['date_from'], $row['date_to'], $row['sort'], 1, 100000);
    $csv = implode(',', array_map('csvCell',$rowsAndCols['columns']))."\n"; foreach ($rowsAndCols['rows'] as $r){ $line=[]; foreach ($rowsAndCols['columns'] as $c){ $line[]=csvCell($r[$c]??''); } $csv.=implode(',',$line)."\n"; }
    $to = json_decode($row['recipients_json'], true) ?: [];
    foreach ($to as $addr){ mailWithAttachment($addr, 'Scheduled Report: '.$row['name'], 'Attached scheduled report.', 'report.csv', $csv, 'text/csv'); }
    $count++;
    // rudimentary next_run_at bump
    $next = date('Y-m-d H:i:s', strtotime('+1 day'));
    $up = $db->prepare('UPDATE report_schedules SET last_run_at = :now, next_run_at = :next WHERE id = :id');
    $up->execute([':now'=>$now, ':next'=>$next, ':id'=>$row['id']]);
  }
  ApiResponse::success(['ran'=>$count],'Done');
}
