<?php
/**
 * Recruitment API: job postings, candidates, applications
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$database = new Database();
$db = $database->getConnection();

function canRecruit($user){
  return in_array(($user['role_slug'] ?? ''), ['super_admin','admin','hr_head','hr_officer'], true);
}

switch ($method) {
  case 'GET':
    if ($action === 'jobs') listJobs($db);
    elseif ($action === 'job') getJob($db);
    elseif ($action === 'candidates') listCandidates($db);
    elseif ($action === 'applications') listApplications($db);
    elseif ($action === 'lists') lists($db);
    elseif ($action === 'list_interviews') listInterviews($db);
    else ApiResponse::error('Invalid action');
    break;
  case 'POST':
    if ($action === 'create_job') createJob($db);
    elseif ($action === 'update_job') updateJob($db);
    elseif ($action === 'toggle_job') toggleJob($db);
    elseif ($action === 'create_candidate') createCandidate($db);
    elseif ($action === 'update_candidate') updateCandidate($db);
    elseif ($action === 'create_application') createApplication($db);
    elseif ($action === 'update_application') updateApplication($db);
    elseif ($action === 'upload_resume') uploadResume($db);
    elseif ($action === 'upload_cover') uploadCover($db);
    elseif ($action === 'schedule_interview') scheduleInterview($db);
    elseif ($action === 'convert_hired') convertHired($db);
    else ApiResponse::error('Invalid action');
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function lists(PDO $db){
  $u = getCurrentUser();
  $cid = (int)$u['company_id'];
  try{
    $deps = $db->prepare("SELECT id, name FROM departments WHERE company_id = :cid AND status <> 'inactive' ORDER BY name");
    $deps->execute([':cid'=>$cid]);
    $pos = $db->prepare("SELECT id, title FROM positions WHERE company_id = :cid AND status <> 'inactive' ORDER BY title");
    $pos->execute([':cid'=>$cid]);
    ApiResponse::success([
      'departments' => $deps->fetchAll(),
      'positions' => $pos->fetchAll(),
      'employment_types' => ['full_time','part_time','contract','internship','temporary'],
      'job_statuses' => ['draft','open','closed'],
      'application_statuses' => ['applied','shortlisted','interview','offered','hired','rejected'],
      'candidate_statuses' => ['new','active','inactive']
    ]);
  }catch(Throwable $e){ ApiResponse::error('Failed to load lists: '.$e->getMessage(), 500); }
}

function listJobs(PDO $db){
  $u = getCurrentUser();
  $cid = (int)$u['company_id'];
  $status = isset($_GET['status']) ? trim((string)$_GET['status']) : '';
  $deptId = isset($_GET['department_id']) ? (int)$_GET['department_id'] : 0;
  $conds = ['jp.company_id = :cid']; $params = [':cid'=>$cid];
  if ($status !== '' && in_array($status, ['draft','open','closed'], true)) { $conds[] = 'jp.status = :st'; $params[':st'] = $status; }
  if ($deptId > 0) { $conds[] = 'jp.department_id = :did'; $params[':did'] = $deptId; }
  try{
    $st = $db->prepare("SELECT jp.id, jp.title, jp.department_id, d.name AS department_name,
                               jp.position_id, p.title AS position_title,
                               jp.employment_type, jp.min_salary, jp.max_salary, jp.remote_work,
                               jp.deadline, jp.status, jp.created_at
                        FROM job_postings jp
                        LEFT JOIN departments d ON d.id = jp.department_id
                        LEFT JOIN positions p ON p.id = jp.position_id
                        WHERE ".implode(' AND ',$conds)."
                        ORDER BY jp.created_at DESC");
    $st->execute($params);
    ApiResponse::success($st->fetchAll());
  }catch(Throwable $e){ ApiResponse::error('Failed to load jobs: '.$e->getMessage(), 500); }
}

function getJob(PDO $db){
  $u = getCurrentUser(); $cid = (int)$u['company_id'];
  $id = isset($_GET['id']) ? (int)$_GET['id'] : 0; if ($id<=0) ApiResponse::error('Invalid job');
  try{
    $st = $db->prepare("SELECT jp.*, d.name AS department_name, p.title AS position_title
                        FROM job_postings jp
                        LEFT JOIN departments d ON d.id = jp.department_id
                        LEFT JOIN positions p ON p.id = jp.position_id
                        WHERE jp.id = :id AND jp.company_id = :cid LIMIT 1");
    $st->execute([':id'=>$id, ':cid'=>$cid]);
    if ($st->rowCount()===0) ApiResponse::notFound('Job not found');
    ApiResponse::success($st->fetch());
  }catch(Throwable $e){ ApiResponse::error('Failed to load job: '.$e->getMessage(), 500); }
}

function listCandidates(PDO $db){
  $u = getCurrentUser();
  $cid = (int)$u['company_id'];
  try{
    $st = $db->prepare("SELECT id, first_name, last_name, email, phone, source, status, linkedin, portfolio, resume_path, created_at
                        FROM candidates WHERE company_id = :cid ORDER BY created_at DESC");
    $st->execute([':cid'=>$cid]);
    ApiResponse::success($st->fetchAll());
  }catch(Throwable $e){ ApiResponse::error('Failed to load candidates: '.$e->getMessage(), 500); }
}

function listApplications(PDO $db){
  $u = getCurrentUser();
  $cid = (int)$u['company_id'];
  $status = isset($_GET['status']) ? trim((string)$_GET['status']) : '';
  $deptId = isset($_GET['department_id']) ? (int)$_GET['department_id'] : 0;
  $jobId = isset($_GET['job_id']) ? (int)$_GET['job_id'] : 0;
  try{
    $conds = ['jp.company_id = :cid']; $params = [':cid'=>$cid];
    if ($status !== '' && in_array($status, ['applied','shortlisted','interview','offered','hired','rejected'], true)) { $conds[] = 'a.status = :st'; $params[':st'] = $status; }
    if ($deptId > 0) { $conds[] = 'jp.department_id = :did'; $params[':did'] = $deptId; }
    if ($jobId > 0) { $conds[] = 'a.job_posting_id = :jid'; $params[':jid'] = $jobId; }
    $sql = "SELECT a.id, a.job_posting_id, a.candidate_id, a.status, a.application_date, a.cover_letter, a.notes,
                               jp.title AS job_title,
                               c.first_name, c.last_name, c.email, c.resume_path
                        FROM job_applications a
                        JOIN job_postings jp ON jp.id = a.job_posting_id AND jp.company_id = :cid
                        JOIN candidates c ON c.id = a.candidate_id AND c.company_id = :cid
                        WHERE ".implode(' AND ', $conds)."
                        ORDER BY a.application_date DESC, a.id DESC";
    $st = $db->prepare($sql);
    $st->execute($params);
    ApiResponse::success($st->fetchAll());
  }catch(Throwable $e){ ApiResponse::error('Failed to load applications: '.$e->getMessage(), 500); }
}

function createJob(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $title = trim((string)($in['title'] ?? ''));
  if ($title === '') ApiResponse::error('Title is required');
  try{
    $st = $db->prepare("INSERT INTO job_postings (company_id, position_id, department_id, title, description, requirements, responsibilities,
                        min_salary, max_salary, employment_type, remote_work, deadline, status, posted_by, created_at)
                        VALUES (:cid, :pid, :did, :title, :desc, :reqs, :resp, :min, :max, :etype, :remote, :deadline, :status, :by, NOW())");
    $st->execute([
      ':cid'=>$u['company_id'],
      ':pid'=> isset($in['position_id']) ? (int)$in['position_id'] : null,
      ':did'=> isset($in['department_id']) ? (int)$in['department_id'] : null,
      ':title'=>$title,
      ':desc'=> $in['description'] ?? null,
      ':reqs'=> $in['requirements'] ?? null,
      ':resp'=> $in['responsibilities'] ?? null,
      ':min'=> isset($in['min_salary']) ? (float)$in['min_salary'] : null,
      ':max'=> isset($in['max_salary']) ? (float)$in['max_salary'] : null,
      ':etype'=> $in['employment_type'] ?? 'full_time',
      ':remote'=> !empty($in['remote_work']) ? 1 : 0,
      ':deadline'=> $in['deadline'] ?? null,
      ':status'=> $in['status'] ?? 'open',
      ':by'=> $u['id']
    ]);
    ApiResponse::success(['id'=>(int)$db->lastInsertId()], 'Job created');
  }catch(Throwable $e){ ApiResponse::error('Failed to create job: '.$e->getMessage(), 500); }
}

function updateJob(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $id = isset($in['id']) ? (int)$in['id'] : 0; if ($id<=0) ApiResponse::error('Invalid job');
  // ensure belongs to company
  $chk = $db->prepare('SELECT company_id FROM job_postings WHERE id = :id'); $chk->execute([':id'=>$id]);
  $cid = (int)$chk->fetchColumn(); if (!$cid || $cid !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  $allowed = ['position_id','department_id','title','description','requirements','responsibilities','min_salary','max_salary','employment_type','remote_work','deadline','status'];
  $set = []; $params = [':id'=>$id];
  foreach($allowed as $f){ if (array_key_exists($f,$in)){ $set[] = "$f = :$f"; $params[":$f"] = $in[$f]; } }
  if (!$set) ApiResponse::error('No fields to update');
  $sql = 'UPDATE job_postings SET '.implode(', ',$set).', updated_at = NOW() WHERE id = :id';
  try{ $st = $db->prepare($sql); $st->execute($params); ApiResponse::success(null,'Job updated'); }catch(Throwable $e){ ApiResponse::error('Update failed: '.$e->getMessage(),500); }
}

function toggleJob(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $id = isset($in['id']) ? (int)$in['id'] : 0; $status = in_array(($in['status'] ?? ''), ['draft','open','closed'], true) ? $in['status'] : 'open';
  if ($id<=0) ApiResponse::error('Invalid job');
  try{ $st = $db->prepare('UPDATE job_postings SET status = :s, updated_at = NOW() WHERE id = :id AND company_id = :cid'); $st->execute([':s'=>$status, ':id'=>$id, ':cid'=>$u['company_id']]); ApiResponse::success(null,'Job status updated'); }catch(Throwable $e){ ApiResponse::error('Failed to update job',500); }
}

function createCandidate(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $first = trim((string)($in['first_name'] ?? '')); $last = trim((string)($in['last_name'] ?? ''));
  if ($first === '' && $last === '') ApiResponse::error('Name required');
  try{
    $st = $db->prepare('INSERT INTO candidates (company_id, first_name, last_name, email, phone, address, resume_path, linkedin, portfolio, source, status, notes, created_at)
                        VALUES (:cid, :fn, :ln, :em, :ph, :addr, :resume, :li, :pf, :src, :st, :notes, NOW())');
    $st->execute([
      ':cid'=>$u['company_id'], ':fn'=>$first, ':ln'=>$last,
      ':em'=> $in['email'] ?? null, ':ph'=> $in['phone'] ?? null, ':addr'=> $in['address'] ?? null,
      ':resume'=> $in['resume_path'] ?? null, ':li'=> $in['linkedin'] ?? null, ':pf'=> $in['portfolio'] ?? null,
      ':src'=> $in['source'] ?? 'direct', ':st'=> $in['status'] ?? 'new', ':notes'=> $in['notes'] ?? null
    ]);
    ApiResponse::success(['id'=>(int)$db->lastInsertId()], 'Candidate created');
  }catch(Throwable $e){ ApiResponse::error('Failed to create candidate: '.$e->getMessage(), 500); }
}

function updateCandidate(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $id = isset($in['id']) ? (int)$in['id'] : 0; if ($id<=0) ApiResponse::error('Invalid candidate');
  // ensure belongs to company
  $chk = $db->prepare('SELECT company_id FROM candidates WHERE id = :id'); $chk->execute([':id'=>$id]);
  $cid = (int)$chk->fetchColumn(); if (!$cid || $cid !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  $allowed = ['first_name','last_name','email','phone','address','resume_path','linkedin','portfolio','source','status','notes'];
  $set = []; $params = [':id'=>$id];
  foreach($allowed as $f){ if (array_key_exists($f,$in)){ $set[] = "$f = :$f"; $params[":$f"] = $in[$f]; } }
  if (!$set) ApiResponse::error('No fields to update');
  $sql = 'UPDATE candidates SET '.implode(', ',$set).', updated_at = NOW() WHERE id = :id';
  try{ $st = $db->prepare($sql); $st->execute($params); ApiResponse::success(null,'Candidate updated'); }catch(Throwable $e){ ApiResponse::error('Update failed: '.$e->getMessage(),500); }
}

function createApplication(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $jobId = isset($in['job_posting_id']) ? (int)$in['job_posting_id'] : 0;
  $candId = isset($in['candidate_id']) ? (int)$in['candidate_id'] : 0;
  if ($jobId<=0 || $candId<=0) ApiResponse::error('Job and candidate are required');
  try{
    // company ownership
    $j = $db->prepare('SELECT company_id FROM job_postings WHERE id = :id'); $j->execute([':id'=>$jobId]);
    $c = $db->prepare('SELECT company_id FROM candidates WHERE id = :id'); $c->execute([':id'=>$candId]);
    $jc = (int)$j->fetchColumn(); $cc = (int)$c->fetchColumn();
    if (!$jc || !$cc || $jc !== (int)$u['company_id'] || $cc !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
    $st = $db->prepare('INSERT INTO job_applications (job_posting_id, candidate_id, status, application_date, cover_letter, notes) VALUES (:jid,:cid,:st, NOW(), :cl, :notes)');
    $st->execute([':jid'=>$jobId, ':cid'=>$candId, ':st'=>$in['status'] ?? 'applied', ':cl'=>$in['cover_letter'] ?? null, ':notes'=>$in['notes'] ?? null]);
    ApiResponse::success(['id'=>(int)$db->lastInsertId()], 'Application created');
  }catch(Throwable $e){ ApiResponse::error('Failed to create application: '.$e->getMessage(), 500); }
}

function updateApplication(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $id = isset($in['id']) ? (int)$in['id'] : 0; if ($id<=0) ApiResponse::error('Invalid application');
  // ensure belongs to company
  $chk = $db->prepare('SELECT jp.company_id FROM job_applications a JOIN job_postings jp ON jp.id = a.job_posting_id WHERE a.id = :id');
  $chk->execute([':id'=>$id]);
  $cid = (int)$chk->fetchColumn(); if (!$cid || $cid !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  $allowed = ['status','notes','cover_letter'];
  $set = []; $params = [':id'=>$id];
  foreach($allowed as $f){ if (array_key_exists($f,$in)){ $set[] = "$f = :$f"; $params[":$f"] = $in[$f]; } }
  if (!$set) ApiResponse::error('No fields to update');
  $sql = 'UPDATE job_applications SET '.implode(', ',$set).', application_date = application_date WHERE id = :id';
  try{ $st = $db->prepare($sql); $st->execute($params); ApiResponse::success(null,'Application updated'); }catch(Throwable $e){ ApiResponse::error('Update failed: '.$e->getMessage(),500); }
}

// ==== File Uploads ====
function uploadResume(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $cid = (int)$u['company_id'];
  $candId = isset($_POST['candidate_id']) ? (int)$_POST['candidate_id'] : 0; if ($candId<=0) ApiResponse::error('Invalid candidate');
  // verify candidate belongs to company
  $chk = $db->prepare('SELECT company_id FROM candidates WHERE id = :id'); $chk->execute([':id'=>$candId]);
  $cc = (int)$chk->fetchColumn(); if ($cc !== $cid) ApiResponse::forbidden('Not allowed');
  $rel = saveUploadedFile('recruitment/resumes');
  if (!$rel) ApiResponse::error('Upload failed');
  try{ $st = $db->prepare('UPDATE candidates SET resume_path = :p WHERE id = :id'); $st->execute([':p'=>$rel, ':id'=>$candId]); ApiResponse::success(['path'=>$rel],'Resume uploaded'); }catch(Throwable $e){ ApiResponse::error('Failed to save resume path: '.$e->getMessage(),500); }
}

function uploadCover(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $cid = (int)$u['company_id'];
  $appId = isset($_POST['application_id']) ? (int)$_POST['application_id'] : 0; if ($appId<=0) ApiResponse::error('Invalid application');
  $chk = $db->prepare('SELECT jp.company_id FROM job_applications a JOIN job_postings jp ON jp.id = a.job_posting_id WHERE a.id = :id'); $chk->execute([':id'=>$appId]);
  $cc = (int)$chk->fetchColumn(); if ($cc !== $cid) ApiResponse::forbidden('Not allowed');
  $rel = saveUploadedFile('recruitment/covers'); if (!$rel) ApiResponse::error('Upload failed');
  // store file path in notes JSON
  try{
    $sel = $db->prepare('SELECT notes FROM job_applications WHERE id = :id'); $sel->execute([':id'=>$appId]); $notes = $sel->fetchColumn();
    $obj = json_decode($notes, true); if (!is_array($obj)) $obj = [];
    $obj['cover_file'] = $rel;
    $up = $db->prepare('UPDATE job_applications SET notes = :n WHERE id = :id'); $up->execute([':n'=>json_encode($obj), ':id'=>$appId]);
    ApiResponse::success(['path'=>$rel],'Cover letter uploaded');
  }catch(Throwable $e){ ApiResponse::error('Failed to update application: '.$e->getMessage(),500); }
}

function saveUploadedFile($subdir){
  if (!isset($_FILES['file']) || !is_array($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) return null;
  $fn = $_FILES['file']['name'] ?? 'file';
  $ext = strtolower(pathinfo($fn, PATHINFO_EXTENSION));
  $allowed = ['pdf','doc','docx','png','jpg','jpeg','txt'];
  if (!in_array($ext, $allowed, true)) return null;
  if (($_FILES['file']['size'] ?? 0) > 10*1024*1024) return null; // 10MB
  $base = dirname(__DIR__); // project root
  $dir = rtrim($base, DIRECTORY_SEPARATOR).DIRECTORY_SEPARATOR.'uploads'.DIRECTORY_SEPARATOR.$subdir;
  if (!is_dir($dir)) @mkdir($dir, 0775, true);
  $safe = preg_replace('/[^a-zA-Z0-9._-]/','_', basename($fn));
  $name = date('Ymd_His').'_'.$safe;
  $path = $dir.DIRECTORY_SEPARATOR.$name;
  if (!@move_uploaded_file($_FILES['file']['tmp_name'], $path)) return null;
  $rel = 'uploads/'.trim($subdir,'/').'/'.$name;
  return $rel;
}

// ==== Interviews via activities ====
function scheduleInterview(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = $_POST ?: (json_decode(file_get_contents('php://input'), true) ?: []);
  $appId = isset($in['application_id']) ? (int)$in['application_id'] : 0; if ($appId<=0) ApiResponse::error('Invalid application');
  $when = trim((string)($in['scheduled_at'] ?? '')); if ($when==='') ApiResponse::error('scheduled_at is required');
  $interviewer = trim((string)($in['interviewer'] ?? ''));
  $notes = trim((string)($in['notes'] ?? ''));
  $rating = isset($in['rating']) ? (int)$in['rating'] : null;
  try{
    // ensure ownership
    $chk = $db->prepare('SELECT jp.company_id FROM job_applications a JOIN job_postings jp ON jp.id = a.job_posting_id WHERE a.id = :id'); $chk->execute([':id'=>$appId]);
    $cid = (int)$chk->fetchColumn(); if ($cid !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
    $md = json_encode(['scheduled_at'=>$when, 'interviewer'=>$interviewer, 'notes'=>$notes, 'rating'=>$rating]);
    $ins = $db->prepare("INSERT INTO activities (company_id, user_id, action, target_type, target_id, description, metadata, created_at)
                         VALUES (:cid, :uid, 'interview', 'job_application', :tid, :desc, :meta, NOW())");
    $ins->execute([':cid'=>$u['company_id'], ':uid'=>$u['id'], ':tid'=>$appId, ':desc'=>'Interview scheduled', ':meta'=>$md]);
    ApiResponse::success(null,'Interview scheduled');
  }catch(Throwable $e){ ApiResponse::error('Failed to schedule: '.$e->getMessage(),500); }
}

function listInterviews(PDO $db){
  $u = getCurrentUser();
  $appId = isset($_GET['application_id']) ? (int)$_GET['application_id'] : 0; if ($appId<=0) ApiResponse::error('Invalid application');
  try{
    $st = $db->prepare("SELECT id, description, metadata, created_at FROM activities WHERE company_id = :cid AND action = 'interview' AND target_type = 'job_application' AND target_id = :tid ORDER BY created_at DESC");
    $st->execute([':cid'=>$u['company_id'], ':tid'=>$appId]);
    $rows = $st->fetchAll();
    foreach ($rows as &$r){ $r['metadata'] = json_decode($r['metadata'], true); }
    ApiResponse::success($rows);
  }catch(Throwable $e){ ApiResponse::error('Failed to load interviews: '.$e->getMessage(),500); }
}

// ==== Conversion to Employee ====
function convertHired(PDO $db){
  $u = getCurrentUser(); if (!canRecruit($u)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $appId = isset($in['application_id']) ? (int)$in['application_id'] : 0; if ($appId<=0) ApiResponse::error('Invalid application');
  try{
    $st = $db->prepare('SELECT a.status, a.candidate_id, a.job_posting_id, c.first_name, c.last_name, c.email, c.phone FROM job_applications a JOIN candidates c ON c.id = a.candidate_id WHERE a.id = :id');
    $st->execute([':id'=>$appId]);
    if ($st->rowCount()===0) ApiResponse::notFound('Application not found');
    $row = $st->fetch();
    if (($row['status'] ?? '') !== 'hired') ApiResponse::error('Application is not in hired status');
    // fetch job to get dept/position
    $j = $db->prepare('SELECT department_id, position_id FROM job_postings WHERE id = :id AND company_id = :cid');
    $j->execute([':id'=>$row['job_posting_id'], ':cid'=>$u['company_id']]);
    $job = $j->fetch() ?: [];

    // dedupe by email
    if (!empty($row['email'])){
      $d = $db->prepare('SELECT id FROM employees WHERE company_id = :cid AND (email = :em OR official_email = :em) LIMIT 1');
      $d->execute([':cid'=>$u['company_id'], ':em'=>$row['email']]);
      $exists = $d->fetchColumn();
      if ($exists) ApiResponse::success(['employee_id'=>(int)$exists], 'Already converted');
    }

    $empNum = 'EMP'.date('YmdHis').substr((string)rand(1000,9999),-4);
    // Use status column for employee status to match schema
    $ins = $db->prepare("INSERT INTO employees (company_id, department_id, position_id, employee_number, first_name, last_name, email, phone, hire_date, status, created_by, created_at)
                         VALUES (:cid, :dep, :pos, :enum, :fn, :ln, :em, :ph, CURDATE(), 'active', :by, NOW())");
    $ins->execute([':cid'=>$u['company_id'], ':dep'=>$job['department_id']??null, ':pos'=>$job['position_id']??null, ':enum'=>$empNum, ':fn'=>$row['first_name']??'', ':ln'=>$row['last_name']??'', ':em'=>$row['email']??null, ':ph'=>$row['phone']??null, ':by'=>$u['id']]);
    $eid = (int)$db->lastInsertId();
    ApiResponse::success(['employee_id'=>$eid, 'employee_number'=>$empNum], 'Candidate converted to employee');
  }catch(Throwable $e){ ApiResponse::error('Conversion failed: '.$e->getMessage(),500); }
}
