<?php
/**
 * Payroll Management API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

// Ensure payout-related columns exist in payrolls
ensurePayrollPayoutSchema($db);
// Ensure new component calculation columns exist (idempotent)
ensurePayrollComponentCalcSchema($db);
// Ensure employee default payout columns exist (idempotent)
ensureEmployeePaymentSchema($db);

// Check if a table has a specific column (cached per request)
function tableHasColumn(PDO $db, $table, $column) {
    static $cache = [];
    $key = $table . '::' . $column;
    if (array_key_exists($key, $cache)) return $cache[$key];
    try {
        $q = $db->prepare("SELECT 1 FROM information_schema.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c");
        $q->bindValue(':t', $table);
        $q->bindValue(':c', $column);
        $q->execute();
        return $cache[$key] = ($q->rowCount() > 0);
    } catch (Throwable $e) {
        return $cache[$key] = false;
    }
}
// Ensure payout columns on payrolls table (idempotent)
function ensurePayrollPayoutSchema(PDO $db) {
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN payout_channel VARCHAR(32) NULL AFTER status"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN payout_status VARCHAR(32) NULL AFTER payout_channel"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN payout_reference VARCHAR(191) NULL AFTER payout_status"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN payout_meta LONGTEXT NULL AFTER payout_reference"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN paid_amount DECIMAL(12,2) NULL AFTER payout_meta"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN paid_by INT NULL AFTER paid_amount"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE payrolls ADD COLUMN paid_at TIMESTAMP NULL AFTER paid_by"); } catch (Throwable $e) {}
}

// Ensure component tables have calc_mode and rate columns for percentage-based calculations
function ensurePayrollComponentCalcSchema(PDO $db) {
    $tables = ['grade_allowances','employee_allowances','grade_deductions','employee_deductions'];
    foreach ($tables as $t) {
        try { $db->exec("ALTER TABLE $t ADD COLUMN calc_mode ENUM('fixed','percent_basic','percent_gross') NOT NULL DEFAULT 'fixed' AFTER amount"); } catch (Throwable $e) {}
        try { $db->exec("ALTER TABLE $t ADD COLUMN rate DECIMAL(5,2) NULL AFTER calc_mode"); } catch (Throwable $e) {}
    }
}

// Ensure employee payment default columns (bank/momo)
function ensureEmployeePaymentSchema(PDO $db) {
    try { $db->exec("ALTER TABLE employees ADD COLUMN bank_code VARCHAR(32) NULL AFTER phone"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN bank_account VARCHAR(64) NULL AFTER bank_code"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN momo_provider VARCHAR(64) NULL AFTER bank_account"); } catch (Throwable $e) {}
    try { $db->exec("ALTER TABLE employees ADD COLUMN momo_number VARCHAR(64) NULL AFTER momo_provider"); } catch (Throwable $e) {}
}

// Simple storage config path helper
function storage_config_path($rel){
    $base = realpath(__DIR__ . '/../storage/config');
    if ($base === false) {
        @mkdir(__DIR__ . '/../storage/config', 0777, true);
        $base = realpath(__DIR__ . '/../storage/config');
    }
    return rtrim($base ?: (__DIR__ . '/../storage/config'), '/\\') . '/' . ltrim($rel, '/\\');
}

// Load Paystack config for a company from storage/config/paystack-<company_id>.json
function paystack_load_config(int $companyId): array {
    $file = storage_config_path('paystack-' . $companyId . '.json');
    $cfg = is_file($file) ? (json_decode(@file_get_contents($file), true) ?: []) : [];
    // Allow env override
    if (empty($cfg['secret_key']) && getenv('PAYSTACK_SECRET_KEY')) $cfg['secret_key'] = getenv('PAYSTACK_SECRET_KEY');
    if (empty($cfg['currency'])) $cfg['currency'] = 'GHS';
    return $cfg;
}

// Minimal HTTP POST JSON helper
function http_post_json(string $url, array $payload, array $headers = []): array {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, array_merge(['Content-Type: application/json'], $headers));
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $body = curl_exec($ch);
    $errno = curl_errno($ch);
    $err = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    $json = null; if ($body !== false) { $json = json_decode($body, true); }
    return ['ok' => ($errno === 0 && $code >= 200 && $code < 300), 'status' => $code, 'body' => $body, 'json' => $json, 'error' => $errno ? $err : null];
}

function paystack_create_recipient(array $cfg, array $recipient): array {
    $headers = ['Authorization: Bearer ' . ($cfg['secret_key'] ?? '')];
    return http_post_json('https://api.paystack.co/transferrecipient', $recipient, $headers);
}

function paystack_initiate_transfer(array $cfg, array $transfer): array {
    $headers = ['Authorization: Bearer ' . ($cfg['secret_key'] ?? '')];
    return http_post_json('https://api.paystack.co/transfer', $transfer, $headers);
}

function renderPayslipHtml(PDO $db, $id) {
    $user = getCurrentUser();
    if (!$id) { header('Content-Type: text/plain; charset=UTF-8'); echo 'Payroll ID required'; exit; }

    // Build same base query as getPayslip
    $query = "SELECT 
                p.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                e.email,
                e.date_of_birth,
                e.hire_date,
                d.name as department_name,
                pos.title as position_title,
                c.name as company_name,
                c.address as company_address,
                c.logo AS company_logo
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN positions pos ON e.position_id = pos.id
              LEFT JOIN companies c ON c.id = p.company_id
              WHERE p.id = :id";

    $params = [':id' => $id];
    if ($user['role_slug'] === 'employee') {
        $empStmt = $db->prepare("SELECT id FROM employees WHERE user_id = :user_id");
        $empStmt->bindValue(':user_id', $user['id']);
        $empStmt->execute();
        $employee = $empStmt->fetch();
        if (!$employee) { header('Content-Type: text/plain; charset=UTF-8'); echo 'Access denied'; exit; }
        $query .= " AND p.employee_id = :employee_id";
        $params[':employee_id'] = $employee['id'];
    } else {
        $query .= " AND p.company_id = :company_id";
        $params[':company_id'] = $user['company_id'];
    }
    $query .= " AND p.status IN ('calculated','approved','paid')";

    $stmt = $db->prepare($query);
    foreach ($params as $k=>$v) { $stmt->bindValue($k, $v); }
    $stmt->execute();
    if ($stmt->rowCount() === 0) { header('Content-Type: text/plain; charset=UTF-8'); echo 'Payslip not found'; exit; }
    $payslip = $stmt->fetch();
    // Attach detail
    $basic = (float)($payslip['basic_salary'] ?? 0);
    $gross = (float)($payslip['gross_salary'] ?? ($basic + (float)($payslip['total_allowances'] ?? 0)));
    $payslip['allowances'] = getPayrollAllowances($db, $payslip['employee_id'], $payslip['pay_period_end'], $basic, $gross);
    $payslip['deductions'] = getPayrollDeductions($db, $payslip['employee_id'], $payslip['pay_period_end'], $basic, $gross);

    $html = buildPayslipHtmlServer($payslip);
    if (!empty($_GET['print'])) {
        // Inject print-on-load script just before </head>
        $html = preg_replace('/<\/head>/', "<script>window.addEventListener('DOMContentLoaded',function(){try{window.print();}catch(e){}});</script></head>", $html, 1);
    }
    header('Content-Type: text/html; charset=UTF-8');
    echo $html;
    exit;
}

// ===== Payslip Email Helpers ===== //

function loadPayslipData(PDO $db, int $id) {
    // Load core payslip fields without role scoping (internal use)
    $q = $db->prepare("SELECT p.*, 
                              CONCAT(e.first_name,' ',e.last_name) AS employee_name, 
                              e.employee_number, e.email,
                              e.date_of_birth, e.hire_date,
                              d.name AS department_name, pos.title AS position_title,
                              c.name AS company_name, c.address AS company_address, c.logo AS company_logo
                        FROM payrolls p
                        JOIN employees e ON p.employee_id = e.id
                        LEFT JOIN departments d ON e.department_id = d.id
                        LEFT JOIN positions pos ON e.position_id = pos.id
                        LEFT JOIN companies c ON c.id = p.company_id
                        WHERE p.id = :id");
    $q->execute([':id'=>$id]);
    if ($q->rowCount() === 0) return null;
    $p = $q->fetch();
    // Attach details
    $basic = (float)($p['basic_salary'] ?? 0);
    $gross = (float)($p['gross_salary'] ?? ($basic + (float)($p['total_allowances'] ?? 0)));
    $p['allowances'] = getPayrollAllowances($db, $p['employee_id'], $p['pay_period_end'], $basic, $gross);
    $p['deductions'] = getPayrollDeductions($db, $p['employee_id'], $p['pay_period_end'], $basic, $gross);
    return $p;
}

function buildPayslipHtmlServer(array $p) {
    // Mirror the client template in a minimal way
    $fmt = function($v){ $n = (float)($v ?? 0); return number_format($n, 2, '.', ''); };
    $allowances = is_array($p['allowances'] ?? null) ? $p['allowances'] : [];
    $deductions = is_array($p['deductions'] ?? null) ? $p['deductions'] : [];
    $earnRows = array_merge(
        [ ['name'=>'Basic Pay','amount'=>(float)($p['basic_salary']??0)] ],
        array_map(fn($a)=> ['name'=>$a['name'] ?? ($a['code'] ?? 'Allowance'), 'amount'=>(float)$a['amount']], $allowances)
    );
    $dedRows = array_merge(
        [ ['name'=>'Employee SSF (5.5%)','amount'=>(float)($p['ssnit_employee']??0)], ['name'=>'PAYE','amount'=>(float)($p['total_tax']??0)] ],
        array_map(fn($d)=> ['name'=>$d['name'] ?? ($d['code'] ?? 'Deduction'), 'amount'=>(float)$d['amount']], $deductions)
    );
    $max = max(count($earnRows), count($dedRows));
    while (count($earnRows) < $max) $earnRows[] = ['name'=>'','amount'=>''];
    while (count($dedRows) < $max) $dedRows[] = ['name'=>'','amount'=>''];
    $totalE = (float)($p['basic_salary']??0) + (float)($p['total_allowances']??0);
    $totalD = (float)($p['ssnit_employee']??0) + (float)($p['total_tax']??0) + (float)($p['total_deductions']??0);
    $net = (float)($p['net_salary'] ?? ($totalE - $totalD));
    $periodStart = $p['pay_period_start'] ?? '';
    $periodMonth = $periodStart ? date('F Y', strtotime($periodStart)) : '';
    $companyName = $p['company_name'] ?? 'Company';
    $companyAddr = $p['company_address'] ?? '';
    // Resolve company logo URL
    $logoPath = trim((string)($p['company_logo'] ?? ''));
    if ($logoPath === '') { $logoPath = 'assets/img/smartquantumhrlogo.png'; }
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $script = $_SERVER['SCRIPT_NAME'] ?? '/';
    $appDir = rtrim(dirname($script), '/\\');
    $rootDir = rtrim(dirname($appDir), '/\\');
    $rootBase = $rootDir === '' ? '/' : $rootDir . '/';
    if (preg_match('/^(?:https?:)?\/\//i', $logoPath) || preg_match('/^data:/i', $logoPath)) {
        $logoUrl = $logoPath;
    } elseif (strpos($logoPath, '/') === 0) {
        $logoUrl = $scheme . '://' . $host . $logoPath;
    } else {
        $logoUrl = $scheme . '://' . $host . $rootBase . ltrim($logoPath, '/');
    }

    $rows = '';
    for ($i=0; $i<$max; $i++) {
        $er = $earnRows[$i]; $dr = $dedRows[$i];
        $rows .= '<tr>'
            . '<td>'.htmlspecialchars((string)$er['name']).'</td>'
            . '<td style="text-align:right">'.($er['amount']===''? '' : $fmt($er['amount'])).'</td>'
            . '<td>'.htmlspecialchars((string)$dr['name']).'</td>'
            . '<td style="text-align:right">'.($dr['amount']===''? '' : $fmt($dr['amount'])).'</td>'
            . '</tr>';
    }

    $html = '<!doctype html><html><head><meta charset="utf-8" />'
      . '<style>body{font-family:Times New Roman,serif;color:#000} .sheet{width:980px;margin:20px auto;border:2px solid #000;padding:12px} .title{text-align:center} .title h1{margin:0;font-size:28px;font-weight:bold} .title h2{margin:2px 0 0 0;font-size:16px;font-weight:normal} .subtitle{text-align:center;margin:10px 0;font-size:16px;font-weight:bold} table{width:100%;border-collapse:collapse} th,td{border:1px solid #000;padding:6px 8px;font-size:14px} th{background:#f5f5f5} .no-border td{border:none} .right{text-align:right} .center{text-align:center} .info td{border:none;padding:2px 6px}</style>'
      . '<script>(function(){var url="logs.php";function send(p){try{navigator.sendBeacon(url,new Blob([JSON.stringify(p)],{type:"application/json"}))}catch(e){try{fetch(url,{method:"POST",headers:{"Content-Type":"application/json"},credentials:"same-origin",body:JSON.stringify(p)})}catch(_){}}}window.addEventListener("DOMContentLoaded",function(){try{send({level:"info",message:"payslip_html_loaded",context:{payroll_id:"'.htmlspecialchars((string)($p['id']??'')).'"}})}catch(_){}});window.addEventListener("error",function(e){try{send({level:"error",message:"payslip_html_error",context:{error:String(e.message||""),stack:String((e.error&&e.error.stack)||"")}})}catch(_){}});})();</script>'
      . '<title>Payslip</title></head><body>'
      . '<div class="sheet">'
      . '<div class="title">'
      . '<img src="'.htmlspecialchars($logoUrl).'" alt="Company Logo" style="height:50px;margin-bottom:6px;"/>'
      . '<h1>'.htmlspecialchars($companyName).'</h1><h2>'.htmlspecialchars($companyAddr).'</h2></div>'
      . '<div class="subtitle">Payslip for the period of '.htmlspecialchars($periodMonth).'</div>'
      . '<table class="info">'
      . '<tr><td>Employee Id</td><td>: '.htmlspecialchars((string)($p['employee_number']??'')).'</td><td>Name</td><td>: '.htmlspecialchars((string)($p['employee_name']??'')).'</td></tr>'
      . '<tr><td>Department</td><td>: '.htmlspecialchars((string)($p['department_name']??'')).'</td><td>Designation</td><td>: '.htmlspecialchars((string)($p['position_title']??'')).'</td></tr>'
      . '<tr><td>Date Of Birth</td><td>: '.htmlspecialchars((string)($p['date_of_birth']??'')).'</td><td>Hire Date</td><td>: '.htmlspecialchars((string)($p['hire_date']??'')).'</td></tr>'
      . '<tr><td>Pay Date</td><td>: '.htmlspecialchars((string)($p['pay_period_end']??'')).'</td><td>Employer SSF</td><td>: '.$fmt($p['ssnit_employer']??0).'</td></tr>'
      . '</table>'
      . '<table><thead><tr><th>Earnings</th><th class="right">Amount</th><th>Deductions</th><th class="right">Amount</th></tr></thead><tbody>'
      . $rows
      . '<tr><th>Total Earnings</th><th class="right">'.$fmt($totalE).'</th><th>Total Deductions</th><th class="right">'.$fmt($totalD).'</th></tr>'
      . '<tr><th colspan="2" class="center">Net Pay (Rounded)</th><th colspan="2" class="center">'.$fmt($net).'</th></tr>'
      . '</tbody></table>'
      . '<div class="center" style="margin-top:12px;font-style:italic">(All figures in Ghana Cedi)</div>'
      . '</div></body></html>';
    return $html;
}

function sendPayslipEmail(PDO $db, int $payrollId) {
    $p = loadPayslipData($db, $payrollId);
    if (!$p) return; // nothing to do
    $to = trim((string)($p['email'] ?? ''));
    if ($to === '') return;
    $html = buildPayslipHtmlServer($p);
    $period = ($p['pay_period_start'] ?? '') . ' to ' . ($p['pay_period_end'] ?? '');
    $subject = 'Payslip for ' . $period;
    $boundary = '=_Part_'.bin2hex(random_bytes(8));
    // Resolve From (per company)
    $fromName = 'SmartQuantumHR';
    $fromEmail = 'no-reply@localhost';
    try {
        $cid = (int)($p['company_id'] ?? 0);
        if ($cid) {
            $hasName = tableHasColumn($db, 'companies', 'email_from_name');
            $hasEmail = tableHasColumn($db, 'companies', 'email_from_email');
            if ($hasName || $hasEmail) {
                $st = $db->prepare('SELECT email_from_name, email_from_email FROM companies WHERE id = :id');
                $st->bindValue(':id', $cid, PDO::PARAM_INT);
                $st->execute();
                $row = $st->fetch() ?: [];
                if (!empty($row['email_from_name'])) $fromName = $row['email_from_name'];
                if (!empty($row['email_from_email'])) $fromEmail = $row['email_from_email'];
            } else {
                $file = realpath(__DIR__ . '/../storage/config') ? (__DIR__ . '/../storage/config/email-' . $cid . '.json') : (__DIR__ . '/../storage/config/email-' . $cid . '.json');
                if (is_file($file)) {
                    $cfg = json_decode(@file_get_contents($file), true) ?: [];
                    if (!empty($cfg['email_from_name'])) $fromName = $cfg['email_from_name'];
                    if (!empty($cfg['email_from_email'])) $fromEmail = $cfg['email_from_email'];
                }
            }
        }
    } catch (Throwable $e) { /* keep defaults */ }
    $headers = '';
    $headers .= "MIME-Version: 1.0\r\n";
    $headers .= "From: ".$fromName." <".$fromEmail.">\r\n";
    $headers .= "Content-Type: multipart/mixed; boundary=\"$boundary\"\r\n";
    $message = '';
    $message .= "--$boundary\r\n";
    $message .= "Content-Type: text/plain; charset=UTF-8\r\n\r\n";
    $message .= "Dear ".$p['employee_name'].",\r\nYour payslip for $period is attached.\r\n\r\nRegards,\r\nHR\r\n";
    $message .= "\r\n--$boundary\r\n";
    $filename = 'payslip-'.preg_replace('/[^0-9-]/','', (string)($p['pay_period_end'] ?? 'period')).'-'.preg_replace('/\s+/','_', strtolower((string)($p['employee_name']??'employee'))).'.html';
    $message .= "Content-Type: text/html; name=\"$filename\"\r\n";
    $message .= "Content-Transfer-Encoding: base64\r\n";
    $message .= "Content-Disposition: attachment; filename=\"$filename\"\r\n\r\n";
    $message .= chunk_split(base64_encode($html));
    $message .= "--$boundary--\r\n";
    try {
        @mail($to, $subject, $message, $headers);
        try { app_log('info', 'payslip_email_sent', ['to'=>$to, 'payroll_id'=>$payrollId]); } catch (Throwable $e) { /* ignore */ }
    } catch (Throwable $e) {
        try { app_log('error', 'payslip_email_failed', ['to'=>$to, 'payroll_id'=>$payrollId, 'error'=>$e->getMessage()]); } catch (Throwable $e2) { /* ignore */ }
    }
}

// ===== Salary Payouts (Bank/Integration/Paystack) and Reports ===== //

function payoutPayroll(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $ids = [];
    if (!empty($_GET['id'])) $ids[] = (int)$_GET['id'];
    if (isset($in['id'])) $ids[] = (int)$in['id'];
    if (isset($in['ids']) && is_array($in['ids'])) {
        foreach ($in['ids'] as $v) { if (ctype_digit(strval($v))) $ids[] = (int)$v; }
    }
    $ids = array_values(array_unique(array_filter($ids)));
    if (empty($ids)) ApiResponse::error('Payroll id(s) required');

    $channel = strtolower(trim((string)($in['channel'] ?? '')));
    $allowedChannels = ['bank_manual','integration','paystack_bank','paystack_momo'];
    if (!in_array($channel, $allowedChannels, true)) ApiResponse::error('Invalid payout channel');

    // Optional details (bank/momo etc.)
    $details = is_array($in['details'] ?? null) ? $in['details'] : [];
    $amountOverride = isset($in['amount']) ? (float)$in['amount'] : null; // if set, overrides net
    $refOverride = isset($in['reference']) ? trim((string)$in['reference']) : '';

    $results = [];
    foreach ($ids as $pid) {
        try {
            // Load payroll ensuring company scope and approved
            $q = $db->prepare("SELECT p.*, e.company_id, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.email, e.phone,
                                      e.bank_code, e.bank_account, e.momo_provider, e.momo_number
                               FROM payrolls p JOIN employees e ON p.employee_id = e.id 
                               WHERE p.id = :id AND e.company_id = :cid AND p.status IN ('approved','paid')");
            $q->execute([':id'=>$pid, ':cid'=>$user['company_id']]);
            if ($q->rowCount() === 0) { $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Not found or not approved']; continue; }
            $p = $q->fetch();
            $currentStatus = strtolower((string)$p['status']);
            if ($currentStatus !== 'approved' && $currentStatus !== 'paid') {
                $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Not approved'];
                continue;
            }
            $net = (float)($p['net_salary'] ?? 0);
            $payAmount = $amountOverride !== null ? $amountOverride : $net;
            $periodTxt = trim(($p['pay_period_start'] ?? '').' to '.($p['pay_period_end'] ?? ''));

            $newPayrollStatus = $currentStatus; // may become 'paid'
            $payoutStatus = 'queued';
            $payoutRef = $refOverride !== '' ? $refOverride : (strtoupper(substr($channel,0,3)) . '-' . date('YmdHis') . '-' . $pid);
            $meta = [ 'channel'=>$channel, 'requested_by'=>$user['id'], 'details'=>$details ];

            if ($channel === 'bank_manual' || $channel === 'integration') {
                $newPayrollStatus = 'paid';
                $payoutStatus = 'completed';
                $meta['note'] = $in['note'] ?? null;
            } else {
                // Paystack channels
                $cfg = paystack_load_config((int)$user['company_id']);
                if (empty($cfg['secret_key'])) {
                    $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Paystack not configured'];
                    continue;
                }
                // Create recipient
                $recipientName = trim((string)($details['recipient_name'] ?? ($p['employee_name'] ?? 'Employee')));
                $currency = $cfg['currency'] ?? 'GHS';
                if ($channel === 'paystack_momo') {
                    $acc = trim((string)($details['account_number'] ?? $p['momo_number'] ?? $p['phone'] ?? ''));
                    $provider = trim((string)($details['momo_provider'] ?? ($p['momo_provider'] ?? '')));
                    if ($acc === '' || $provider === '') { $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Missing MoMo account_number or provider']; continue; }
                    $recPayload = [
                        'type' => 'mobile_money',
                        'name' => $recipientName,
                        'account_number' => $acc,
                        'bank_code' => $provider,
                        'currency' => $currency
                    ];
                    $r1 = paystack_create_recipient($cfg, $recPayload);
                } else { // paystack_bank
                    $acc = trim((string)($details['account_number'] ?? ($p['bank_account'] ?? '')));
                    $bankCode = trim((string)($details['bank_code'] ?? ($p['bank_code'] ?? '')));
                    if ($acc === '' || $bankCode === '') { $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Missing bank account_number or bank_code']; continue; }
                    $recPayload = [
                        'type' => 'bank_account',
                        'name' => $recipientName,
                        'account_number' => $acc,
                        'bank_code' => $bankCode,
                        'currency' => $currency
                    ];
                    $r1 = paystack_create_recipient($cfg, $recPayload);
                }
                $meta['recipient_request'] = $recPayload;
                $meta['recipient_response'] = $r1;
                if (!$r1['ok'] || empty($r1['json']['status']) || empty($r1['json']['data']['recipient_code'])) {
                    $results[] = ['id'=>$pid,'ok'=>false,'message'=>'Failed to create recipient'];
                    // Persist failure details
                    $u = $db->prepare("UPDATE payrolls SET payout_channel = :ch, payout_status = 'failed', payout_reference = :ref, payout_meta = :meta, updated_at = NOW() WHERE id = :id");
                    $u->execute([':ch'=>$channel, ':ref'=>$payoutRef, ':meta'=>json_encode($meta), ':id'=>$pid]);
                    continue;
                }
                $recipientCode = $r1['json']['data']['recipient_code'];
                $amountKobo = (int)round($payAmount * 100); // Paystack uses lowest currency unit
                $reason = 'Salary ' . $periodTxt;
                $trPayload = [
                    'source' => 'balance',
                    'amount' => $amountKobo,
                    'recipient' => $recipientCode,
                    'reason' => $reason,
                    'currency' => $currency
                ];
                $r2 = paystack_initiate_transfer($cfg, $trPayload);
                $meta['transfer_request'] = ['payload'=>$trPayload];
                $meta['transfer_response'] = $r2;
                if ($r2['ok'] && !empty($r2['json']['data']['transfer_code'])) {
                    $payoutRef = (string)$r2['json']['data']['transfer_code'];
                    $ps = strtolower((string)($r2['json']['data']['status'] ?? 'pending'));
                    $payoutStatus = in_array($ps, ['success','completed']) ? 'completed' : ($ps ?: 'queued');
                    if ($payoutStatus === 'completed') { $newPayrollStatus = 'paid'; }
                } else {
                    $payoutStatus = 'failed';
                }
            }

            // Persist
            $sql = "UPDATE payrolls SET payout_channel = :ch, payout_status = :ps, payout_reference = :ref, payout_meta = :meta, 
                    paid_amount = :amt, paid_by = :by, paid_at = CASE WHEN :setpaid = 1 THEN NOW() ELSE paid_at END, 
                    status = CASE WHEN :setstatus = 1 THEN 'paid' ELSE status END, updated_at = NOW() WHERE id = :id";
            $stmt = $db->prepare($sql);
            $setPaid = ($newPayrollStatus === 'paid') ? 1 : 0;
            $stmt->bindValue(':ch', $channel);
            $stmt->bindValue(':ps', $payoutStatus);
            $stmt->bindValue(':ref', $payoutRef);
            $stmt->bindValue(':meta', json_encode($meta));
            $stmt->bindValue(':amt', $payAmount);
            $stmt->bindValue(':by', $setPaid ? $user['id'] : null);
            $stmt->bindValue(':setpaid', $setPaid, PDO::PARAM_INT);
            $stmt->bindValue(':setstatus', $setPaid, PDO::PARAM_INT);
            $stmt->bindValue(':id', $pid, PDO::PARAM_INT);
            $stmt->execute();

            $results[] = ['id'=>$pid,'ok'=>true,'payout_status'=>$payoutStatus,'payroll_status'=>$newPayrollStatus,'reference'=>$payoutRef];
        } catch (Throwable $e) {
            $results[] = ['id'=>$pid,'ok'=>false,'message'=>$e->getMessage()];
        }
    }
    ApiResponse::success(['results'=>$results], 'Payout processed');
}

function reportPayouts(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $month = isset($_GET['period_month']) ? (int)$_GET['period_month'] : null;
    $year = isset($_GET['period_year']) ? (int)$_GET['period_year'] : null;
    $format = strtolower($_GET['format'] ?? '') === 'csv' ? 'csv' : 'json';

    $sql = "SELECT p.id, e.employee_number, CONCAT(e.first_name,' ',e.last_name) AS employee_name,
                   p.pay_period_start, p.pay_period_end, p.net_salary, p.status,
                   p.payout_channel, p.payout_status, p.payout_reference, p.paid_amount, p.paid_at
            FROM payrolls p JOIN employees e ON p.employee_id = e.id
            WHERE p.company_id = :cid AND p.status IN ('approved','paid')";
    $params = [':cid'=>$user['company_id']];
    if ($month) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $month; }
    if ($year)  { $sql .= " AND YEAR(p.pay_period_start) = :y";  $params[':y'] = $year; }
    $sql .= " ORDER BY p.pay_period_start ASC, employee_name ASC";

    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll();

    if ($format === 'csv') {
        $csvRows = [];
        foreach ($rows as $r) {
            $csvRows[] = [
                $r['employee_number'], $r['employee_name'], $r['pay_period_start'], $r['pay_period_end'],
                number_format((float)$r['net_salary'],2,'.',''),
                strtoupper($r['status']), strtoupper((string)($r['payout_channel'] ?? '')),
                strtoupper((string)($r['payout_status'] ?? '')), $r['payout_reference'] ?? '',
                number_format((float)($r['paid_amount'] ?? 0),2,'.',''), $r['paid_at'] ?? ''
            ];
        }
        csvOutput('payouts.csv', ['Employee #','Employee Name','Period Start','Period End','Net','Payroll Status','Payout Channel','Payout Status','Reference','Paid Amount','Paid At'], $csvRows);
    }
    ApiResponse::success($rows);
}

function sendPayslipAction(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $in = json_decode(file_get_contents('php://input'), true) ?? [];
    $ids = [];
    if (isset($in['id'])) $ids[] = (int)$in['id'];
    if (isset($in['ids']) && is_array($in['ids'])) {
        foreach ($in['ids'] as $v) { if (ctype_digit(strval($v))) $ids[] = (int)$v; }
    }
    $ids = array_values(array_unique(array_filter($ids)));
    if (empty($ids)) ApiResponse::error('Payroll id(s) required');
    $sent = 0; $errors = [];
    foreach ($ids as $pid) {
        try { sendPayslipEmail($db, (int)$pid); $sent++; } catch (Throwable $e) { $errors[] = ['id'=>$pid,'error'=>$e->getMessage()]; }
    }
    ApiResponse::success(['sent'=>$sent,'errors'=>$errors], 'Payslip emails triggered');
}

// HR/Admin preview of any payroll (including 'calculated') with allowances/deductions detail
function getPayrollPreview(PDO $db, $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) { ApiResponse::error('Payroll ID required'); }
    try {
        // Fetch payroll scoped to company, any status
        $q = $db->prepare("SELECT 
                    p.*,
                    CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                    e.employee_number,
                    e.email
                  FROM payrolls p
                  JOIN employees e ON p.employee_id = e.id
                  WHERE p.id = :id AND p.company_id = :cid");
        $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
        if ($q->rowCount() === 0) { ApiResponse::notFound('Payroll not found'); }
        $payroll = $q->fetch();
        // Attach detailed components
        $basic = (float)($payroll['basic_salary'] ?? 0);
        $gross = (float)($payroll['gross_salary'] ?? ($basic + (float)($payroll['total_allowances'] ?? 0)));
        $payroll['allowances'] = getPayrollAllowances($db, $payroll['employee_id'], $payroll['pay_period_end'], $basic, $gross);
        $payroll['deductions'] = getPayrollDeductions($db, $payroll['employee_id'], $payroll['pay_period_end'], $basic, $gross);
        ApiResponse::success($payroll);
    } catch (Throwable $e) {
        ApiResponse::error('Failed to load preview: ' . $e->getMessage(), 500);
    }
}

// Bulk approval for all selected calculated payrolls (HR-Head/Admin only)
function approvePayrollBulk(PDO $db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $ids = isset($input['ids']) && is_array($input['ids']) ? array_values(array_filter($input['ids'], fn($v)=>ctype_digit(strval($v)))) : [];
    $periodMonth = isset($input['period_month']) ? (int)$input['period_month'] : null;
    $periodYear  = isset($input['period_year']) ? (int)$input['period_year'] : null;

    try {
        $db->beginTransaction();
        // Build candidate list
        $candidateIds = [];
        if (!empty($ids)) {
            // Scope to company and calculated
            $in = implode(',', array_fill(0, count($ids), '?'));
            $s = $db->prepare("SELECT p.id FROM payrolls p JOIN employees e ON p.employee_id = e.id WHERE p.id IN ($in) AND e.company_id = ? AND p.status = 'calculated'");
            foreach ($ids as $i=>$v) { $s->bindValue($i+1, (int)$v, PDO::PARAM_INT); }
            $s->bindValue(count($ids)+1, $user['company_id'], PDO::PARAM_INT);
            $s->execute();
            $candidateIds = array_map(fn($r)=> (int)$r['id'], $s->fetchAll());
        } else {
            // Use month/year filters to find calculated records in company
            $sql = "SELECT p.id FROM payrolls p WHERE p.company_id = :cid AND p.status = 'calculated'";
            $params = [':cid'=>$user['company_id']];
            if ($periodMonth) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $periodMonth; }
            if ($periodYear)  { $sql .= " AND YEAR(p.pay_period_start)  = :y"; $params[':y'] = $periodYear; }
            $s = $db->prepare($sql);
            foreach ($params as $k=>$v) { $s->bindValue($k, $v); }
            $s->execute();
            $candidateIds = array_map(fn($r)=> (int)$r['id'], $s->fetchAll());
        }

        if (empty($candidateIds)) {
            $db->commit();
            ApiResponse::success(['approved_count'=>0, 'ids'=>[]], 'No payrolls to approve');
        }

        // Approve in batch
        $in2 = implode(',', array_fill(0, count($candidateIds), '?'));
        $u = $db->prepare("UPDATE payrolls SET status = 'approved', approved_by = ?, approved_at = NOW(), updated_at = NOW() WHERE id IN ($in2)");
        $idx = 1;
        $u->bindValue($idx++, $user['id'], PDO::PARAM_INT);
        foreach ($candidateIds as $pid) { $u->bindValue($idx++, $pid, PDO::PARAM_INT); }
        $u->execute();

        // Notifications per approved payroll and email payslips (best-effort)
        foreach ($candidateIds as $pid) {
            try { createPayrollNotification($db, $pid, 'payroll_approved'); } catch (Throwable $e) { /* ignore */ }
            try { sendPayslipEmail($db, (int)$pid); } catch (Throwable $e) { /* ignore */ }
        }

        $db->commit();
        ApiResponse::success(['approved_count'=>count($candidateIds), 'ids'=>$candidateIds], 'Approved');
    } catch (Throwable $e) {
        if ($db->inTransaction()) $db->rollBack();
        ApiResponse::error('Failed to approve all: ' . $e->getMessage(), 500);
    }
}

// Try to resolve and persist an employee's grade_id using textual hints (current_grade/starting_grade) if those columns exist
function resolveEmployeeGradeId(PDO $db, $employeeId, $companyId) {
    $hint = null;
    try {
        if (tableHasColumn($db, 'employees', 'current_grade')) {
            $q = $db->prepare("SELECT current_grade FROM employees WHERE id = :id");
            $q->execute([':id' => $employeeId]);
            $cg = $q->fetchColumn();
            if ($cg !== false && $cg !== null && trim($cg) !== '') { $hint = $cg; }
        }
        if ($hint === null && tableHasColumn($db, 'employees', 'starting_grade')) {
            $q = $db->prepare("SELECT starting_grade FROM employees WHERE id = :id");
            $q->execute([':id' => $employeeId]);
            $sg = $q->fetchColumn();
            if ($sg !== false && $sg !== null && trim($sg) !== '') { $hint = $sg; }
        }
        if ($hint !== null) {
            $mg = $db->prepare("SELECT id FROM grades WHERE company_id = :cid AND (LOWER(code) = LOWER(:g) OR LOWER(name) = LOWER(:g)) LIMIT 1");
            $mg->execute([':cid' => $companyId, ':g' => $hint]);
            $gid = $mg->fetchColumn();
            if ($gid) {
                $gid = (int)$gid;
                try {
                    $up = $db->prepare("UPDATE employees SET grade_id = :gid, updated_at = NOW() WHERE id = :eid");
                    $up->execute([':gid' => $gid, ':eid' => $employeeId]);
                } catch (Throwable $e) { /* ignore persist errors */ }
                return $gid;
            }
        }
    } catch (Throwable $e) {
        // ignore
    }
    return null;
}

switch ($method) {
    case 'GET':
        if ($action === 'payslip') {
            getPayslip($db, $id);
        } elseif ($action === 'payslip_html') {
            renderPayslipHtml($db, $id);
        } elseif ($action === 'preview') {
            getPayrollPreview($db, $id);
        } elseif ($action === 'my_payslips') {
            // Always return only the current user's payslips regardless of role
            $u = getCurrentUser();
            getEmployeePayrolls($db, $u);
        } elseif ($action === 'calculate') {
            calculatePayroll($db);
        } elseif ($action === 'setup_lists') {
            getPayrollSetupLists($db);
        } elseif ($action === 'report_paye') {
            reportPAYE($db);
        } elseif ($action === 'report_ssnit') {
            reportSSNIT($db);
        } elseif ($action === 'report_summary') {
            reportSummary($db);
        } elseif ($action === 'report_payslips') {
            reportPayslips($db);
        } elseif ($action === 'report_payouts') {
            reportPayouts($db);
        } elseif ($id) {
            getPayroll($db, $id);
        } else {
            getPayrolls($db);
        }
        break;
    
    case 'POST':
        if ($action === 'process') {
            processPayroll($db);
        } elseif ($action === 'approve') {
            approvePayroll($db, $id);
        } elseif ($action === 'approve_all') {
            approvePayrollBulk($db);
        } elseif ($action === 'save_allowance_type') {
            saveAllowanceType($db);
        } elseif ($action === 'save_deduction_type') {
            saveDeductionType($db);
        } elseif ($action === 'delete_allowance_type') {
            deleteAllowanceType($db, $id);
        } elseif ($action === 'delete_deduction_type') {
            deleteDeductionType($db, $id);
        } elseif ($action === 'payout') {
            payoutPayroll($db);
        } elseif ($action === 'send_payslip') {
            sendPayslipAction($db);
        } else {
            createPayroll($db);
        }
        break;
    
    case 'PUT':
        updatePayroll($db, $id);
        break;
    
    case 'DELETE':
        deletePayroll($db, $id);
        break;
    
    default:
        ApiResponse::error('Method not allowed', 405);
}

function getPayrollSetupLists($db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    try {
        $a = $db->prepare("SELECT id, name, code, taxable, taxable_rate FROM allowance_types WHERE company_id = :cid ORDER BY name");
        $a->bindValue(':cid', $user['company_id']);
        $a->execute();
        $allowances = $a->fetchAll();

        $d = $db->prepare("SELECT id, name, code FROM deduction_types WHERE company_id = :cid ORDER BY name");
        $d->bindValue(':cid', $user['company_id']);
        $d->execute();
        $deductions = $d->fetchAll();

        ApiResponse::success(['allowance_types' => $allowances, 'deduction_types' => $deductions]);
    } catch (Throwable $e) {
        ApiResponse::error('Failed to load setup lists: ' . $e->getMessage(), 500);
    }
}

// Notify an employee when a payroll is skipped because it already exists for the given period
function notifyPayrollDuplicate(PDO $db, $employeeId, $periodStart, $periodEnd) {
    try {
        $q = $db->prepare("SELECT user_id FROM employees WHERE id = :eid");
        $q->bindValue(':eid', $employeeId, PDO::PARAM_INT);
        $q->execute();
        $uid = $q->fetchColumn();
        if ($uid) {
            $notif = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, :type, :title, :content, NOW())");
            $notif->bindValue(':uid', $uid, PDO::PARAM_INT);
            $notif->bindValue(':type', 'payroll_duplicate');
            $notif->bindValue(':title', 'Payroll Skipped (Duplicate)');
            $notif->bindValue(':content', "Payroll for period $periodStart to $periodEnd was not processed because a record already exists.");
            $notif->execute();
        }
    } catch (Throwable $e) {
        try { app_log('warning', 'notify_duplicate_failed', ['employee_id'=>$employeeId, 'error'=>$e->getMessage()]); } catch (Throwable $e2) { /* ignore */ }
    }
}

// ===== Reporting (PAYE, SSNIT, Summary, Payslips) ===== //

function mustBeHr($user) {
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
}

function csvOutput($filename, $headers, $rows) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    $out = fopen('php://output', 'w');
    fputcsv($out, $headers);
    foreach ($rows as $r) { fputcsv($out, $r); }
    fclose($out);
    exit;
}

function reportPAYE(PDO $db) {
    $user = getCurrentUser();
    mustBeHr($user);
    $month = isset($_GET['period_month']) ? (int)$_GET['period_month'] : null;
    $year = isset($_GET['period_year']) ? (int)$_GET['period_year'] : null;
    $format = strtolower($_GET['format'] ?? '') === 'csv' ? 'csv' : 'json';

    $sql = "SELECT p.id, p.employee_id, p.pay_period_start, p.pay_period_end, p.basic_salary, p.taxable_allowances, p.exempt_allowances,
                   p.chargeable_income, p.total_tax AS paye, p.status,
                   e.employee_number, CONCAT(e.first_name, ' ', e.last_name) AS employee_name
            FROM payrolls p
            JOIN employees e ON p.employee_id = e.id
            WHERE p.company_id = :cid AND p.status IN ('approved','paid')";
    $params = [':cid'=>$user['company_id']];
    if ($month) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $month; }
    if ($year)  { $sql .= " AND YEAR(p.pay_period_start) = :y";  $params[':y'] = $year; }
    $sql .= " ORDER BY p.pay_period_start ASC, employee_name ASC";

    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll();

    if ($format === 'csv') {
        $csvRows = [];
        foreach ($rows as $r) {
            $csvRows[] = [
                $r['employee_number'], $r['employee_name'], $r['pay_period_start'], $r['pay_period_end'],
                number_format((float)$r['basic_salary'],2,'.',''),
                number_format((float)$r['taxable_allowances'],2,'.',''),
                number_format((float)$r['exempt_allowances'],2,'.',''),
                number_format((float)$r['chargeable_income'],2,'.',''),
                number_format((float)$r['paye'],2,'.',''),
                strtoupper($r['status'])
            ];
        }
        csvOutput('paye_schedule.csv', ['Employee #','Employee Name','Period Start','Period End','Basic','Taxable Allowances','Exempt Allowances','Chargeable Income','PAYE','Status'], $csvRows);
    }
    ApiResponse::success($rows);
}

function reportSSNIT(PDO $db) {
    $user = getCurrentUser();
    mustBeHr($user);
    $month = isset($_GET['period_month']) ? (int)$_GET['period_month'] : null;
    $year = isset($_GET['period_year']) ? (int)$_GET['period_year'] : null;
    $format = strtolower($_GET['format'] ?? '') === 'csv' ? 'csv' : 'json';

    $sql = "SELECT p.id, p.employee_id, p.pay_period_start, p.pay_period_end, p.gross_salary,
                   p.ssnit_employee, p.ssnit_employer, p.status,
                   e.employee_number, CONCAT(e.first_name, ' ', e.last_name) AS employee_name
            FROM payrolls p
            JOIN employees e ON p.employee_id = e.id
            WHERE p.company_id = :cid AND p.status IN ('approved','paid')";
    $params = [':cid'=>$user['company_id']];
    if ($month) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $month; }
    if ($year)  { $sql .= " AND YEAR(p.pay_period_start) = :y";  $params[':y'] = $year; }
    $sql .= " ORDER BY p.pay_period_start ASC, employee_name ASC";
    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll();

    if ($format === 'csv') {
        $csvRows = [];
        foreach ($rows as $r) {
            $csvRows[] = [
                $r['employee_number'], $r['employee_name'], $r['pay_period_start'], $r['pay_period_end'],
                number_format((float)$r['gross_salary'],2,'.',''),
                number_format((float)$r['ssnit_employee'],2,'.',''),
                number_format((float)$r['ssnit_employer'],2,'.',''),
                strtoupper($r['status'])
            ];
        }
        csvOutput('ssnit_contribution_report.csv', ['Employee #','Employee Name','Period Start','Period End','Gross','SSNIT (Employee 5.5%)','SSNIT (Employer 13%)','Status'], $csvRows);
    }
    ApiResponse::success($rows);
}

function reportSummary(PDO $db) {
    $user = getCurrentUser();
    mustBeHr($user);
    $month = isset($_GET['period_month']) ? (int)$_GET['period_month'] : null;
    $year = isset($_GET['period_year']) ? (int)$_GET['period_year'] : null;
    $format = strtolower($_GET['format'] ?? '') === 'csv' ? 'csv' : 'json';

    $sql = "SELECT p.*,
                   e.employee_number, CONCAT(e.first_name, ' ', e.last_name) AS employee_name
            FROM payrolls p
            JOIN employees e ON p.employee_id = e.id
            WHERE p.company_id = :cid AND p.status IN ('approved','paid')";
    $params = [':cid'=>$user['company_id']];
    if ($month) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $month; }
    if ($year)  { $sql .= " AND YEAR(p.pay_period_start) = :y";  $params[':y'] = $year; }
    $sql .= " ORDER BY p.pay_period_start ASC, employee_name ASC";
    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll();

    $totals = [
        'total_gross' => 0.0,
        'ssnit_employee' => 0.0,
        'ssnit_employer' => 0.0,
        'paye' => 0.0,
        'other_deductions' => 0.0,
        'total_net' => 0.0,
        'employer_cost' => 0.0
    ];
    foreach ($rows as $r) {
        $totals['total_gross'] += (float)$r['gross_salary'];
        $totals['ssnit_employee'] += (float)$r['ssnit_employee'];
        $totals['ssnit_employer'] += (float)$r['ssnit_employer'];
        $totals['paye'] += (float)$r['total_tax'];
        $totals['other_deductions'] += (float)$r['total_deductions'];
        $totals['total_net'] += (float)$r['net_salary'];
        $totals['employer_cost'] += ((float)$r['gross_salary'] + (float)$r['ssnit_employer']);
    }

    if ($format === 'csv') {
        $csvRows = [];
        foreach ($rows as $r) {
            $csvRows[] = [
                $r['employee_number'], $r['employee_name'], $r['pay_period_start'], $r['pay_period_end'],
                number_format((float)$r['gross_salary'],2,'.',''),
                number_format((float)$r['ssnit_employee'],2,'.',''),
                number_format((float)$r['ssnit_employer'],2,'.',''),
                number_format((float)$r['total_tax'],2,'.',''),
                number_format((float)$r['total_deductions'],2,'.',''),
                number_format((float)$r['net_salary'],2,'.',''),
                number_format(((float)$r['gross_salary'] + (float)$r['ssnit_employer']),2,'.','')
            ];
        }
        // Note: Totals row omitted for CSV to keep a clean per-row export; aggregate in Excel if needed.
        csvOutput('payroll_summary.csv', ['Employee #','Employee Name','Period Start','Period End','Gross','SSNIT Emp 5.5%','SSNIT Empr 13%','PAYE','Other Deductions','Net','Employer Cost'], $csvRows);
    }

    ApiResponse::success(['rows'=>$rows, 'totals'=>$totals]);
}

function reportPayslips(PDO $db) {
    $user = getCurrentUser();
    mustBeHr($user);
    $month = isset($_GET['period_month']) ? (int)$_GET['period_month'] : null;
    $year = isset($_GET['period_year']) ? (int)$_GET['period_year'] : null;
    $format = strtolower($_GET['format'] ?? '') === 'csv' ? 'csv' : 'json';

    $sql = "SELECT p.*, e.employee_number, CONCAT(e.first_name,' ',e.last_name) AS employee_name
            FROM payrolls p
            JOIN employees e ON p.employee_id = e.id
            WHERE p.company_id = :cid AND p.status IN ('approved','paid')";
    $params = [':cid'=>$user['company_id']];
    if ($month) { $sql .= " AND MONTH(p.pay_period_start) = :m"; $params[':m'] = $month; }
    if ($year)  { $sql .= " AND YEAR(p.pay_period_start) = :y";  $params[':y'] = $year; }
    $sql .= " ORDER BY p.pay_period_start ASC, employee_name ASC";
    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    $rows = $st->fetchAll();

    if ($format === 'csv') {
        $csvRows = [];
        foreach ($rows as $r) {
            $csvRows[] = [
                $r['employee_number'], $r['employee_name'], $r['pay_period_start'], $r['pay_period_end'],
                number_format((float)$r['basic_salary'],2,'.',''),
                number_format((float)$r['total_allowances'],2,'.',''),
                number_format((float)$r['total_deductions'],2,'.',''),
                number_format((float)$r['total_tax'],2,'.',''),
                number_format((float)$r['net_salary'],2,'.','')
            ];
        }
        csvOutput('payslips.csv', ['Employee #','Employee Name','Period Start','Period End','Basic','Allowances','Other Deductions','PAYE','Net'], $csvRows);
    }

    ApiResponse::success($rows);
}

function saveAllowanceType($db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($input['id']) ? (int)$input['id'] : 0;
    $name = trim($input['name'] ?? '');
    $code = trim($input['code'] ?? '');
    $taxable = isset($input['taxable']) ? (int)!!$input['taxable'] : 1;
    $taxable_rate = isset($input['taxable_rate']) && $input['taxable_rate'] !== '' ? (float)$input['taxable_rate'] : null; // percent 0-100
    if ($name === '' || $code === '') ApiResponse::error('Name and Code are required');
    try {
        if ($id > 0) {
            $q = $db->prepare("UPDATE allowance_types SET name = :n, code = :c, taxable = :t, taxable_rate = :tr WHERE id = :id AND company_id = :cid");
            $q->execute([':n'=>$name, ':c'=>$code, ':t'=>$taxable, ':tr'=>$taxable_rate, ':id'=>$id, ':cid'=>$user['company_id']]);
        } else {
            $q = $db->prepare("INSERT INTO allowance_types (company_id, name, code, taxable, taxable_rate) VALUES (:cid, :n, :c, :t, :tr)");
            $q->execute([':cid'=>$user['company_id'], ':n'=>$name, ':c'=>$code, ':t'=>$taxable, ':tr'=>$taxable_rate]);
        }
        ApiResponse::success(null, 'Saved');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to save allowance type: '.$e->getMessage(), 500);
    }
}

// Ghana PAYE calculator (monthly bands)
function ghana_paye_tax($chargeableIncome) {
    $remaining = max(0.0, (float)$chargeableIncome);
    $tax = 0.0;
    $bands = [
        // [band_amount, rate]
        [490.00, 0.00],       // First 490 tax free
        [110.00, 0.05],       // Next 110 at 5%
        [130.00, 0.10],       // Next 130 at 10%
        [3166.67, 0.175],     // Next 3,166.67 at 17.5%
        [16000.00, 0.25],     // Next 16,000 at 25%
        [30520.00, 0.30],     // Next 30,520 at 30%
        // Remainder taxed at 35%
    ];
    foreach ($bands as [$limit, $rate]) {
        if ($remaining <= 0) break;
        $slice = min($remaining, $limit);
        $tax += $slice * $rate;
        $remaining -= $slice;
    }
    if ($remaining > 0) {
        $tax += $remaining * 0.35;
    }
    return round($tax, 2);
}

// Compute allowance breakdown combining employee and grade defaults
function computeAllowanceBreakdown(PDO $db, int $employeeId, ?int $gradeId, string $payPeriodEnd, float $basicSalary): array {
    // Fetch all applicable allowances (employee active + grade defaults) with calc params
    $items = [];
    try {
        $sqlEmp = "SELECT at.taxable, at.taxable_rate, ea.amount, COALESCE(ea.calc_mode,'fixed') AS calc_mode, COALESCE(ea.rate,0) AS rate
                   FROM employee_allowances ea
                   LEFT JOIN allowance_types at ON ea.allowance_type_id = at.id
                   WHERE ea.employee_id = :eid AND ea.status = 'active' AND (ea.end_date IS NULL OR ea.end_date >= :pend)";
        $st = $db->prepare($sqlEmp);
        $st->execute([':eid'=>$employeeId, ':pend'=>$payPeriodEnd]);
        foreach ($st->fetchAll() as $r) { $items[] = $r; }
        if ($gradeId) {
            $sqlGr = "SELECT at.taxable, at.taxable_rate, ga.amount, COALESCE(ga.calc_mode,'fixed') AS calc_mode, COALESCE(ga.rate,0) AS rate
                      FROM grade_allowances ga
                      LEFT JOIN allowance_types at ON ga.allowance_type_id = at.id
                      WHERE ga.grade_id = :gid";
            $gs = $db->prepare($sqlGr);
            $gs->execute([':gid'=>$gradeId]);
            foreach ($gs->fetchAll() as $r) { $items[] = $r; }
        }
    } catch (Throwable $e) { /* fall through with empty */ }

    $fixed = 0.0; $pctBasic = 0.0; $pctGrossRates = 0.0;
    foreach ($items as $it) {
        $mode = strtolower((string)$it['calc_mode']);
        $rate = (float)($it['rate'] ?? 0);
        if ($mode === 'percent_basic') { $pctBasic += ($basicSalary * $rate/100.0); }
        elseif ($mode === 'percent_gross') { $pctGrossRates += ($rate/100.0); }
        else { $fixed += (float)($it['amount'] ?? 0); }
    }
    // Pre-gross excludes % of gross items
    $preGross = $basicSalary + $fixed + $pctBasic;
    $R = max(0.0, min(0.95, $pctGrossRates)); // clamp to avoid division by zero
    $gross = $R > 0 ? ($preGross / (1.0 - $R)) : $preGross;

    $total = $fixed + $pctBasic; // start sum; add %gross per-item below
    $taxable = 0.0; $exempt = 0.0;
    foreach ($items as $it) {
        $mode = strtolower((string)$it['calc_mode']);
        $rate = (float)($it['rate'] ?? 0);
        $amt = 0.0;
        if ($mode === 'percent_basic') { $amt = $basicSalary * $rate/100.0; }
        elseif ($mode === 'percent_gross') { $amt = $gross * $rate/100.0; }
        else { $amt = (float)($it['amount'] ?? 0); }
        $total += ($mode==='percent_gross') ? $amt : 0.0; // others already counted
        $isTaxable = (int)($it['taxable'] ?? 1) === 1;
        if ($isTaxable) {
            $tr = (float)($it['taxable_rate'] ?? 100);
            $taxable += $amt * ($tr/100.0);
            $exempt  += $amt - ($amt * ($tr/100.0));
        } else {
            $exempt += $amt;
        }
    }
    return ['total'=>round($total,2), 'taxable'=>round($taxable,2), 'exempt'=>round($exempt,2), 'gross'=>round($gross,2)];
}

// ===== Loans helpers (used for deductions and auto-servicing) =====
function ensureLoanTables(PDO $db){
    try {
        $db->exec("CREATE TABLE IF NOT EXISTS employee_loans (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            employee_id INT NOT NULL,
            principal DECIMAL(12,2) NOT NULL,
            interest_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
            interest_type ENUM('flat','reducing') NOT NULL DEFAULT 'flat',
            term_months INT NOT NULL,
            start_date DATE NOT NULL,
            total_payable DECIMAL(12,2) NULL,
            monthly_installment DECIMAL(12,2) NULL,
            balance_outstanding DECIMAL(12,2) NULL,
            status ENUM('pending','approved','active','completed','rejected','cancelled') NOT NULL DEFAULT 'pending',
            approved_by INT NULL,
            approved_at DATETIME NULL,
            created_by INT NULL,
            created_at DATETIME NULL,
            updated_at DATETIME NULL,
            INDEX idx_company (company_id), INDEX idx_employee (employee_id), INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
        $db->exec("CREATE TABLE IF NOT EXISTS loan_schedules (
            id INT AUTO_INCREMENT PRIMARY KEY,
            loan_id INT NOT NULL,
            due_date DATE NOT NULL,
            amount_due DECIMAL(12,2) NOT NULL,
            amount_paid DECIMAL(12,2) NULL,
            paid_at DATETIME NULL,
            payroll_id INT NULL,
            status ENUM('due','partial','paid','skipped') NOT NULL DEFAULT 'due',
            INDEX idx_loan (loan_id), INDEX idx_due (due_date), INDEX idx_status (status)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    } catch (Throwable $e) { /* ignore */ }
}

function loanMonthlyDue(PDO $db, int $employeeId, string $payPeriodEnd): float {
    ensureLoanTables($db);
    $monthStart = date('Y-m-01', strtotime($payPeriodEnd));
    $monthEnd   = date('Y-m-t', strtotime($payPeriodEnd));
    $sql = "SELECT COALESCE(SUM(ls.amount_due - COALESCE(ls.amount_paid,0)),0) AS due
            FROM loan_schedules ls
            JOIN employee_loans l ON l.id = ls.loan_id
            WHERE l.employee_id = :eid
              AND l.status IN ('approved','active')
              AND ls.status IN ('due','partial')
              AND ls.due_date BETWEEN :s AND :e";
    try { $st = $db->prepare($sql); $st->execute([':eid'=>$employeeId, ':s'=>$monthStart, ':e'=>$monthEnd]); return (float)$st->fetchColumn(); } catch (Throwable $e){ return 0.0; }
}

function refreshLoanBalance(PDO $db, int $loanId){
    try {
        $sum = $db->prepare('SELECT l.total_payable, COALESCE(SUM(ls.amount_paid),0) AS paid FROM employee_loans l LEFT JOIN loan_schedules ls ON ls.loan_id = l.id WHERE l.id = :id');
        $sum->execute([':id'=>$loanId]);
        $row = $sum->fetch(); if (!$row) return;
        $bal = max(0.0, (float)$row['total_payable'] - (float)$row['paid']);
        $status = $bal <= 0.009 ? 'completed' : 'approved';
        $u = $db->prepare('UPDATE employee_loans SET balance_outstanding = :bal, status = :st, updated_at = NOW() WHERE id = :id');
        $u->execute([':bal'=>$bal, ':st'=>$status, ':id'=>$loanId]);
    } catch (Throwable $e) { /* ignore */ }
}

function computeDeductionsTotal(PDO $db, int $employeeId, ?int $gradeId, string $payPeriodEnd, float $basicSalary, float $gross): float {
    $total = 0.0;
    try {
        $sqlEmp = "SELECT ed.amount, COALESCE(ed.calc_mode,'fixed') AS calc_mode, COALESCE(ed.rate,0) AS rate
                   FROM employee_deductions ed
                   WHERE ed.employee_id = :eid AND ed.status = 'active' AND (ed.end_date IS NULL OR ed.end_date >= :pend)";
        $st = $db->prepare($sqlEmp);
        $st->execute([':eid'=>$employeeId, ':pend'=>$payPeriodEnd]);
        foreach ($st->fetchAll() as $r) {
            $mode = strtolower((string)$r['calc_mode']); $rate = (float)($r['rate'] ?? 0);
            if ($mode === 'percent_basic') $total += $basicSalary * $rate/100.0;
            elseif ($mode === 'percent_gross') $total += $gross * $rate/100.0;
            else $total += (float)($r['amount'] ?? 0);
        }
        if ($gradeId) {
            $sqlGr = "SELECT gd.amount, COALESCE(gd.calc_mode,'fixed') AS calc_mode, COALESCE(gd.rate,0) AS rate
                      FROM grade_deductions gd WHERE gd.grade_id = :gid";
            $gs = $db->prepare($sqlGr);
            $gs->execute([':gid'=>$gradeId]);
            foreach ($gs->fetchAll() as $r) {
                $mode = strtolower((string)$r['calc_mode']); $rate = (float)($r['rate'] ?? 0);
                if ($mode === 'percent_basic') $total += $basicSalary * $rate/100.0;
                elseif ($mode === 'percent_gross') $total += $gross * $rate/100.0;
                else $total += (float)($r['amount'] ?? 0);
            }
        }
    } catch (Throwable $e) { /* ignore */ }
    // Loan deduction for the month
    $loan = loanMonthlyDue($db, (int)$employeeId, (string)$payPeriodEnd);
    return round($total + (float)$loan, 2);
}

function saveDeductionType($db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($input['id']) ? (int)$input['id'] : 0;
    $name = trim($input['name'] ?? '');
    $code = trim($input['code'] ?? '');
    if ($name === '' || $code === '') ApiResponse::error('Name and Code are required');
    try {
        if ($id > 0) {
            $q = $db->prepare("UPDATE deduction_types SET name = :n, code = :c WHERE id = :id AND company_id = :cid");
            $q->execute([':n'=>$name, ':c'=>$code, ':id'=>$id, ':cid'=>$user['company_id']]);
        } else {
            $q = $db->prepare("INSERT INTO deduction_types (company_id, name, code) VALUES (:cid, :n, :c)");
            $q->execute([':cid'=>$user['company_id'], ':n'=>$name, ':c'=>$code]);
        }
        ApiResponse::success(null, 'Saved');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to save deduction type: '.$e->getMessage(), 500);
    }
}

function deleteAllowanceType($db, $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('ID required');
    try {
        $q = $db->prepare("DELETE FROM allowance_types WHERE id = :id AND company_id = :cid");
        $q->execute([':id'=>(int)$id, ':cid'=>$user['company_id']]);
        ApiResponse::success(null, 'Deleted');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to delete allowance type: '.$e->getMessage(), 500);
    }
}

function deleteDeductionType($db, $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('ID required');
    try {
        $q = $db->prepare("DELETE FROM deduction_types WHERE id = :id AND company_id = :cid");
        $q->execute([':id'=>(int)$id, ':cid'=>$user['company_id']]);
        ApiResponse::success(null, 'Deleted');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to delete deduction type: '.$e->getMessage(), 500);
    }
}

function getPayrolls($db) {
    $user = getCurrentUser();
    $role = $user['role_slug'];
    
    // Check permissions - only HR and admin can view all payrolls
    if (!in_array($role, ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        // Employees can only see their own payrolls
        if ($role === 'employee') {
            getEmployeePayrolls($db, $user);
            return;
        } else {
            ApiResponse::forbidden('Insufficient permissions');
        }
    }
    
    $query = "SELECT 
                p.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                d.name as department_name,
                CONCAT(approver.first_name, ' ', approver.last_name) as approved_by_name
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN users u_approver ON p.approved_by = u_approver.id
              LEFT JOIN employees approver ON u_approver.employee_id = approver.id
              WHERE p.company_id = :company_id";
    
    $params = [':company_id' => $user['company_id']];
    
    // Apply filters
    if (isset($_GET['status']) && !empty($_GET['status'])) {
        $query .= " AND p.status = :status";
        $params[':status'] = $_GET['status'];
    }
    
    if (isset($_GET['period_month']) && !empty($_GET['period_month'])) {
        $query .= " AND MONTH(p.pay_period_start) = :period_month";
        $params[':period_month'] = $_GET['period_month'];
    }
    
    if (isset($_GET['period_year']) && !empty($_GET['period_year'])) {
        $query .= " AND YEAR(p.pay_period_start) = :period_year";
        $params[':period_year'] = $_GET['period_year'];
    }
    
    $query .= " ORDER BY p.pay_period_start DESC, e.first_name ASC";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    $payrolls = $stmt->fetchAll();
    ApiResponse::success($payrolls);
}

function getPayroll($db, $id) {
    $user = getCurrentUser();
    if (!$id) ApiResponse::error('Payroll ID required');
    $query = "SELECT 
                p.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                d.name as department_name,
                CONCAT(approver.first_name, ' ', approver.last_name) as approved_by_name
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN users u_approver ON p.approved_by = u_approver.id
              LEFT JOIN employees approver ON u_approver.employee_id = approver.id
              WHERE p.id = :id";
    $params = [':id' => $id];
    if ($user['role_slug'] === 'employee') {
        // employees can only see their own approved/paid payslips
        $empStmt = $db->prepare("SELECT id FROM employees WHERE user_id = :uid");
        $empStmt->bindValue(':uid', $user['id']);
        $empStmt->execute();
        if ($empStmt->rowCount() === 0) ApiResponse::forbidden('Access denied');
        $employee = $empStmt->fetch();
        $query .= " AND p.employee_id = :employee_id AND p.status IN ('approved','paid')";
        $params[':employee_id'] = $employee['id'];
    } else {
        // scope by company for HR/Admin roles
        $query .= " AND p.company_id = :company_id";
        $params[':company_id'] = $user['company_id'];
    }
    $stmt = $db->prepare($query);
    foreach ($params as $k => $v) { $stmt->bindValue($k, $v); }
    $stmt->execute();
    if ($stmt->rowCount() === 0) ApiResponse::notFound('Payroll not found');
    ApiResponse::success($stmt->fetch());
}

function createPayroll($db) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    foreach (['employee_id','pay_period_start','pay_period_end'] as $f) {
        if (empty($input[$f])) ApiResponse::error("Field '$f' is required");
    }
    // prevent duplicate for same period
    $check = $db->prepare("SELECT id FROM payrolls WHERE employee_id = :eid AND pay_period_start = :s AND pay_period_end = :e");
    $check->execute([':eid'=>$input['employee_id'], ':s'=>$input['pay_period_start'], ':e'=>$input['pay_period_end']]);
    if ($check->rowCount() > 0) ApiResponse::error('Payroll already exists for this period');
    $calc = calculateEmployeePayroll($db, (int)$input['employee_id'], $input['pay_period_start'], $input['pay_period_end'], $user['company_id']);
    if (!$calc) ApiResponse::error('Employee not found');
    $ins = $db->prepare("INSERT INTO payrolls (
        company_id, employee_id, pay_period_start, pay_period_end,
        basic_salary, gross_salary, total_allowances, taxable_allowances, exempt_allowances,
        total_deductions, ssnit_employee, ssnit_employer, chargeable_income,
        total_tax, net_salary, status, created_by, created_at
    ) VALUES (
        :cid, :eid, :s, :e, :basic, :gross, :allow, :taxable_allowances, :exempt_allowances,
        :deduct, :ssnit_employee, :ssnit_employer, :chargeable_income,
        :tax, :net, 'calculated', :uid, NOW()
    )");
    $ins->execute([
        ':cid'=>$user['company_id'],
        ':eid'=>$input['employee_id'],
        ':s'=>$input['pay_period_start'],
        ':e'=>$input['pay_period_end'],
        ':basic'=>$calc['basic_salary'],
        ':gross'=>$calc['gross_salary'],
        ':allow'=>$calc['total_allowances'],
        ':taxable_allowances'=>$calc['taxable_allowances'],
        ':exempt_allowances'=>$calc['exempt_allowances'],
        ':deduct'=>$calc['total_deductions'],
        ':ssnit_employee'=>$calc['ssnit_employee'],
        ':ssnit_employer'=>$calc['ssnit_employer'],
        ':chargeable_income'=>$calc['chargeable_income'],
        ':tax'=>$calc['total_tax'],
        ':net'=>$calc['net_salary'],
        ':uid'=>$user['id']
    ]);
    ApiResponse::success(['id' => $db->lastInsertId()], 'Payroll created');
}

function updatePayroll($db, $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Payroll ID required');
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $status = $input['status'] ?? null;
    if ($status !== 'paid') ApiResponse::error('Only status=paid update is supported');
    // ensure record belongs to company and is approved
    $q = $db->prepare("SELECT p.id FROM payrolls p JOIN employees e ON p.employee_id = e.id WHERE p.id = :id AND e.company_id = :cid AND p.status = 'approved'");
    $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    if ($q->rowCount() === 0) ApiResponse::error('Payroll not found or not approvable for payment');
    // Mark paid
    $u = $db->prepare("UPDATE payrolls SET status = 'paid', paid_at = NOW(), updated_at = NOW() WHERE id = :id");
    $u->execute([':id'=>$id]);

    // Auto-service loan installments for this employee for the payroll month
    try {
        // Load payroll to get employee and period
        $p = $db->prepare("SELECT employee_id, pay_period_end FROM payrolls WHERE id = :id");
        $p->execute([':id'=>$id]);
        $row = $p->fetch();
        if ($row) {
            $eid = (int)$row['employee_id'];
            $pend = (string)$row['pay_period_end'];
            $monthStart = date('Y-m-01', strtotime($pend));
            $monthEnd   = date('Y-m-t', strtotime($pend));
            ensureLoanTables($db);
            // Update schedules
            $upd = $db->prepare("UPDATE loan_schedules ls
                                 JOIN employee_loans l ON l.id = ls.loan_id
                                 SET ls.amount_paid = ls.amount_due,
                                     ls.status = 'paid',
                                     ls.paid_at = NOW(),
                                     ls.payroll_id = :pid
                                 WHERE l.employee_id = :eid
                                   AND ls.status IN ('due','partial')
                                   AND ls.due_date BETWEEN :s AND :e");
            $upd->execute([':pid'=>$id, ':eid'=>$eid, ':s'=>$monthStart, ':e'=>$monthEnd]);
            // Refresh balances for affected loans
            $ids = $db->prepare("SELECT DISTINCT l.id FROM loan_schedules ls JOIN employee_loans l ON l.id = ls.loan_id WHERE l.employee_id = :eid AND ls.due_date BETWEEN :s AND :e");
            $ids->execute([':eid'=>$eid, ':s'=>$monthStart, ':e'=>$monthEnd]);
            foreach ($ids->fetchAll() as $r){ refreshLoanBalance($db, (int)$r['id']); }
        }
    } catch (Throwable $e) { /* best-effort */ }

    ApiResponse::success(null, 'Payroll marked as paid and loan schedules serviced');
}

function deletePayroll($db, $id) {
    $user = getCurrentUser();
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Payroll ID required');
    // allow delete only when still calculated
    $q = $db->prepare("DELETE p FROM payrolls p JOIN employees e ON p.employee_id = e.id WHERE p.id = :id AND e.company_id = :cid AND p.status = 'calculated'");
    $q->execute([':id'=>$id, ':cid'=>$user['company_id']]);
    if ($q->rowCount() === 0) ApiResponse::error('Payroll not found or cannot be deleted');
    ApiResponse::success(null, 'Payroll deleted');
}

function getEmployeePayrolls($db, $user) {
    // Get employee ID for current user
    $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
    $empStmt = $db->prepare($empQuery);
    $empStmt->bindParam(':user_id', $user['id']);
    $empStmt->execute();
    
    if ($empStmt->rowCount() === 0) {
        ApiResponse::success([]);
    }
    
    $employee = $empStmt->fetch();
    
    $query = "SELECT 
                p.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                d.name as department_name
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              WHERE p.employee_id = :employee_id AND p.status IN ('approved', 'paid')
              ORDER BY p.pay_period_start DESC";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':employee_id', $employee['id']);
    $stmt->execute();
    
    $payrolls = $stmt->fetchAll();
    ApiResponse::success($payrolls);
}

function calculatePayroll($db) {
    $user = getCurrentUser();
    
    // Check permissions
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['employee_id']) || !isset($input['pay_period_start']) || !isset($input['pay_period_end'])) {
        ApiResponse::error('Employee ID and pay period are required');
    }
    
    try {
        // Get employee details
        $empQuery = "SELECT e.*, p.title as position_title FROM employees e LEFT JOIN positions p ON e.position_id = p.id WHERE e.id = :employee_id AND e.company_id = :company_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindValue(':employee_id', $input['employee_id']);
        $empStmt->bindValue(':company_id', $user['company_id']);
        $empStmt->execute();
        if ($empStmt->rowCount() === 0) ApiResponse::error('Employee not found');
        $employee = $empStmt->fetch();

        // Delegate to common calculator
        $calc = calculateEmployeePayroll($db, (int)$input['employee_id'], $input['pay_period_start'], $input['pay_period_end'], $user['company_id']);
        if (!$calc) ApiResponse::error('Unable to calculate payroll');
        $calculation = array_merge($calc, [
            'employee_id' => (int)$input['employee_id'],
            'employee_name' => $employee['first_name'] . ' ' . $employee['last_name'],
            'employee_number' => $employee['employee_number'],
            'position' => $employee['position_title'],
            'pay_period_start' => $input['pay_period_start'],
            'pay_period_end' => $input['pay_period_end']
        ]);
        ApiResponse::success($calculation);
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to calculate payroll: ' . $e->getMessage());
    }
}

function processPayroll($db) {
    $user = getCurrentUser();
    
    // Check permissions
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['employees']) || !is_array($input['employees'])) {
        ApiResponse::error('Employee list is required');
    }
    
    if (!isset($input['pay_period_start']) || !isset($input['pay_period_end'])) {
        ApiResponse::error('Pay period is required');
    }
    
    try {
        $db->beginTransaction();
        
        $processedCount = 0;
        $errors = [];
        
        foreach ($input['employees'] as $employeeId) {
            try {
                // Check if payroll already exists for this period
                $checkQuery = "SELECT id FROM payrolls 
                              WHERE employee_id = :employee_id 
                              AND pay_period_start = :pay_period_start 
                              AND pay_period_end = :pay_period_end";
                
                $checkStmt = $db->prepare($checkQuery);
                $checkStmt->bindParam(':employee_id', $employeeId);
                $checkStmt->bindParam(':pay_period_start', $input['pay_period_start']);
                $checkStmt->bindParam(':pay_period_end', $input['pay_period_end']);
                $checkStmt->execute();
                
                if ($checkStmt->rowCount() > 0) {
                    $errors[] = "Skipped duplicate payroll for employee ID $employeeId (period {$input['pay_period_start']} to {$input['pay_period_end']})";
                    notifyPayrollDuplicate($db, $employeeId, $input['pay_period_start'], $input['pay_period_end']);
                    continue;
                }
                
                // Get employee details and calculate payroll
                $calculation = calculateEmployeePayroll($db, $employeeId, $input['pay_period_start'], $input['pay_period_end'], $user['company_id']);
                
                if ($calculation) {
                    // Insert payroll record
                    $insertQuery = "INSERT INTO payrolls (
                                      company_id, employee_id, pay_period_start, pay_period_end,
                                      basic_salary, gross_salary, total_allowances, taxable_allowances, exempt_allowances,
                                      total_deductions, ssnit_employee, ssnit_employer, chargeable_income,
                                      total_tax, net_salary, status, created_by, created_at
                                    ) VALUES (
                                      :company_id, :employee_id, :pay_period_start, :pay_period_end,
                                      :basic_salary, :gross_salary, :total_allowances, :taxable_allowances, :exempt_allowances,
                                      :total_deductions, :ssnit_employee, :ssnit_employer, :chargeable_income,
                                      :total_tax, :net_salary, 'calculated', :created_by, NOW()
                                    )";
                    
                    $insertStmt = $db->prepare($insertQuery);
                    $insertStmt->bindParam(':company_id', $user['company_id']);
                    $insertStmt->bindParam(':employee_id', $employeeId);
                    $insertStmt->bindParam(':pay_period_start', $input['pay_period_start']);
                    $insertStmt->bindParam(':pay_period_end', $input['pay_period_end']);
                    $insertStmt->bindParam(':basic_salary', $calculation['basic_salary']);
                    $insertStmt->bindParam(':gross_salary', $calculation['gross_salary']);
                    $insertStmt->bindParam(':total_allowances', $calculation['total_allowances']);
                    $insertStmt->bindParam(':taxable_allowances', $calculation['taxable_allowances']);
                    $insertStmt->bindParam(':exempt_allowances', $calculation['exempt_allowances']);
                    $insertStmt->bindParam(':total_deductions', $calculation['total_deductions']);
                    $insertStmt->bindParam(':ssnit_employee', $calculation['ssnit_employee']);
                    $insertStmt->bindParam(':ssnit_employer', $calculation['ssnit_employer']);
                    $insertStmt->bindParam(':chargeable_income', $calculation['chargeable_income']);
                    $insertStmt->bindParam(':total_tax', $calculation['total_tax']);
                    $insertStmt->bindParam(':net_salary', $calculation['net_salary']);
                    $insertStmt->bindParam(':created_by', $user['id']);
                    
                    $insertStmt->execute();
                    $processedCount++;
                }
                
            } catch (Exception $e) {
                $errors[] = "Error processing employee ID $employeeId: " . $e->getMessage();
            }
        }
        
        $db->commit();
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'payroll_processed', [
            'processed_count' => $processedCount,
            'pay_period' => $input['pay_period_start'] . ' to ' . $input['pay_period_end']
        ]);
        
        $response = [
            'processed_count' => $processedCount,
            'total_employees' => count($input['employees']),
            'errors' => $errors
        ];
        
        ApiResponse::success($response, 'Payroll processing completed');
        
    } catch (Exception $e) {
        $db->rollback();
        ApiResponse::error('Failed to process payroll: ' . $e->getMessage());
    }
}

function calculateEmployeePayroll($db, $employeeId, $payPeriodStart, $payPeriodEnd, $companyId) {
    // Get employee details (including grade)
    $empQuery = "SELECT id, company_id, grade_id, salary FROM employees WHERE id = :employee_id AND company_id = :company_id";
    $empStmt = $db->prepare($empQuery);
    $empStmt->bindValue(':employee_id', $employeeId);
    $empStmt->bindValue(':company_id', $companyId);
    $empStmt->execute();
    if ($empStmt->rowCount() === 0) return null;
    $employee = $empStmt->fetch();
    $gradeId = $employee['grade_id'] ?? null;
    // Attempt to resolve grade automatically using helper (checks column existence internally)
    if (!$gradeId) {
        $maybe = resolveEmployeeGradeId($db, $employeeId, $companyId);
        if ($maybe) { $gradeId = (int)$maybe; }
    }
    if (!$gradeId) {
        try { app_log('warning', 'payroll_calc_grade_missing', ['employee_id' => $employeeId]); } catch (Throwable $e) { /* ignore */ }
    }
    $basicSalary = (float)($employee['salary'] ?? 0);
    // Fallback to grade base salary if employee salary missing/zero
    if (($basicSalary === 0.0 || $basicSalary === null) && $gradeId) {
        try {
            $gs = $db->prepare("SELECT base_salary FROM grades WHERE id = :gid AND company_id = :cid");
            $gs->execute([':gid'=>$gradeId, ':cid'=>$companyId]);
            $gbs = $gs->fetchColumn();
            if ($gbs !== false && $gbs !== null) $basicSalary = (float)$gbs;
        } catch (Throwable $e) { /* ignore and keep 0 */ }
    }

    // Allowances: taxable/exempt + totals (employee + grade defaults)
    $ab = computeAllowanceBreakdown($db, $employeeId, $gradeId, $payPeriodEnd, $basicSalary);
    $totalAllowances = (float)$ab['total'];
    $taxableAllowances = (float)$ab['taxable'];
    $exemptAllowances = (float)$ab['exempt'];

    // Gross Income
    $gross = $basicSalary + $totalAllowances;

    // Deductions (other than SSNIT/PAYE)
    $otherDeductions = computeDeductionsTotal($db, $employeeId, $gradeId, $payPeriodEnd, $basicSalary, $gross);

    // SSNIT (Ghana) - employee 5.5% of basic salary, employer 13% of gross per latest guidance
    $ssnitEmployee = round($basicSalary * 0.055, 2);
    $ssnitEmployer = round($gross * 0.13, 2);

    // Chargeable income excludes SSNIT employee and exempt allowances (non-taxable allowances)
    $chargeable = max(0, ($basicSalary + $taxableAllowances) - $ssnitEmployee);

    // PAYE (Ghana bands)
    $paye = ghana_paye_tax($chargeable);

    // Net Salary
    $net = $gross - $ssnitEmployee - $paye - $otherDeductions;

    return [
        'basic_salary' => round($basicSalary, 2),
        'total_allowances' => round($totalAllowances, 2),
        'taxable_allowances' => round($taxableAllowances, 2),
        'exempt_allowances' => round($exemptAllowances, 2),
        'gross_salary' => round($gross, 2),
        'ssnit_employee' => $ssnitEmployee,
        'ssnit_employer' => $ssnitEmployer,
        'chargeable_income' => round($chargeable, 2),
        'total_deductions' => round($otherDeductions, 2),
        'total_tax' => round($paye, 2),
        'net_salary' => round($net, 2)
    ];
}

function approvePayroll($db, $id) {
    $user = getCurrentUser();
    
    // Check permissions
    if (!in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    
    if (!$id) {
        ApiResponse::error('Payroll ID required');
    }
    
    try {
        // Check if payroll exists and is calculated
        $checkQuery = "SELECT p.*, e.company_id 
                       FROM payrolls p 
                       JOIN employees e ON p.employee_id = e.id 
                       WHERE p.id = :id AND p.status = 'calculated' AND e.company_id = :company_id";
        
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id);
        $checkStmt->bindParam(':company_id', $user['company_id']);
        $checkStmt->execute();
        
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Payroll not found or already processed');
        }
        
        // Update payroll status
        $query = "UPDATE payrolls SET 
                    status = 'approved',
                    approved_by = :approved_by,
                    approved_at = NOW(),
                    updated_at = NOW()
                  WHERE id = :id";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id);
        $stmt->bindParam(':approved_by', $user['id']);
        $stmt->execute();
        
        // Create notification for employee
        createPayrollNotification($db, $id, 'payroll_approved');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'payroll_approved', ['payroll_id' => $id]);
        
        // Attempt to email the payslip to the employee (non-blocking best-effort)
        try { sendPayslipEmail($db, (int)$id); } catch (Throwable $e) { /* ignore */ }

        ApiResponse::success(null, 'Payroll approved successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to approve payroll: ' . $e->getMessage());
    }
}

function getPayslip($db, $id) {
    $user = getCurrentUser();
    
    if (!$id) {
        ApiResponse::error('Payroll ID required');
    }
    
    // Get payroll details with employee/company info
    $query = "SELECT 
                p.*,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                e.email,
                e.date_of_birth,
                e.hire_date,
                d.name as department_name,
                pos.title as position_title,
                c.name as company_name,
                c.address as company_address,
                c.logo AS company_logo
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN positions pos ON e.position_id = pos.id
              LEFT JOIN companies c ON c.id = p.company_id
              WHERE p.id = :id";
    
    $params = [':id' => $id];
    
    // Check permissions - employees can only see their own payslips
    if ($user['role_slug'] === 'employee') {
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $employee = $empStmt->fetch();
            $query .= " AND p.employee_id = :employee_id";
            $params[':employee_id'] = $employee['id'];
        } else {
            ApiResponse::forbidden('Access denied');
        }
    } else {
        // HR can see all payslips
        $query .= " AND p.company_id = :company_id";
        $params[':company_id'] = $user['company_id'];
    }
    
    $query .= " AND p.status IN ('approved', 'paid')";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $payslip = $stmt->fetch();
        
        // Get detailed allowances and deductions
        $basic = (float)($payslip['basic_salary'] ?? 0);
        $gross = (float)($payslip['gross_salary'] ?? ($basic + (float)($payslip['total_allowances'] ?? 0)));
        $payslip['allowances'] = getPayrollAllowances($db, $payslip['employee_id'], $payslip['pay_period_end'], $basic, $gross);
        $payslip['deductions'] = getPayrollDeductions($db, $payslip['employee_id'], $payslip['pay_period_end'], $basic, $gross);
        
        ApiResponse::success($payslip);
    } else {
        ApiResponse::notFound('Payslip not found');
    }
}

function getPayrollAllowances(PDO $db, int $employeeId, string $payPeriodEnd, float $basicSalary, float $gross) {
    // Compute resolved amount for each item using calc_mode/rate
    $sql = "(
                SELECT at.name, at.code,
                       CASE COALESCE(ea.calc_mode,'fixed')
                         WHEN 'percent_basic' THEN :basic * COALESCE(ea.rate,0)/100
                         WHEN 'percent_gross' THEN :gross * COALESCE(ea.rate,0)/100
                         ELSE COALESCE(ea.amount,0)
                       END AS amount,
                       'employee' AS source
                FROM employee_allowances ea
                JOIN allowance_types at ON ea.allowance_type_id = at.id
                WHERE ea.employee_id = :eid 
                  AND ea.status = 'active'
                  AND (ea.end_date IS NULL OR ea.end_date >= :pend)
            )
            UNION ALL
            (
                SELECT at2.name, at2.code,
                       CASE COALESCE(ga.calc_mode,'fixed')
                         WHEN 'percent_basic' THEN :basic2 * COALESCE(ga.rate,0)/100
                         WHEN 'percent_gross' THEN :gross2 * COALESCE(ga.rate,0)/100
                         ELSE COALESCE(ga.amount,0)
                       END AS amount,
                       'grade' AS source
                FROM grade_allowances ga
                JOIN employees e ON e.id = :eid2
                JOIN allowance_types at2 ON ga.allowance_type_id = at2.id
                WHERE ga.grade_id = e.grade_id
            )";
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':basic', $basicSalary);
    $stmt->bindValue(':gross', $gross);
    $stmt->bindValue(':eid', $employeeId, PDO::PARAM_INT);
    $stmt->bindValue(':pend', $payPeriodEnd);
    $stmt->bindValue(':basic2', $basicSalary);
    $stmt->bindValue(':gross2', $gross);
    $stmt->bindValue(':eid2', $employeeId, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetchAll();
}

function getPayrollDeductions(PDO $db, int $employeeId, string $payPeriodEnd, float $basicSalary, float $gross) {
    // Include employee deductions, grade defaults, and loan repayment
    ensureLoanTables($db);
    $monthStart = date('Y-m-01', strtotime($payPeriodEnd));
    $monthEnd   = date('Y-m-t', strtotime($payPeriodEnd));
    $sql = "(
                SELECT dt.name, dt.code,
                       CASE COALESCE(ed.calc_mode,'fixed')
                         WHEN 'percent_basic' THEN :basic * COALESCE(ed.rate,0)/100
                         WHEN 'percent_gross' THEN :gross * COALESCE(ed.rate,0)/100
                         ELSE COALESCE(ed.amount,0)
                       END AS amount,
                       'employee' AS source
                FROM employee_deductions ed
                JOIN deduction_types dt ON ed.deduction_type_id = dt.id
                WHERE ed.employee_id = :eid 
                  AND ed.status = 'active'
                  AND (ed.end_date IS NULL OR ed.end_date >= :pend)
            )
            UNION ALL
            (
                SELECT dt2.name, dt2.code,
                       CASE COALESCE(gd.calc_mode,'fixed')
                         WHEN 'percent_basic' THEN :basic2 * COALESCE(gd.rate,0)/100
                         WHEN 'percent_gross' THEN :gross2 * COALESCE(gd.rate,0)/100
                         ELSE COALESCE(gd.amount,0)
                       END AS amount,
                       'grade' AS source
                FROM grade_deductions gd
                JOIN employees e ON e.id = :eid2
                JOIN deduction_types dt2 ON gd.deduction_type_id = dt2.id
                WHERE gd.grade_id = e.grade_id
            )
            UNION ALL
            (
                SELECT 'Loan Repayment' AS name, 'LOAN' AS code, 
                       COALESCE(SUM(ls.amount_due - COALESCE(ls.amount_paid,0)),0) AS amount,
                       'loan' AS source
                FROM loan_schedules ls
                JOIN employee_loans l ON l.id = ls.loan_id
                WHERE l.employee_id = :eid3
                  AND l.status IN ('approved','active')
                  AND ls.status IN ('due','partial')
                  AND ls.due_date BETWEEN :s AND :e
            )";
    $stmt = $db->prepare($sql);
    $stmt->bindValue(':basic', $basicSalary);
    $stmt->bindValue(':gross', $gross);
    $stmt->bindValue(':eid', $employeeId, PDO::PARAM_INT);
    $stmt->bindValue(':pend', $payPeriodEnd);
    $stmt->bindValue(':basic2', $basicSalary);
    $stmt->bindValue(':gross2', $gross);
    $stmt->bindValue(':eid2', $employeeId, PDO::PARAM_INT);
    $stmt->bindValue(':eid3', $employeeId, PDO::PARAM_INT);
    $stmt->bindValue(':s', $monthStart);
    $stmt->bindValue(':e', $monthEnd);
    $stmt->execute();
    return $stmt->fetchAll();
}

function createPayrollNotification($db, $payrollId, $type) {
    // Get payroll details
    $query = "SELECT p.*, 
                     CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                     e.user_id as employee_user_id
              FROM payrolls p
              JOIN employees e ON p.employee_id = e.id
              WHERE p.id = :payroll_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':payroll_id', $payrollId);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $payroll = $stmt->fetch();
        
        if ($type === 'payroll_approved' && $payroll['employee_user_id']) {
            $notifQuery = "INSERT INTO notifications (user_id, type, title, content, created_at) 
                           VALUES (:user_id, :type, :title, :content, NOW())";
            $notifStmt = $db->prepare($notifQuery);
            $notifStmt->bindParam(':user_id', $payroll['employee_user_id']);
            $notifStmt->bindParam(':type', $type);
            $notifStmt->bindParam(':title', 'Payslip Available');
            $notifStmt->bindParam(':content', "Your payslip for {$payroll['pay_period_start']} to {$payroll['pay_period_end']} is now available.");
            $notifStmt->execute();
        }
    }
}
?>
