<?php
/**
 * Onboarding Checklists API
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$database = new Database();
$db = $database->getConnection();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$user = getCurrentUser();

ensureTables($db);

switch ($method) {
  case 'GET':
    if ($action === 'instance' && $id) { getInstance($db, $id, $user); }
    else { listInstances($db, $user); }
    break;
  case 'POST':
    if ($action === 'create') createInstance($db, $user);
    elseif ($action === 'task_update') updateTask($db, $user);
    else ApiResponse::error('Unknown action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function ensureTables(PDO $db){
  $db->exec("CREATE TABLE IF NOT EXISTS onboarding_instances (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    employee_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    status ENUM('open','completed','cancelled') NOT NULL DEFAULT 'open',
    started_at DATETIME NOT NULL,
    completed_at DATETIME NULL,
    created_by INT NOT NULL,
    created_at DATETIME NOT NULL,
    INDEX(company_id), INDEX(employee_id), INDEX(status)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS onboarding_tasks (
    id INT AUTO_INCREMENT PRIMARY KEY,
    instance_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    assignee_role ENUM('hr','it','manager','employee') NOT NULL,
    requires_esign TINYINT(1) NOT NULL DEFAULT 0,
    status ENUM('pending','done') NOT NULL DEFAULT 'pending',
    esign_name VARCHAR(255) NULL,
    esign_user_id INT NULL,
    esign_at DATETIME NULL,
    completed_at DATETIME NULL,
    notes TEXT NULL,
    INDEX(instance_id), INDEX(assignee_role), INDEX(status)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

function canManage(array $user){ return in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer']); }

function listInstances(PDO $db, array $user){
  $companyId = (int)$user['company_id'];
  $mine = isset($_GET['mine']) ? (int)$_GET['mine'] : 0;
  $empId = null;
  try { $st=$db->prepare('SELECT id FROM employees WHERE user_id = :uid'); $st->execute([':uid'=>$user['id']]); $empId = $st->fetchColumn() ?: null; } catch (Throwable $e) { $empId = null; }
  $sql = "SELECT oi.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.employee_number,
                 SUM(ot.status='done') AS tasks_done, COUNT(ot.id) AS tasks_total
          FROM onboarding_instances oi
          JOIN employees e ON oi.employee_id = e.id
          LEFT JOIN onboarding_tasks ot ON ot.instance_id = oi.id
          WHERE e.company_id = :cid";
  $params = [':cid'=>$companyId];
  if ($mine && $empId) { $sql .= ' AND oi.employee_id = :eid'; $params[':eid'] = $empId; }
  $sql .= ' GROUP BY oi.id ORDER BY oi.started_at DESC';
  $st = $db->prepare($sql); foreach($params as $k=>$v) $st->bindValue($k,$v); $st->execute();
  ApiResponse::success($st->fetchAll());
}

function getInstance(PDO $db, int $id, array $user){
  $st = $db->prepare("SELECT oi.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.employee_number, e.manager_id
                      FROM onboarding_instances oi JOIN employees e ON oi.employee_id = e.id WHERE oi.id = :id AND e.company_id = :cid");
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  if ($st->rowCount()===0) ApiResponse::notFound('Not found');
  $inst = $st->fetch();
  // RBAC: employees can view own; managers can view their direct's; HR can view all
  if (!canManage($user)) {
    $empId = null; try { $q=$db->prepare('SELECT id FROM employees WHERE user_id = :uid'); $q->execute([':uid'=>$user['id']]); $empId=$q->fetchColumn()?:null; } catch(Throwable $e){ $empId=null; }
    if (!$empId || ($empId != $inst['employee_id'] && $empId != ($inst['manager_id'] ?? -1))) {
      ApiResponse::forbidden('Insufficient permissions');
    }
  }
  $t = $db->prepare('SELECT * FROM onboarding_tasks WHERE instance_id = :id ORDER BY id');
  $t->execute([':id'=>$id]);
  $tasks = $t->fetchAll();
  ApiResponse::success(['instance'=>$inst, 'tasks'=>$tasks]);
}

function createInstance(PDO $db, array $user){
  if (!canManage($user)) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $employee_id = (int)($in['employee_id'] ?? 0);
  $title = trim((string)($in['title'] ?? 'Onboarding'));
  $tasks = is_array($in['tasks'] ?? null) ? $in['tasks'] : [];
  if ($employee_id<=0) ApiResponse::error('employee_id required');
  $st = $db->prepare('INSERT INTO onboarding_instances (company_id, employee_id, title, status, started_at, created_by, created_at) VALUES (:cid,:eid,:title,\'open\',NOW(),:uid,NOW())');
  $st->execute([':cid'=>$user['company_id'], ':eid'=>$employee_id, ':title'=>$title, ':uid'=>$user['id']]);
  $iid = (int)$db->lastInsertId();
  if ($tasks){
    $ins = $db->prepare('INSERT INTO onboarding_tasks (instance_id, title, assignee_role, requires_esign) VALUES (:iid,:title,:role,:es)');
    foreach ($tasks as $t){
      $title = trim((string)($t['title'] ?? ''));
      $role = in_array(($t['assignee_role'] ?? ''), ['hr','it','manager','employee']) ? $t['assignee_role'] : 'employee';
      $es = !empty($t['requires_esign']) ? 1 : 0;
      if ($title!=='') $ins->execute([':iid'=>$iid, ':title'=>$title, ':role'=>$role, ':es'=>$es]);
    }
  }
  ApiResponse::success(['id'=>$iid], 'Onboarding created');
}

function updateTask(PDO $db, array $user){
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $instance_id = (int)($in['instance_id'] ?? 0);
  $task_id = (int)($in['task_id'] ?? 0);
  $status = $in['status'] ?? '';
  $esign_name = isset($in['esign_name']) ? trim((string)$in['esign_name']) : null;
  if ($instance_id<=0 || $task_id<=0) ApiResponse::error('instance_id and task_id required');
  // Load instance and task
  $st = $db->prepare('SELECT oi.*, e.user_id AS emp_user_id, e.manager_id FROM onboarding_instances oi JOIN employees e ON oi.employee_id = e.id WHERE oi.id = :id AND e.company_id = :cid');
  $st->execute([':id'=>$instance_id, ':cid'=>$user['company_id']]);
  if ($st->rowCount()===0) ApiResponse::notFound('Instance not found');
  $inst = $st->fetch();
  $tt = $db->prepare('SELECT * FROM onboarding_tasks WHERE id = :tid AND instance_id = :iid');
  $tt->execute([':tid'=>$task_id, ':iid'=>$instance_id]);
  if ($tt->rowCount()===0) ApiResponse::notFound('Task not found');
  $task = $tt->fetch();
  // Permission: based on assignee_role
  $role = $user['role_slug'] ?? '';
  $empId = null; try { $q=$db->prepare('SELECT id FROM employees WHERE user_id = :uid'); $q->execute([':uid'=>$user['id']]); $empId=$q->fetchColumn()?:null; } catch(Throwable $e) { $empId=null; }
  $allowed = false;
  if (in_array($role, ['super_admin','admin','hr_head','hr_officer']) && $task['assignee_role']==='hr') $allowed = true;
  if ($task['assignee_role']==='employee' && $inst['emp_user_id'] == $user['id']) $allowed = true;
  if ($task['assignee_role']==='manager' && $empId && $empId == ($inst['manager_id'] ?? -1)) $allowed = true;
  if ($task['assignee_role']==='it' && in_array($role,['super_admin','admin'])) $allowed = true; // simple rule for IT
  if (!$allowed) ApiResponse::forbidden('Not allowed to update this task');
  // Update
  $fields = [];$params=[':id'=>$task_id];
  if ($status==='done'){ $fields[]='status=\'done\''; $fields[]='completed_at = NOW()'; }
  if ($esign_name!==null){ $fields[] = 'esign_name = :nm'; $fields[] = 'esign_user_id = :uid'; $fields[] = 'esign_at = NOW()'; $params[':nm']=$esign_name; $params[':uid']=$user['id']; }
  if (!$fields) ApiResponse::success(null,'No changes');
  $sql = 'UPDATE onboarding_tasks SET '.implode(',', $fields).' WHERE id = :id';
  $u = $db->prepare($sql); foreach($params as $k=>$v){ $u->bindValue($k,$v); } $u->execute();
  // If all tasks done, mark instance
  $c = $db->prepare("SELECT SUM(status='done') AS d, COUNT(*) AS t FROM onboarding_tasks WHERE instance_id = :iid");
  $c->execute([':iid'=>$instance_id]); $row=$c->fetch();
  if ((int)($row['t']??0)>0 && (int)($row['d']??0) === (int)$row['t']){
    $db->prepare("UPDATE onboarding_instances SET status='completed', completed_at = NOW() WHERE id = :id")->execute([':id'=>$instance_id]);
  }
  ApiResponse::success(null,'Updated');
}
