<?php
/**
 * Messages API - simple direct messages for wishes
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$database = new Database();
$db = $database->getConnection();

switch ($method) {
  case 'GET':
    if ($action === 'inbox') getInbox($db);
    elseif ($action === 'sent') getSent($db);
    elseif ($action === 'recipients') getRecipients($db);
    else ApiResponse::error('Invalid action');
    break;
  case 'POST':
    if ($action === 'send') sendMessage($db);
    elseif ($action === 'mark_read') markRead($db);
    else ApiResponse::error('Invalid action');
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function sendMessage(PDO $db){
  $me = getCurrentUser();
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $recipientId = isset($in['recipient_user_id']) ? (int)$in['recipient_user_id'] : 0;
  $subject = trim((string)($in['subject'] ?? ''));
  $content = trim((string)($in['content'] ?? ''));
  if ($recipientId <= 0) ApiResponse::error('Recipient required');
  if ($content === '') ApiResponse::error('Message content required');
  if (strlen($content) > 2000) ApiResponse::error('Message too long');

  // verify recipient belongs to same company
  try {
    $st = $db->prepare('SELECT company_id FROM users WHERE id = :uid LIMIT 1');
    $st->bindValue(':uid', $recipientId, PDO::PARAM_INT); $st->execute();
    $rc = $st->fetchColumn();
    if (!$rc || (int)$rc !== (int)$me['company_id']) ApiResponse::forbidden('Cannot message this user');
  } catch (Throwable $e) { ApiResponse::error('Recipient check failed'); }

  try {
    $ins = $db->prepare('INSERT INTO messages (sender_id, recipient_id, subject, content, created_at) VALUES (:s, :r, :subj, :cnt, NOW())');
    $ins->execute([':s'=>$me['id'], ':r'=>$recipientId, ':subj'=>$subject, ':cnt'=>$content]);

    // Optional: create a notification for the recipient if table exists
    try {
      $notif = $db->prepare('INSERT INTO notifications (user_id, type, title, content, data, created_by, created_at) VALUES (:uid, :type, :title, :content, :data, :by, NOW())');
      $payload = json_encode(['from_user_id'=>$me['id']]);
      $title = $subject !== '' ? $subject : 'New message';
      $notif->execute([':uid'=>$recipientId, ':type'=>'wish', ':title'=>$title, ':content'=>$content, ':data'=>$payload, ':by'=>$me['id']]);
    } catch (Throwable $e) { /* ignore */ }

    ApiResponse::success(null, 'Message sent');
  } catch (Throwable $e) {
    ApiResponse::error('Failed to send message: '.$e->getMessage(), 500);
  }
}

function getInbox(PDO $db){
  $me = getCurrentUser();
  try{
    $st = $db->prepare("SELECT m.id, m.subject, m.content, m.created_at, m.read_at,
                               m.sender_id,
                               su.first_name AS sender_first_name, su.last_name AS sender_last_name, su.username AS sender_username
                        FROM messages m
                        JOIN users su ON su.id = m.sender_id AND su.company_id = :cid
                        WHERE m.recipient_id = :me
                        ORDER BY m.created_at DESC
                        LIMIT 500");
    $st->execute([':cid'=>$me['company_id'], ':me'=>$me['id']]);
    ApiResponse::success($st->fetchAll());
  } catch (Throwable $e) { ApiResponse::error('Failed to load inbox: '.$e->getMessage(), 500); }
}

function getSent(PDO $db){
  $me = getCurrentUser();
  try{
    $st = $db->prepare("SELECT m.id, m.subject, m.content, m.created_at, m.read_at,
                               m.recipient_id,
                               ru.first_name AS recipient_first_name, ru.last_name AS recipient_last_name, ru.username AS recipient_username
                        FROM messages m
                        JOIN users ru ON ru.id = m.recipient_id AND ru.company_id = :cid
                        WHERE m.sender_id = :me
                        ORDER BY m.created_at DESC
                        LIMIT 500");
    $st->execute([':cid'=>$me['company_id'], ':me'=>$me['id']]);
    ApiResponse::success($st->fetchAll());
  } catch (Throwable $e) { ApiResponse::error('Failed to load sent messages: '.$e->getMessage(), 500); }
}

function getRecipients(PDO $db){
  $me = getCurrentUser();
  $q = isset($_GET['q']) ? trim((string)$_GET['q']) : '';
  try{
    if ($q !== '') {
      $like = '%'.$q.'%';
      $st = $db->prepare("SELECT u.id, u.username, u.first_name, u.last_name, d.name AS department_name
                          FROM users u
                          LEFT JOIN employees e ON e.id = u.employee_id
                          LEFT JOIN departments d ON d.id = e.department_id
                          WHERE u.company_id = :cid AND u.status = 'active' AND u.id <> :me
                            AND (u.username LIKE :q OR u.first_name LIKE :q OR u.last_name LIKE :q OR CONCAT(u.first_name,' ',u.last_name) LIKE :q OR d.name LIKE :q)
                          ORDER BY u.first_name, u.last_name
                          LIMIT 20");
      $st->execute([':cid'=>$me['company_id'], ':me'=>$me['id'], ':q'=>$like]);
    } else {
      $st = $db->prepare("SELECT u.id, u.username, u.first_name, u.last_name, d.name AS department_name
                          FROM users u
                          LEFT JOIN employees e ON e.id = u.employee_id
                          LEFT JOIN departments d ON d.id = e.department_id
                          WHERE u.company_id = :cid AND u.status = 'active' AND u.id <> :me
                          ORDER BY u.first_name, u.last_name
                          LIMIT 20");
      $st->execute([':cid'=>$me['company_id'], ':me'=>$me['id']]);
    }
    ApiResponse::success($st->fetchAll());
  } catch (Throwable $e) { ApiResponse::error('Failed to load recipients: '.$e->getMessage(), 500); }
}

function markRead(PDO $db){
  $me = getCurrentUser();
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $id = isset($in['id']) ? (int)$in['id'] : 0;
  if ($id <= 0) ApiResponse::error('Invalid message');
  try{
    // ensure message belongs to the user as recipient
    $chk = $db->prepare('SELECT 1 FROM messages m JOIN users r ON r.id = m.recipient_id WHERE m.id = :id AND m.recipient_id = :me AND r.company_id = :cid');
    $chk->execute([':id'=>$id, ':me'=>$me['id'], ':cid'=>$me['company_id']]);
    if ($chk->rowCount() === 0) ApiResponse::forbidden('Not allowed');
    $up = $db->prepare('UPDATE messages SET read_at = IFNULL(read_at, NOW()) WHERE id = :id');
    $up->execute([':id'=>$id]);
    ApiResponse::success(null, 'Marked as read');
  } catch (Throwable $e) { ApiResponse::error('Failed to update message: '.$e->getMessage(), 500); }
}
