<?php
/**
 * Current User (Me) API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$database = new Database();
$db = $database->getConnection();

switch ($method) {
  case 'GET':
    if ($action === 'profile') profile($db);
    elseif ($action === 'settings') settings($db);
    else ApiResponse::error('Invalid action');
    break;
  case 'POST':
    if ($action === 'update_profile') updateProfile($db);
    elseif ($action === 'update_settings') updateSettings($db);
    elseif ($action === 'change_password') changePassword($db);
    elseif ($action === 'upload_avatar') uploadAvatar($db);
    else ApiResponse::error('Invalid action');
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function refreshSessionUser(PDO $db, int $uid){
  try{
    $q = $db->prepare("SELECT u.*, ur.role_id, r.slug as role_slug, r.name as role_name
                       FROM users u 
                       LEFT JOIN user_roles ur ON u.id = ur.user_id 
                       LEFT JOIN roles r ON ur.role_id = r.id 
                       WHERE u.id = :id LIMIT 1");
    $q->bindValue(':id', $uid, PDO::PARAM_INT); $q->execute();
    $row = $q->fetch(); if ($row){ unset($row['password']); $_SESSION['user'] = $row; }
  }catch(Throwable $e){ /* ignore */ }
}

function profile(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id']; $cid = (int)$u['company_id'];
  try{
    $sql = "SELECT u.id, u.username, u.email, u.first_name, u.last_name, u.phone, u.avatar, u.language, u.timezone, u.status, u.last_login_at, u.created_at, u.updated_at,
                   e.id AS employee_id, e.employee_number, e.hire_date, e.department_id, e.position_id, e.branch_id,
                   d.name AS department_name, p.title AS position_title, b.name AS branch_name
            FROM users u
            LEFT JOIN employees e ON e.id = u.employee_id AND e.company_id = :cid
            LEFT JOIN departments d ON d.id = e.department_id
            LEFT JOIN positions p ON p.id = e.position_id
            LEFT JOIN branches b ON b.id = e.branch_id
            WHERE u.id = :uid AND u.company_id = :cid LIMIT 1";
    $st = $db->prepare($sql); $st->execute([':uid'=>$uid, ':cid'=>$cid]);
    if ($st->rowCount()===0) ApiResponse::notFound('User not found');
    $row = $st->fetch();
    // roles
    $r = $db->prepare("SELECT r.name, r.slug, r.level FROM user_roles ur JOIN roles r ON r.id=ur.role_id WHERE ur.user_id = :uid ORDER BY r.level DESC, r.name");
    $r->execute([':uid'=>$uid]);
    $row['roles'] = $r->fetchAll();
    ApiResponse::success($row);
  }catch(Throwable $e){ ApiResponse::error('Failed to load profile: '.$e->getMessage(), 500); }
}

function settings(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id'];
  try{
    $st = $db->prepare('SELECT language, timezone FROM users WHERE id = :id');
    $st->execute([':id'=>$uid]);
    $row = $st->fetch() ?: ['language'=>null,'timezone'=>null];
    ApiResponse::success($row);
  }catch(Throwable $e){ ApiResponse::error('Failed to load settings', 500); }
}

function updateProfile(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id']; $cid = (int)$u['company_id'];
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $first = trim((string)($in['first_name'] ?? ''));
  $last  = trim((string)($in['last_name'] ?? ''));
  $phone = isset($in['phone']) ? trim((string)$in['phone']) : null;
  $email = isset($in['email']) ? trim((string)$in['email']) : null;
  $username = isset($in['username']) ? trim((string)$in['username']) : null;
  try{
    // Uniqueness checks for username/email if provided
    if ($username !== null && $username !== ''){
      $q = $db->prepare('SELECT 1 FROM users WHERE username = :un AND id <> :id AND company_id = :cid LIMIT 1');
      $q->execute([':un'=>$username, ':id'=>$uid, ':cid'=>$cid]);
      if ($q->fetchColumn()) ApiResponse::error('Username already in use');
    }
    if ($email !== null && $email !== ''){
      $q = $db->prepare('SELECT 1 FROM users WHERE email = :em AND id <> :id AND company_id = :cid LIMIT 1');
      $q->execute([':em'=>$email, ':id'=>$uid, ':cid'=>$cid]);
      if ($q->fetchColumn()) ApiResponse::error('Email already in use');
    }
    $fields = [];$params = [':id'=>$uid];
    if ($first !== ''){ $fields[] = 'first_name = :fn'; $params[':fn']=$first; }
    if ($last !== ''){ $fields[] = 'last_name = :ln'; $params[':ln']=$last; }
    if ($phone !== null){
      if ($phone === '') { $fields[] = 'phone = NULL'; }
      else { $fields[] = 'phone = :ph'; $params[':ph']=$phone; }
    }
    if ($email !== null){
      if ($email === '') { $fields[] = 'email = NULL'; }
      else { $fields[] = 'email = :em'; $params[':em']=$email; }
    }
    if ($username !== null && $username !== ''){ $fields[] = 'username = :un'; $params[':un']=$username; }
    if (!$fields) ApiResponse::error('No fields to update');
    $sql = 'UPDATE users SET '.implode(', ',$fields).', updated_at = NOW() WHERE id = :id AND company_id = :cid';
    $params[':cid']=$cid;
    $st = $db->prepare($sql); $st->execute($params);
    refreshSessionUser($db, $uid);
    ApiResponse::success(null, 'Profile updated');
  }catch(Throwable $e){ ApiResponse::error('Failed to update profile: '.$e->getMessage(), 500); }
}

function updateSettings(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id'];
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $language = isset($in['language']) ? trim((string)$in['language']) : null;
  $timezone = isset($in['timezone']) ? trim((string)$in['timezone']) : null;
  try{
    $fields = [];$params=[':id'=>$uid];
    if ($language !== null){ $fields[] = 'language = :lang'; $params[':lang']=$language; }
    if ($timezone !== null){ $fields[] = 'timezone = :tz'; $params[':tz']=$timezone; }
    if (!$fields) ApiResponse::error('No settings to update');
    $sql = 'UPDATE users SET '.implode(', ',$fields).', updated_at = NOW() WHERE id = :id';
    $st = $db->prepare($sql); $st->execute($params);
    refreshSessionUser($db, $uid);
    ApiResponse::success(null, 'Settings saved');
  }catch(Throwable $e){ ApiResponse::error('Failed to update settings: '.$e->getMessage(), 500); }
}

function changePassword(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id'];
  $in = json_decode(file_get_contents('php://input'), true) ?: [];
  $current = (string)($in['current_password'] ?? '');
  $new = (string)($in['new_password'] ?? '');
  if ($current === '' || $new === '') ApiResponse::error('Current and new password are required');
  try{
    $st = $db->prepare('SELECT password FROM users WHERE id = :id'); $st->execute([':id'=>$uid]);
    $hash = $st->fetchColumn(); if (!$hash || !password_verify($current, $hash)) ApiResponse::error('Current password is incorrect', 400);
    $newHash = password_hash($new, PASSWORD_DEFAULT);
    $up = $db->prepare('UPDATE users SET password = :pw, password_changed_at = NOW(), must_change_password = 0 WHERE id = :id');
    $up->execute([':pw'=>$newHash, ':id'=>$uid]);
    ApiResponse::success(null, 'Password changed');
  }catch(Throwable $e){ ApiResponse::error('Failed to change password: '.$e->getMessage(), 500); }
}

function uploadAvatar(PDO $db){
  $u = getCurrentUser(); $uid = (int)$u['id'];
  if (!isset($_FILES['file']) || !is_array($_FILES['file'])) ApiResponse::error('No file uploaded');
  $err = $_FILES['file']['error'] ?? UPLOAD_ERR_OK; if ($err !== UPLOAD_ERR_OK) ApiResponse::error('Upload error');
  $name = $_FILES['file']['name'] ?? 'avatar';
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  $allowed = ['png','jpg','jpeg','gif','webp'];
  if (!in_array($ext, $allowed, true)) ApiResponse::error('Unsupported file type');
  if (($_FILES['file']['size'] ?? 0) > 5*1024*1024) ApiResponse::error('File too large');
  try{
    $base = dirname(__DIR__);
    $dir = rtrim($base, DIRECTORY_SEPARATOR).DIRECTORY_SEPARATOR.'uploads'.DIRECTORY_SEPARATOR.'avatars';
    if (!is_dir($dir)) @mkdir($dir, 0775, true);
    $safe = preg_replace('/[^a-zA-Z0-9._-]/','_', basename($name));
    $fname = 'avatar_'.$uid.'_'.date('YmdHis').'.'.$ext;
    $dest = $dir.DIRECTORY_SEPARATOR.$fname;
    if (!@move_uploaded_file($_FILES['file']['tmp_name'], $dest)) ApiResponse::error('Failed to save file');
    $rel = 'uploads/avatars/'.$fname;
    $st = $db->prepare('UPDATE users SET avatar = :a, updated_at = NOW() WHERE id = :id');
    $st->execute([':a'=>$rel, ':id'=>$uid]);
    refreshSessionUser($db, $uid);
    ApiResponse::success(['avatar'=>$rel], 'Avatar updated');
  }catch(Throwable $e){ ApiResponse::error('Failed to upload avatar: '.$e->getMessage(), 500); }
}
