<?php
/**
 * Staff Loans API
 * - Employees apply for loans
 * - HR (Admin/HR Head) approves or rejects
 * - On approval, monthly schedule is generated
 * - Monthly installments can be serviced manually, and are also auto-serviced when payroll is marked paid
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { ApiResponse::success(null, 'OK'); }

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$database = new Database();
$db = $database->getConnection();

function ensureLoanTables(PDO $db){
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS employee_loans (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      employee_id INT NOT NULL,
      loan_type_id INT NULL,
      principal DECIMAL(12,2) NOT NULL,
      interest_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
      interest_type ENUM('flat','reducing') NOT NULL DEFAULT 'flat',
      term_months INT NOT NULL,
      start_date DATE NOT NULL,
      total_payable DECIMAL(12,2) NULL,
      monthly_installment DECIMAL(12,2) NULL,
      balance_outstanding DECIMAL(12,2) NULL,
      status ENUM('pending','approved','active','completed','rejected','cancelled') NOT NULL DEFAULT 'pending',
      approved_by INT NULL,
      approved_at DATETIME NULL,
      created_by INT NOT NULL,
      created_at DATETIME NOT NULL,
      updated_at DATETIME NULL,
      INDEX idx_company (company_id), INDEX idx_employee (employee_id), INDEX idx_status (status)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    $db->exec("CREATE TABLE IF NOT EXISTS loan_schedules (
      id INT AUTO_INCREMENT PRIMARY KEY,
      loan_id INT NOT NULL,
      due_date DATE NOT NULL,
      amount_due DECIMAL(12,2) NOT NULL,
      amount_paid DECIMAL(12,2) NULL,
      paid_at DATETIME NULL,
      payroll_id INT NULL,
      status ENUM('due','partial','paid','skipped') NOT NULL DEFAULT 'due',
      INDEX idx_loan (loan_id), INDEX idx_due (due_date), INDEX idx_status (status)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
    // Best-effort add loan_type_id to older installs
    try { $db->exec("ALTER TABLE employee_loans ADD COLUMN loan_type_id INT NULL"); } catch (Throwable $e) { /* ignore */ }
  } catch (Throwable $e) { /* ignore */ }
}

function ensureLoanTypesTable(PDO $db){
  try {
    $db->exec("CREATE TABLE IF NOT EXISTS loan_types (
      id INT AUTO_INCREMENT PRIMARY KEY,
      company_id INT NOT NULL,
      name VARCHAR(100) NOT NULL,
      code VARCHAR(50) NULL,
      default_term_months INT NOT NULL,
      default_interest_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
      interest_type ENUM('flat','reducing') NOT NULL DEFAULT 'flat',
      status ENUM('active','inactive') NOT NULL DEFAULT 'active',
      created_at DATETIME NULL,
      updated_at DATETIME NULL,
      UNIQUE KEY uq_company_name (company_id, name),
      INDEX idx_company (company_id),
      INDEX idx_status (status)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  } catch (Throwable $e) { /* ignore */ }
}

function mustHrOrAdmin(array $u){ if (!in_array($u['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions'); }
function mustApprover(array $u){ if (!in_array($u['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions'); }

function roles_level_map(){
  return [
    'employee'=>1,'manager'=>2,'hr_officer'=>3,'hr_head'=>4,'admin'=>5,'super_admin'=>6
  ];
}

function getEmployeeRoleSlug(PDO $db, int $employeeId): string {
  try {
    $q = $db->prepare("SELECT COALESCE(r.slug,'employee') AS role_slug, u.id AS user_id
                        FROM employees e
                        LEFT JOIN users u ON u.id = e.user_id
                        LEFT JOIN user_roles ur ON ur.user_id = u.id
                        LEFT JOIN roles r ON r.id = ur.role_id
                        WHERE e.id = :id LIMIT 1");
    $q->execute([':id'=>$employeeId]);
    $row = $q->fetch();
    return $row && !empty($row['role_slug']) ? $row['role_slug'] : 'employee';
  } catch (Throwable $e) { return 'employee'; }
}

function getEmployeeUserId(PDO $db, int $employeeId): ?int {
  try { $q=$db->prepare('SELECT user_id FROM employees WHERE id = :id'); $q->execute([':id'=>$employeeId]); $v=$q->fetchColumn(); return $v? (int)$v : null; } catch (Throwable $e){ return null; }
}

function requiredApproverRoleForApplicant(string $applicantRole): string {
  switch ($applicantRole) {
    case 'hr_officer': return 'hr_head';
    case 'hr_head': return 'admin';
    case 'admin': return 'super_admin';
    default: return 'hr_officer'; // employee/manager
  }
}

switch ($method) {
  case 'GET':
    if ($action === 'list') listLoans($db);
    elseif ($action === 'show') showLoan($db, $id);
    elseif ($action === 'list_types') listLoanTypes($db);
    elseif ($action === 'get_type') getLoanType($db, $id);
    else ApiResponse::error('Unknown action', 400);
    break;
  case 'POST':
    if ($action === 'apply') applyLoan($db);
    elseif ($action === 'approve') approveLoan($db, $id);
    elseif ($action === 'reject') rejectLoan($db, $id);
    elseif ($action === 'service_month') serviceLoanMonth($db);
    elseif ($action === 'create_type') createLoanType($db);
    elseif ($action === 'update_type') updateLoanType($db, $id);
    elseif ($action === 'delete_type') deleteLoanType($db, $id);
    else ApiResponse::error('Unknown action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function listLoans(PDO $db){
  ensureLoanTables($db);
  ensureLoanTypesTable($db);
  $u = getCurrentUser();
  $userId = isset($u['id']) ? (int)$u['id'] : 0;
  $companyId = isset($u['company_id']) ? (int)$u['company_id'] : 0;
  $role = isset($u['role_slug']) ? strtolower($u['role_slug']) : '';
  $isApplicant = in_array($role, ['employee','manager']);
  $params = [];
  // Resilient query without depending on roles tables
  $sql = "SELECT l.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.employee_number,
                 'employee' AS applicant_role,
                 t.name AS loan_type_name, t.code AS loan_type_code
          FROM employee_loans l 
          JOIN employees e ON e.id = l.employee_id
          LEFT JOIN loan_types t ON t.id = l.loan_type_id";
  if ($isApplicant) {
    $sql .= ' WHERE l.employee_id = :eid';
    $params[':eid'] = $userId > 0 ? getEmployeeId($db, $userId) : -1;
  } else {
    // Default to company scope for HR/Admin or unknown role
    $sql .= ' WHERE l.company_id = :cid';
    $params[':cid'] = $companyId > 0 ? $companyId : -1;
  }
  $sql .= ' ORDER BY l.created_at DESC';
  try {
    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k,$v); }
    $st->execute();
    $rows = $st->fetchAll();
    ApiResponse::success($rows);
  } catch (Throwable $e) {
    ApiResponse::error('Failed to load loans: '.$e->getMessage(), 500);
  }
}

function showLoan(PDO $db, int $id){
  ensureLoanTables($db);
  ensureLoanTypesTable($db);
  if ($id<=0) ApiResponse::error('id required');
  $u = getCurrentUser();
  $st = $db->prepare("SELECT l.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name, e.employee_number,
                             t.name AS loan_type_name, t.code AS loan_type_code
                      FROM employee_loans l 
                      JOIN employees e ON e.id = l.employee_id
                      LEFT JOIN loan_types t ON t.id = l.loan_type_id
                      WHERE l.id = :id");
  $st->execute([':id'=>$id]);
  if ($st->rowCount()===0) ApiResponse::notFound('Loan not found');
  $loan = $st->fetch();
  // Scope check: employee can only see own loan; HR must be same company
  if (in_array($u['role_slug'], ['employee','manager'])) {
    $eid = getEmployeeId($db, $u['id']); if ((int)$loan['employee_id'] !== (int)$eid) ApiResponse::forbidden('Not allowed');
  } else if ((int)$loan['company_id'] !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  // Load schedule
  $sc = $db->prepare('SELECT * FROM loan_schedules WHERE loan_id = :id ORDER BY due_date ASC');
  $sc->execute([':id'=>$id]);
  $loan['schedule'] = $sc->fetchAll();
  ApiResponse::success($loan);
}

function applyLoan(PDO $db){
  ensureLoanTables($db);
  ensureLoanTypesTable($db);
  $u = getCurrentUser();
  // Employees or HR can file application
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $employeeId = isset($in['employee_id']) ? (int)$in['employee_id'] : 0;
  if (in_array($u['role_slug'], ['employee','manager'])) { $employeeId = getEmployeeId($db, $u['id']); }
  else if (!in_array($u['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) { ApiResponse::forbidden('Insufficient permissions'); }
  // HR roles can apply for themselves without specifying employee_id
  if ($employeeId<=0 && in_array($u['role_slug'], ['hr_officer','hr_head'])) {
    $employeeId = getEmployeeId($db, $u['id']);
  }
  if ($employeeId<=0) ApiResponse::error('employee_id required');
  // Ownership
  $chk = $db->prepare('SELECT company_id FROM employees WHERE id = :id'); $chk->execute([':id'=>$employeeId]); $cid=$chk->fetchColumn();
  if (!$cid || (int)$cid !== (int)$u['company_id']) ApiResponse::forbidden('Employee not in your company');
  $principal = (float)($in['principal'] ?? 0);
  $term = (int)($in['term_months'] ?? 0);
  $rate = (float)($in['interest_rate'] ?? 0);
  $itype = in_array(($in['interest_type'] ?? 'flat'), ['flat','reducing']) ? $in['interest_type'] : 'flat';
  $loanTypeId = isset($in['loan_type_id']) ? (int)$in['loan_type_id'] : 0;
  if ($loanTypeId>0){
    // Load type to override defaults if missing or zero
    $tq = $db->prepare("SELECT * FROM loan_types WHERE id = :id AND company_id = :cid AND status = 'active'");
    $tq->execute([':id'=>$loanTypeId, ':cid'=>$u['company_id']]);
    $t = $tq->fetch();
    if ($t){
      if ($term<=0) $term = (int)$t['default_term_months'];
      if ($rate<=0) $rate = (float)$t['default_interest_rate'];
      if (!isset($in['interest_type']) || $in['interest_type']==='') $itype = $t['interest_type'];
    } else {
      $loanTypeId = 0; // invalid type for this company
    }
  }
  $start = $in['start_date'] ?? date('Y-m-d', strtotime('first day of next month'));
  if ($principal<=0 || $term<=0) ApiResponse::error('principal and term_months required');
  $ins = $db->prepare("INSERT INTO employee_loans (company_id, employee_id, loan_type_id, principal, interest_rate, interest_type, term_months, start_date, status, created_by, created_at)
                       VALUES (:cid,:eid,:lt,:p,:r,:t,:m,:s,'pending',:uid,NOW())");
  $ins->execute([':cid'=>$u['company_id'],':eid'=>$employeeId, ':lt'=>$loanTypeId?:null, ':p'=>$principal,':r'=>$rate,':t'=>$itype,':m'=>$term,':s'=>$start,':uid'=>$u['id']]);
  ApiResponse::success(['id'=>$db->lastInsertId()], 'Loan application submitted');
}

function approveLoan(PDO $db, int $id){
  ensureLoanTables($db);
  $u = getCurrentUser();
  mustApprover($u);
  if ($id<=0) ApiResponse::error('id required');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  // Fetch
  $st = $db->prepare('SELECT * FROM employee_loans WHERE id = :id'); $st->execute([':id'=>$id]); if ($st->rowCount()===0) ApiResponse::notFound('Loan not found');
  $loan = $st->fetch();
  if ((int)$loan['company_id'] !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  if ($loan['status'] !== 'pending') ApiResponse::error('Only pending loans can be approved');
  // Hierarchical rule & self-approval prevention
  $applicantRole = getEmployeeRoleSlug($db, (int)$loan['employee_id']);
  $minRole = requiredApproverRoleForApplicant($applicantRole);
  $levels = roles_level_map();
  if (($levels[$u['role_slug']] ?? 0) < ($levels[$minRole] ?? 999)) ApiResponse::forbidden('Insufficient approval level');
  $applicantUserId = getEmployeeUserId($db, (int)$loan['employee_id']);
  if ($applicantUserId && (int)$applicantUserId === (int)$u['id']) ApiResponse::forbidden('Cannot approve your own loan');
  // Compute schedule (flat only for now)
  $principal = (float)$loan['principal']; $rate = (float)$loan['interest_rate']; $months = (int)$loan['term_months'];
  $total = $principal + round($principal * ($rate/100.0), 2);
  $monthly = round($total / max(1,$months), 2);
  try {
    $db->beginTransaction();
    $u1 = $db->prepare('UPDATE employee_loans SET status = \'approved\', approved_by = :uid, approved_at = NOW(), total_payable = :tot, monthly_installment = :mon, balance_outstanding = :bal, updated_at = NOW() WHERE id = :id');
    $u1->execute([':uid'=>$u['id'], ':tot'=>$total, ':mon'=>$monthly, ':bal'=>$total, ':id'=>$id]);
    // Generate schedules
    $start = new DateTime($loan['start_date']);
    for ($i=0; $i<$months; $i++){
      $due = clone $start; $due->modify("+{$i} month"); $due->modify('last day of this month');
      $ins = $db->prepare('INSERT INTO loan_schedules (loan_id, due_date, amount_due, status) VALUES (:lid,:due,:amt,\'due\')');
      $ins->execute([':lid'=>$id, ':due'=>$due->format('Y-m-d'), ':amt'=>$monthly]);
    }
    $db->commit();
    ApiResponse::success(null,'Loan approved and schedule generated');
  } catch (Throwable $e){ if ($db->inTransaction()) $db->rollBack(); ApiResponse::error('Failed to approve: '.$e->getMessage(), 500);}  
}

function rejectLoan(PDO $db, int $id){
  ensureLoanTables($db);
  $u = getCurrentUser();
  mustApprover($u);
  if ($id<=0) ApiResponse::error('id required');
  // Check applicant and hierarchy
  $g = $db->prepare('SELECT company_id, employee_id FROM employee_loans WHERE id = :id');
  $g->execute([':id'=>$id]);
  if ($g->rowCount()===0) ApiResponse::notFound('Loan not found');
  $row = $g->fetch();
  if ((int)$row['company_id'] !== (int)$u['company_id']) ApiResponse::forbidden('Not allowed');
  $applicantRole = getEmployeeRoleSlug($db, (int)$row['employee_id']);
  $minRole = requiredApproverRoleForApplicant($applicantRole);
  $levels = roles_level_map();
  if (($levels[$u['role_slug']] ?? 0) < ($levels[$minRole] ?? 999)) ApiResponse::forbidden('Insufficient approval level');
  $applicantUserId = getEmployeeUserId($db, (int)$row['employee_id']);
  if ($applicantUserId && (int)$applicantUserId === (int)$u['id']) ApiResponse::forbidden('Cannot reject your own loan');
  $st = $db->prepare('UPDATE employee_loans SET status = \'rejected\', approved_by = :uid, approved_at = NOW(), updated_at = NOW() WHERE id = :id AND company_id = :cid AND status = \'pending\'');
  $st->execute([':uid'=>$u['id'], ':id'=>$id, ':cid'=>$u['company_id']]);
  if ($st->rowCount()===0) ApiResponse::error('Loan not found or not pending');
  ApiResponse::success(null,'Loan rejected');
}

function serviceLoanMonth(PDO $db){
  ensureLoanTables($db);
  mustHrOrAdmin(getCurrentUser());
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $loanId = (int)($in['loan_id'] ?? 0);
  $month = (string)($in['month'] ?? ''); // YYYY-MM
  if ($loanId<=0 || !preg_match('/^\d{4}-\d{2}$/',$month)) ApiResponse::error('loan_id and month (YYYY-MM) required');
  $start = $month.'-01'; $end = date('Y-m-t', strtotime($start));
  try {
    $db->beginTransaction();
    // Sum due
    $sumQ = $db->prepare('SELECT COALESCE(SUM(amount_due - COALESCE(amount_paid,0)),0) FROM loan_schedules WHERE loan_id = :id AND status IN (\'due\',\'partial\') AND due_date BETWEEN :s AND :e');
    $sumQ->execute([':id'=>$loanId, ':s'=>$start, ':e'=>$end]);
    $due = (float)$sumQ->fetchColumn();
    if ($due <= 0) { $db->commit(); ApiResponse::success(['serviced'=>0],'Nothing due for month'); }
    // Mark as paid in full for the month
    $upd = $db->prepare('UPDATE loan_schedules SET amount_paid = amount_due, status = \'paid\', paid_at = NOW() WHERE loan_id = :id AND status IN (\'due\',\'partial\') AND due_date BETWEEN :s AND :e');
    $upd->execute([':id'=>$loanId, ':s'=>$start, ':e'=>$end]);
    // Update loan balance and status
    refreshLoanBalance($db, $loanId);
    $db->commit();
    ApiResponse::success(['serviced_amount'=>$due],'Serviced');
  } catch (Throwable $e){ if ($db->inTransaction()) $db->rollBack(); ApiResponse::error('Failed to service: '.$e->getMessage(),500);}  
}

function refreshLoanBalance(PDO $db, int $loanId){
  try {
    $sum = $db->prepare('SELECT l.total_payable, COALESCE(SUM(ls.amount_paid),0) AS paid FROM employee_loans l LEFT JOIN loan_schedules ls ON ls.loan_id = l.id WHERE l.id = :id');
    $sum->execute([':id'=>$loanId]);
    $row = $sum->fetch(); if (!$row) return;
    $bal = max(0.0, (float)$row['total_payable'] - (float)$row['paid']);
    $status = $bal <= 0.009 ? 'completed' : 'approved';
    $u = $db->prepare('UPDATE employee_loans SET balance_outstanding = :bal, status = :st, updated_at = NOW() WHERE id = :id');
    $u->execute([':bal'=>$bal, ':st'=>$status, ':id'=>$loanId]);
  } catch (Throwable $e) { /* ignore */ }
}

function getEmployeeId(PDO $db, int $userId){ $q=$db->prepare('SELECT id FROM employees WHERE user_id = :uid'); $q->execute([':uid'=>$userId]); return (int)($q->fetchColumn() ?: 0); }

// ===== Loan Types CRUD =====
function listLoanTypes(PDO $db){
  ensureLoanTypesTable($db);
  $u = getCurrentUser();
  $st = $db->prepare('SELECT * FROM loan_types WHERE company_id = :cid ORDER BY status DESC, name ASC');
  $st->execute([':cid'=>$u['company_id']]);
  ApiResponse::success($st->fetchAll());
}

function getLoanType(PDO $db, int $id){
  ensureLoanTypesTable($db);
  if ($id<=0) ApiResponse::error('id required');
  $u = getCurrentUser();
  $st = $db->prepare('SELECT * FROM loan_types WHERE id = :id AND company_id = :cid');
  $st->execute([':id'=>$id, ':cid'=>$u['company_id']]);
  if ($st->rowCount()===0) ApiResponse::notFound('Type not found');
  ApiResponse::success($st->fetch());
}

function createLoanType(PDO $db){
  ensureLoanTypesTable($db);
  mustHrOrAdmin(getCurrentUser());
  $u = getCurrentUser();
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  if (!isset($in['name']) || trim($in['name'])==='') ApiResponse::error('name required');
  $ins = $db->prepare('INSERT INTO loan_types (company_id, name, code, default_term_months, default_interest_rate, interest_type, status, created_at) VALUES (:cid,:name,:code,:term,:rate,:itype,:status,NOW())');
  $ins->execute([
    ':cid'=>$u['company_id'],
    ':name'=>$in['name'],
    ':code'=>$in['code'] ?? null,
    ':term'=> (int)($in['default_term_months'] ?? 0),
    ':rate'=> (float)($in['default_interest_rate'] ?? 0),
    ':itype'=> in_array(($in['interest_type'] ?? 'flat'), ['flat','reducing']) ? $in['interest_type'] : 'flat',
    ':status'=> in_array(($in['status'] ?? 'active'), ['active','inactive']) ? $in['status'] : 'active',
  ]);
  ApiResponse::success(['id'=>$db->lastInsertId()], 'Type created');
}

function updateLoanType(PDO $db, int $id){
  ensureLoanTypesTable($db);
  mustHrOrAdmin(getCurrentUser());
  if ($id<=0) ApiResponse::error('id required');
  $u = getCurrentUser();
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $allowed = ['name','code','default_term_months','default_interest_rate','interest_type','status'];
  $set = [];$params=[':id'=>$id, ':cid'=>$u['company_id']];
  foreach ($allowed as $f){ if (array_key_exists($f,$in)){ $set[] = "$f = :$f"; $params[":$f"] = $in[$f]; } }
  if (!$set) ApiResponse::error('No fields');
  $sql = 'UPDATE loan_types SET '.implode(', ',$set).', updated_at = NOW() WHERE id = :id AND company_id = :cid';
  $st = $db->prepare($sql);
  foreach ($params as $k=>$v){ $st->bindValue($k,$v); }
  $st->execute();
  ApiResponse::success(null,'Type updated');
}

function deleteLoanType(PDO $db, int $id){
  ensureLoanTypesTable($db);
  mustHrOrAdmin(getCurrentUser());
  if ($id<=0) ApiResponse::error('id required');
  $u = getCurrentUser();
  $st = $db->prepare("UPDATE loan_types SET status='inactive', updated_at = NOW() WHERE id = :id AND company_id = :cid");
  $st->execute([':id'=>$id, ':cid'=>$u['company_id']]);
  ApiResponse::success(null,'Type deactivated');
}
