<?php
/**
 * Leave Management API with Approval Workflow
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

// Ensure grade-based allocation helpers are available to all HTTP methods
require_once __DIR__ . '/_leave_allocations.inc.php';
// Common leave helpers (get/update/cancel request)
require_once __DIR__ . '/_leave_common.inc.php';
// Workflow helpers (multi-stage approvals)
require_once __DIR__ . '/_leave_workflow.inc.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

// Ensure leave_requests has reliever column (best-effort)
if (!function_exists('ensureRelieverColumn')) {
    function ensureRelieverColumn(PDO $db){
        try { $db->exec("ALTER TABLE leave_requests ADD COLUMN reliever_employee_id INT NULL"); } catch (Throwable $e) { /* ignore if exists */ }
    }
}

// ===== Holiday helpers (avoid circular include) ===== //
if (!function_exists('ensureHolidaysTable')) {
    function ensureHolidaysTable(PDO $db){
        try {
            $db->exec("CREATE TABLE IF NOT EXISTS holidays (
                id INT AUTO_INCREMENT PRIMARY KEY,
                company_id INT NOT NULL,
                holiday_date DATE NOT NULL,
                name VARCHAR(150) NOT NULL,
                created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                UNIQUE KEY uniq_company_date (company_id, holiday_date),
                KEY idx_company (company_id),
                KEY idx_date (holiday_date)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        } catch (Throwable $e) { /* ignore */ }
    }
}

if (!function_exists('listHolidayDates')) {
    function listHolidayDates(PDO $db, int $companyId, string $from, string $to): array {
        ensureHolidaysTable($db);
        try {
            $q = $db->prepare("SELECT holiday_date FROM holidays WHERE company_id = :cid AND holiday_date BETWEEN :f AND :t");
            $q->bindValue(':cid', $companyId, PDO::PARAM_INT);
            $q->bindValue(':f', $from);
            $q->bindValue(':t', $to);
            $q->execute();
            $rows = $q->fetchAll();
            $set = [];
            foreach ($rows as $r) { $set[date('Y-m-d', strtotime($r['holiday_date']))] = true; }
            return $set; // associative set for O(1) lookup
        } catch (Throwable $e) { return []; }
    }
}

if (!function_exists('countWorkingDays')) {
    function countWorkingDays(PDO $db, int $companyId, string $start, string $end): int {
        $startTs = strtotime($start);
        $endTs = strtotime($end);
        if ($startTs === false || $endTs === false) return 0;
        if ($endTs < $startTs) { [$startTs, $endTs] = [$endTs, $startTs]; }
        $from = date('Y-m-d', $startTs);
        $to = date('Y-m-d', $endTs);
        $holidays = listHolidayDates($db, $companyId, $from, $to);
        $count = 0;
        for ($ts = $startTs; $ts <= $endTs; $ts += 86400) {
            $dow = (int)date('N', $ts); // 1=Mon..7=Sun
            if ($dow >= 6) continue; // skip Sat/Sun
            $key = date('Y-m-d', $ts);
            if (isset($holidays[$key])) continue; // skip holiday
            $count++;
        }
        return $count;
    }
}

switch ($method) {
    case 'GET':
        if ($action === 'types') {
            getLeaveTypes($db);
        } elseif ($action === 'balance') {
            getLeaveBalance($db);
        } elseif ($action === 'allocations') {
            getLeaveAllocations($db);
        } elseif ($id) {
            getLeaveRequest($db, $id);
        } else {
            getLeaveRequests($db);
        }
        break;
    
    case 'POST':
        if ($action === 'request') {
            createLeaveRequest($db);
        } elseif ($action === 'approve') {
            approveLeaveRequest($db, $id);
        } elseif ($action === 'reject') {
            rejectLeaveRequest($db, $id);
        } elseif ($action === 'allocations_save') {
            saveLeaveAllocations($db);
        }
        break;
    
    case 'PUT':
        updateLeaveRequest($db, $id);
        break;
    
    case 'DELETE':
        cancelLeaveRequest($db, $id);
        break;
    
    default:
        ApiResponse::error('Method not allowed', 405);
}

function getLeaveTypes($db) {
    $user = getCurrentUser();
    // Ensure leave_types table exists
    if (function_exists('ensureLeaveTypesTable')) { ensureLeaveTypesTable($db); }
    
    $query = "SELECT * FROM leave_types WHERE company_id = :company_id ORDER BY name";
    $stmt = $db->prepare($query);
    $stmt->bindParam(':company_id', $user['company_id']);
    $stmt->execute();
    
    $leaveTypes = $stmt->fetchAll();
    ApiResponse::success($leaveTypes);
}

function getLeaveBalance($db) {
    $user = getCurrentUser();
    $employeeId = $_GET['employee_id'] ?? null;
    
    // If no employee_id provided, get current user's employee record
    if (!$employeeId) {
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $emp = $empStmt->fetch();
            $employeeId = $emp['id'];
        } else {
            ApiResponse::error('Employee record not found');
        }
    }
    
    // Ensure required schema before queries
    if (function_exists('ensureLeaveTypesTable')) { ensureLeaveTypesTable($db); }
    if (function_exists('ensureAllocationsTable')) { ensureAllocationsTable($db); }
    if (function_exists('ensureEmployeesCurrentGradeColumn')) { ensureEmployeesCurrentGradeColumn($db); }

    // Determine employee grade for grade-based allocations
    $grade = function_exists('lookupEmployeeGrade') ? lookupEmployeeGrade($db, (int)$employeeId) : null;

    $query = "SELECT 
                lt.id,
                lt.name,
                lt.code,
                COALESCE(la.days_per_year, lt.days_per_year) AS days_per_year,
                COALESCE(SUM(CASE WHEN lr.status = 'approved' THEN lr.days_requested ELSE 0 END), 0) AS used_days,
                COALESCE(SUM(CASE WHEN lr.status = 'pending' THEN lr.days_requested ELSE 0 END), 0) AS pending_days,
                (COALESCE(la.days_per_year, lt.days_per_year) - COALESCE(SUM(CASE WHEN lr.status = 'approved' THEN lr.days_requested ELSE 0 END), 0)) AS remaining_days,
                (COALESCE(la.days_per_year, lt.days_per_year)
                  - COALESCE(SUM(CASE WHEN lr.status = 'approved' THEN lr.days_requested ELSE 0 END), 0)
                  - COALESCE(SUM(CASE WHEN lr.status = 'pending' THEN lr.days_requested ELSE 0 END), 0)
                ) AS available_days
              FROM leave_types lt
              LEFT JOIN leave_allocations la 
                ON la.company_id = :company_id 
               AND la.leave_type_id = lt.id 
               AND (:grade IS NOT NULL AND la.grade = :grade)
              LEFT JOIN leave_requests lr ON lt.id = lr.leave_type_id 
                AND lr.employee_id = :employee_id 
                AND YEAR(lr.start_date) = YEAR(CURDATE())
              WHERE lt.company_id = :company_id
              GROUP BY lt.id, lt.name, lt.code, COALESCE(la.days_per_year, lt.days_per_year)
              ORDER BY lt.name";

    $stmt = $db->prepare($query);
    $stmt->bindValue(':employee_id', (int)$employeeId, PDO::PARAM_INT);
    $stmt->bindValue(':company_id', (int)$user['company_id'], PDO::PARAM_INT);
    // Bind grade twice for the join condition
    if ($grade !== null) {
        $stmt->bindValue(':grade', $grade);
    } else {
        $stmt->bindValue(':grade', null, PDO::PARAM_NULL);
    }
    $stmt->execute();

    $balances = $stmt->fetchAll();
    ApiResponse::success($balances);
}

function getLeaveRequests($db) {
    $user = getCurrentUser();
    $role = $user['role_slug'];
    ensureRelieverColumn($db);
    
    // Base query
    $query = "SELECT 
                lr.*,
                lt.name as leave_type_name,
                lt.color as leave_type_color,
                CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                e.employee_number,
                d.name as department_name,
                CONCAT(approver.first_name, ' ', approver.last_name) as approved_by_name,
                CONCAT(rel.first_name, ' ', rel.last_name) AS reliever_name
              FROM leave_requests lr
              JOIN employees e ON lr.employee_id = e.id
              JOIN leave_types lt ON lr.leave_type_id = lt.id
              LEFT JOIN departments d ON e.department_id = d.id
              LEFT JOIN users u_approver ON lr.approved_by = u_approver.id
              LEFT JOIN employees approver ON u_approver.employee_id = approver.id
              LEFT JOIN employees rel ON rel.id = lr.reliever_employee_id
              WHERE e.company_id = :company_id";
    
    $params = [':company_id' => $user['company_id']];
    
    // Apply role-based filtering
    if ($role === 'employee') {
        // Employees can only see their own requests
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $emp = $empStmt->fetch();
            $query .= " AND lr.employee_id = :employee_id";
            $params[':employee_id'] = $emp['id'];
        } else {
            ApiResponse::success([]); // No employee record found
        }
    } elseif ($role === 'manager') {
        // Managers can see requests from their direct reports
        $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
        $empStmt = $db->prepare($empQuery);
        $empStmt->bindParam(':user_id', $user['id']);
        $empStmt->execute();
        
        if ($empStmt->rowCount() > 0) {
            $manager = $empStmt->fetch();
            $query .= " AND (e.manager_id = :manager_id OR lr.employee_id = :employee_id)";
            $params[':manager_id'] = $manager['id'];
            $params[':employee_id'] = $manager['id'];
        }
    }
    // HR roles can see all requests (no additional filtering)
    
    // Apply additional filters
    if (isset($_GET['status']) && !empty($_GET['status'])) {
        $query .= " AND lr.status = :status";
        $params[':status'] = $_GET['status'];
    }
    
    if (isset($_GET['employee_id']) && !empty($_GET['employee_id'])) {
        $query .= " AND lr.employee_id = :filter_employee_id";
        $params[':filter_employee_id'] = $_GET['employee_id'];
    }

    // New: filter by leave type
    if (isset($_GET['leave_type_id']) && $_GET['leave_type_id'] !== '') {
        $query .= " AND lr.leave_type_id = :filter_leave_type_id";
        $params[':filter_leave_type_id'] = (int)$_GET['leave_type_id'];
    }

    // New: date range overlap filter (from/to)
    if (!empty($_GET['from'])) {
        $query .= " AND lr.end_date >= :filter_from";
        $params[':filter_from'] = $_GET['from'];
    }
    if (!empty($_GET['to'])) {
        $query .= " AND lr.start_date <= :filter_to";
        $params[':filter_to'] = $_GET['to'];
    }

    // New: employee name search
    if (!empty($_GET['q'])) {
        $query .= " AND (e.first_name LIKE :filter_q OR e.last_name LIKE :filter_q OR CONCAT(e.first_name,' ',e.last_name) LIKE :filter_q)";
        $params[':filter_q'] = '%' . $_GET['q'] . '%';
    }
    
    $query .= " ORDER BY lr.created_at DESC";
    
    $stmt = $db->prepare($query);
    foreach ($params as $key => $value) {
        $stmt->bindValue($key, $value);
    }
    $stmt->execute();
    
    $requests = $stmt->fetchAll();

    // If viewing pending and user is an approver, only show items at their current stage
    if (isset($_GET['status']) && $_GET['status'] === 'pending' && canApprove('leave')) {
        ensureApprovalLogsTable($db);
        $filtered = [];
        foreach ($requests as $r) {
            $stageRole = getNextStageRole($db, (int)$r['id']);
            if ($stageRole === null) continue; // already finalized
            $ctx = getRequestContext($db, (int)$r['id']);
            $mgrUserId = $ctx['manager_user_id'] ?? null;
            if (canUserApproveStage($user, $stageRole, $mgrUserId)) {
                $filtered[] = $r;
            }
        }
        $requests = $filtered;
    }

    ApiResponse::success($requests);
}

function createLeaveRequest($db) {
    $user = getCurrentUser();
    ensureRelieverColumn($db);
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Get employee ID for current user
    $empQuery = "SELECT id FROM employees WHERE user_id = :user_id";
    $empStmt = $db->prepare($empQuery);
    $empStmt->bindParam(':user_id', $user['id']);
    $empStmt->execute();
    
    if ($empStmt->rowCount() === 0) {
        ApiResponse::error('Employee record not found');
    }
    
    $employee = $empStmt->fetch();
    $employeeId = $employee['id'];
    
    // Validate required fields
    $required = ['leave_type_id', 'start_date', 'end_date', 'reason'];
    foreach ($required as $field) {
        if (!isset($input[$field]) || empty($input[$field])) {
            ApiResponse::error("Field '$field' is required");
        }
    }
    
    // Calculate working days requested (excludes weekends & company holidays)
    $daysRequested = countWorkingDays($db, (int)$user['company_id'], $input['start_date'], $input['end_date']);
    if ($daysRequested <= 0) { ApiResponse::error('Selected date range contains no working days'); }
    
    // Ensure base tables exist before joins
    if (function_exists('ensureLeaveTypesTable')) { ensureLeaveTypesTable($db); }
    if (function_exists('ensureAllocationsTable')) { ensureAllocationsTable($db); }

    // Check leave balance with grade-based allocations
    $empGrade = function_exists('lookupEmployeeGrade') ? lookupEmployeeGrade($db, (int)$employeeId) : null;
    $balanceQuery = "SELECT 
                       COALESCE(la.days_per_year, lt.days_per_year) AS allowed_days,
                       COALESCE(SUM(CASE WHEN lr.status = 'approved' THEN lr.days_requested ELSE 0 END), 0) AS used_days,
                       COALESCE(SUM(CASE WHEN lr.status = 'pending' THEN lr.days_requested ELSE 0 END), 0) AS pending_days
                     FROM leave_types lt
                     LEFT JOIN leave_allocations la 
                       ON la.company_id = :company_id 
                      AND la.leave_type_id = lt.id 
                      AND (:grade IS NOT NULL AND la.grade = :grade)
                     LEFT JOIN leave_requests lr ON lt.id = lr.leave_type_id 
                       AND lr.employee_id = :employee_id 
                       AND YEAR(lr.start_date) = YEAR(:start_date)
                     WHERE lt.id = :leave_type_id AND lt.company_id = :company_id
                     GROUP BY lt.id, COALESCE(la.days_per_year, lt.days_per_year)";

    $balanceStmt = $db->prepare($balanceQuery);
    $balanceStmt->bindValue(':employee_id', (int)$employeeId, PDO::PARAM_INT);
    $balanceStmt->bindValue(':leave_type_id', (int)$input['leave_type_id'], PDO::PARAM_INT);
    $balanceStmt->bindValue(':start_date', $input['start_date']);
    $balanceStmt->bindValue(':company_id', (int)$user['company_id'], PDO::PARAM_INT);
    if ($empGrade !== null) {
        $balanceStmt->bindValue(':grade', $empGrade);
    } else {
        $balanceStmt->bindValue(':grade', null, PDO::PARAM_NULL);
    }
    $balanceStmt->execute();

    if ($balanceStmt->rowCount() > 0) {
        $balance = $balanceStmt->fetch();
        $allowedDays = (int)($balance['allowed_days'] ?? 0);
        $usedApproved = (int)($balance['used_days'] ?? 0);
        $usedPending = (int)($balance['pending_days'] ?? 0);
        $available = $allowedDays - $usedApproved - $usedPending;
        if ($daysRequested > $available) {
            ApiResponse::error("Insufficient leave balance. Available for new requests: $available day(s). Requested: $daysRequested");
        }
    }
    
    try {
        // Insert leave request
        // Validate optional reliever (same company)
        $relieverId = isset($input['reliever_employee_id']) ? (int)$input['reliever_employee_id'] : 0;
        if ($relieverId > 0) {
            $chkRel = $db->prepare('SELECT id FROM employees WHERE id = :rid AND company_id = :cid');
            $chkRel->execute([':rid'=>$relieverId, ':cid'=>$user['company_id']]);
            if ($chkRel->rowCount() === 0) { $relieverId = 0; }
        }

        $query = "INSERT INTO leave_requests (
                    employee_id, leave_type_id, start_date, end_date, days_requested,
                    reason, reliever_employee_id, status, created_at
                  ) VALUES (
                    :employee_id, :leave_type_id, :start_date, :end_date, :days_requested,
                    :reason, :reliever_id, 'pending', NOW()
                  )";
        
        $stmt = $db->prepare($query);
        $stmt->bindParam(':employee_id', $employeeId);
        $stmt->bindParam(':leave_type_id', $input['leave_type_id']);
        $stmt->bindParam(':start_date', $input['start_date']);
        $stmt->bindParam(':end_date', $input['end_date']);
        $stmt->bindParam(':days_requested', $daysRequested);
        $stmt->bindParam(':reason', $input['reason']);
        if ($relieverId > 0) { $stmt->bindValue(':reliever_id', $relieverId, PDO::PARAM_INT); }
        else { $stmt->bindValue(':reliever_id', null, PDO::PARAM_NULL); }
        
        $stmt->execute();
        $requestId = $db->lastInsertId();
        
        // Create notification for manager/HR and next-stage approvers
        createLeaveNotification($db, $requestId, 'leave_request_submitted');
        $firstStage = getNextStageRole($db, (int)$requestId);
        if ($firstStage !== null) {
            notifyNextStageApprovers($db, (int)$requestId, $firstStage);
        }
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'leave_request_created', ['request_id' => $requestId]);
        
        ApiResponse::success(['id' => $requestId], 'Leave request submitted successfully');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to create leave request: ' . $e->getMessage());
    }
}

function approveLeaveRequest($db, $id) {
    $user = getCurrentUser();
    ensureRelieverColumn($db);
    
    // Check if user can approve leave requests
    if (!canApprove('leave')) {
        ApiResponse::forbidden('Insufficient permissions to approve leave requests');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    try {
        // Verify request exists, is pending, and in same company
        $checkQuery = "SELECT lr.*, e.company_id, e.manager_id 
                       FROM leave_requests lr 
                       JOIN employees e ON lr.employee_id = e.id 
                       WHERE lr.id = :id AND lr.status = 'pending' AND e.company_id = :company_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id, PDO::PARAM_INT);
        $checkStmt->bindParam(':company_id', $user['company_id'], PDO::PARAM_INT);
        $checkStmt->execute();
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Leave request not found or already processed');
        }

        // Stage gating
        ensureApprovalLogsTable($db);
        $stageRole = getNextStageRole($db, (int)$id);
        if ($stageRole === null) {
            ApiResponse::error('Request already fully approved');
        }
        $ctx = getRequestContext($db, (int)$id);
        $mgrUserId = $ctx['manager_user_id'] ?? null;
        if (!canUserApproveStage($user, $stageRole, $mgrUserId)) {
            ApiResponse::forbidden('You are not the authorized approver for this stage');
        }

        // HR roles can override or set reliever before approval
        $canSetReliever = in_array(($user['role_slug'] ?? ''), ['super_admin','admin','hr_head'], true);
        if ($canSetReliever && isset($input['reliever_employee_id'])) {
            $rel = (int)$input['reliever_employee_id'];
            if ($rel > 0) {
                $cr = $db->prepare('SELECT id FROM employees WHERE id = :rid AND company_id = :cid');
                $cr->execute([':rid'=>$rel, ':cid'=>$user['company_id']]);
                if ($cr->rowCount() === 0) { $rel = 0; }
            }
            $upRel = $db->prepare('UPDATE leave_requests SET reliever_employee_id = :rid WHERE id = :id');
            if ($rel > 0) { $upRel->bindValue(':rid', $rel, PDO::PARAM_INT); }
            else { $upRel->bindValue(':rid', null, PDO::PARAM_NULL); }
            $upRel->bindValue(':id', (int)$id, PDO::PARAM_INT);
            $upRel->execute();
        }

        // Insert stage approval log
        $stageIndex = getApprovedStageCount($db, (int)$id) + 1;
        $log = $db->prepare("INSERT INTO leave_approval_logs (leave_request_id, stage, action, approver_user_id, notes, created_at) VALUES (:rid, :stage, 'approved', :uid, :notes, NOW())");
        $log->bindValue(':rid', (int)$id, PDO::PARAM_INT);
        $log->bindValue(':stage', (int)$stageIndex, PDO::PARAM_INT);
        $log->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $log->bindValue(':notes', $input['notes'] ?? null);
        $log->execute();

        // If last stage, finalize request
        $nextStage = getNextStageRole($db, (int)$id);
        if ($nextStage === null) {
            $query = "UPDATE leave_requests SET 
                        status = 'approved',
                        approved_by = :approved_by,
                        approved_at = NOW(),
                        notes = :notes,
                        updated_at = NOW()
                      WHERE id = :id";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
            $stmt->bindParam(':approved_by', $user['id'], PDO::PARAM_INT);
            $stmt->bindParam(':notes', $input['notes'] ?? null);
            $stmt->execute();

            // Create notification for employee on final approval
            createLeaveNotification($db, $id, 'leave_request_approved');
        } else {
            // Notify next-stage approvers
            notifyNextStageApprovers($db, (int)$id, $nextStage);
        }
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'leave_request_stage_approved', ['request_id' => (int)$id, 'stage_role' => $stageRole]);
        
        ApiResponse::success(null, $nextStage === null ? 'Leave request approved successfully' : 'Stage approved');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to approve leave request: ' . $e->getMessage());
    }
}

function rejectLeaveRequest($db, $id) {
    $user = getCurrentUser();
    
    // Check if user can approve leave requests
    if (!canApprove('leave')) {
        ApiResponse::forbidden('Insufficient permissions to reject leave requests');
    }
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!isset($input['rejection_reason']) || empty($input['rejection_reason'])) {
        ApiResponse::error('Rejection reason is required');
    }
    
    try {
        // Verify request exists, is pending, and in same company
        $checkQuery = "SELECT lr.*, e.company_id 
                       FROM leave_requests lr 
                       JOIN employees e ON lr.employee_id = e.id 
                       WHERE lr.id = :id AND lr.status = 'pending' AND e.company_id = :company_id";
        $checkStmt = $db->prepare($checkQuery);
        $checkStmt->bindParam(':id', $id, PDO::PARAM_INT);
        $checkStmt->bindParam(':company_id', $user['company_id'], PDO::PARAM_INT);
        $checkStmt->execute();
        if ($checkStmt->rowCount() === 0) {
            ApiResponse::error('Leave request not found or already processed');
        }

        // Stage gating
        ensureApprovalLogsTable($db);
        $stageIndex = getApprovedStageCount($db, (int)$id) + 1;
        $log = $db->prepare("INSERT INTO leave_approval_logs (leave_request_id, stage, action, approver_user_id, notes, created_at) VALUES (:rid, :stage, 'rejected', :uid, :notes, NOW())");
        $log->bindValue(':rid', (int)$id, PDO::PARAM_INT);
        $log->bindValue(':stage', (int)$stageIndex, PDO::PARAM_INT);
        $log->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $log->bindValue(':notes', $input['rejection_reason']);
        $log->execute();

        // Update request status to rejected
        $query = "UPDATE leave_requests SET 
                    status = 'rejected',
                    approved_by = :approved_by,
                    approved_at = NOW(),
                    rejection_reason = :rejection_reason,
                    updated_at = NOW()
                  WHERE id = :id";
        $stmt = $db->prepare($query);
        $stmt->bindParam(':id', $id, PDO::PARAM_INT);
        $stmt->bindParam(':approved_by', $user['id'], PDO::PARAM_INT);
        $stmt->bindParam(':rejection_reason', $input['rejection_reason']);
        $stmt->execute();
        
        // Create notification for employee
        createLeaveNotification($db, $id, 'leave_request_rejected');
        
        // Log activity
        $auth = new Auth();
        $auth->logActivity($user['id'], 'leave_request_rejected', ['request_id' => (int)$id]);
        
        ApiResponse::success(null, 'Leave request rejected');
        
    } catch (Exception $e) {
        ApiResponse::error('Failed to reject leave request: ' . $e->getMessage());
    }
}

function createLeaveNotification($db, $requestId, $type) {
    // Ensure notifications table exists
    if (function_exists('ensureNotificationsTable')) { ensureNotificationsTable($db); }
    // Get request details
    $query = "SELECT lr.*, e.user_id as employee_user_id, e.manager_id,
                     CONCAT(e.first_name, ' ', e.last_name) as employee_name,
                     lt.name as leave_type_name
              FROM leave_requests lr
              JOIN employees e ON lr.employee_id = e.id
              JOIN leave_types lt ON lr.leave_type_id = lt.id
              WHERE lr.id = :request_id";
    
    $stmt = $db->prepare($query);
    $stmt->bindParam(':request_id', $requestId);
    $stmt->execute();
    
    if ($stmt->rowCount() > 0) {
        $request = $stmt->fetch();
        
        $notifications = [];
        
        switch ($type) {
            case 'leave_request_submitted':
                // Notify manager and HR
                if ($request['manager_id']) {
                    $managerQuery = "SELECT user_id FROM employees WHERE id = :manager_id";
                    $managerStmt = $db->prepare($managerQuery);
                    $managerStmt->bindParam(':manager_id', $request['manager_id']);
                    $managerStmt->execute();
                    
                    if ($managerStmt->rowCount() > 0) {
                        $manager = $managerStmt->fetch();
                        $notifications[] = [
                            'user_id' => $manager['user_id'],
                            'title' => 'New Leave Request',
                            'content' => "{$request['employee_name']} has requested {$request['days_requested']} days of {$request['leave_type_name']} leave."
                        ];
                    }
                }
                break;
                
            case 'leave_request_approved':
            case 'leave_request_rejected':
                // Notify employee
                if ($request['employee_user_id']) {
                    $status = $type === 'leave_request_approved' ? 'approved' : 'rejected';
                    $notifications[] = [
                        'user_id' => $request['employee_user_id'],
                        'title' => 'Leave Request ' . ucfirst($status),
                        'content' => "Your {$request['leave_type_name']} leave request has been {$status}."
                    ];
                }
                break;
        }
        
        // Insert notifications
        foreach ($notifications as $notification) {
            $notifQuery = "INSERT INTO notifications (user_id, type, title, content, created_at) 
                           VALUES (:user_id, :type, :title, :content, NOW())";
            $notifStmt = $db->prepare($notifQuery);
            $notifStmt->bindParam(':user_id', $notification['user_id']);
            $notifStmt->bindParam(':type', $type);
            $notifStmt->bindParam(':title', $notification['title']);
            $notifStmt->bindParam(':content', $notification['content']);
            $notifStmt->execute();
        }
    }
}
?>
