<?php
/**
 * Holidays Management API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';

$db = (new Database())->getConnection();

function ensureHolidaysTable(PDO $db){
    try {
        $db->exec("CREATE TABLE IF NOT EXISTS holidays (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            holiday_date DATE NOT NULL,
            name VARCHAR(150) NOT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            UNIQUE KEY uniq_company_date (company_id, holiday_date),
            KEY idx_company (company_id),
            KEY idx_date (holiday_date)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    } catch (Throwable $e) { /* ignore */ }
}

function mustBeHrRole($user){
    if (!in_array($user['role_slug'], ['super_admin','admin','hr_head','hr_officer'])) {
        ApiResponse::forbidden('Insufficient permissions');
    }
}

switch ($method) {
    case 'GET':
        if ($action === 'list') {
            listHolidays($db);
        } elseif ($action === 'range') {
            listHolidaysInRange($db);
        } else {
            listHolidays($db);
        }
        break;

    case 'POST':
        if ($action === 'save') {
            saveHoliday($db);
        } elseif ($action === 'delete') {
            deleteHoliday($db);
        } else {
            ApiResponse::error('Unknown action');
        }
        break;

    default:
        ApiResponse::error('Method not allowed', 405);
}

function listHolidays(PDO $db){
    $u = getCurrentUser();
    ensureHolidaysTable($db);
    $year = isset($_GET['year']) ? (int)$_GET['year'] : (int)date('Y');
    try {
        $q = $db->prepare("SELECT id, holiday_date, name FROM holidays WHERE company_id = :cid AND YEAR(holiday_date) = :y ORDER BY holiday_date ASC");
        $q->bindValue(':cid', $u['company_id'], PDO::PARAM_INT);
        $q->bindValue(':y', $year, PDO::PARAM_INT);
        $q->execute();
        ApiResponse::success($q->fetchAll());
    } catch (Throwable $e) { ApiResponse::error('Failed to load holidays: '.$e->getMessage(), 500); }
}

function listHolidaysInRange(PDO $db){
    $u = getCurrentUser();
    ensureHolidaysTable($db);
    $from = $_GET['from'] ?? '';
    $to = $_GET['to'] ?? '';
    if ($from === '' || $to === '') ApiResponse::error('from/to required');
    try {
        $q = $db->prepare("SELECT id, holiday_date, name FROM holidays WHERE company_id = :cid AND holiday_date BETWEEN :f AND :t ORDER BY holiday_date ASC");
        $q->bindValue(':cid', $u['company_id'], PDO::PARAM_INT);
        $q->bindValue(':f', $from);
        $q->bindValue(':t', $to);
        $q->execute();
        ApiResponse::success($q->fetchAll());
    } catch (Throwable $e) { ApiResponse::error('Failed to load holidays: '.$e->getMessage(), 500); }
}

function saveHoliday(PDO $db){
    $u = getCurrentUser();
    mustBeHrRole($u);
    ensureHolidaysTable($db);
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    $date = trim((string)($in['holiday_date'] ?? ''));
    $name = trim((string)($in['name'] ?? ''));
    if ($date === '' || $name === '') ApiResponse::error('Holiday date and name are required');
    try {
        if ($id > 0) {
            $q = $db->prepare("UPDATE holidays SET holiday_date = :d, name = :n WHERE id = :id AND company_id = :cid");
            $q->execute([':d'=>$date, ':n'=>$name, ':id'=>$id, ':cid'=>$u['company_id']]);
        } else {
            $q = $db->prepare("INSERT INTO holidays (company_id, holiday_date, name) VALUES (:cid, :d, :n) ON DUPLICATE KEY UPDATE name = VALUES(name)");
            $q->execute([':cid'=>$u['company_id'], ':d'=>$date, ':n'=>$name]);
        }
        ApiResponse::success(null, 'Saved');
    } catch (Throwable $e) { ApiResponse::error('Failed to save: '.$e->getMessage(), 500); }
}

function deleteHoliday(PDO $db){
    $u = getCurrentUser();
    mustBeHrRole($u);
    ensureHolidaysTable($db);
    $in = json_decode(file_get_contents('php://input'), true) ?: [];
    $id = isset($in['id']) ? (int)$in['id'] : 0;
    if ($id <= 0) ApiResponse::error('Invalid id');
    try {
        $q = $db->prepare("DELETE FROM holidays WHERE id = :id AND company_id = :cid");
        $q->execute([':id'=>$id, ':cid'=>$u['company_id']]);
        ApiResponse::success(null, 'Deleted');
    } catch (Throwable $e) { ApiResponse::error('Failed to delete: '.$e->getMessage(), 500); }
}
