<?php
/**
 * Employee Exit Workflow API
 * - HR Officer can request an exit for an employee
 * - HR Head/Admin approves or rejects the exit
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin', '*');
header('Access-Control-Allow-Methods', 'GET, POST, PUT, OPTIONS');
header('Access-Control-Allow-Headers', 'Content-Type');

require_once '../config/database.php';
require_once __DIR__ . '/_sms.inc.php';
// Reuse notification helpers
require_once __DIR__ . '/_leave_workflow.inc.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

$database = new Database();
$db = $database->getConnection();

ensureExitRequestsTable($db);

switch ($method) {
    case 'GET':
        if ($id) {
            getExitRequest($db, $id);
        } else {
            getExitRequests($db);
        }
        break;
    case 'POST':
        if ($action === 'request') {
            createExitRequest($db);
        } elseif ($action === 'approve') {
            approveExitRequest($db, $id);
        } elseif ($action === 'reject') {
            rejectExitRequest($db, $id);
        } else {
            ApiResponse::error('Unknown action', 400);
        }
        break;
    case 'PUT':
        updateExitRequest($db, $id);
        break;
    default:
        ApiResponse::error('Method not allowed', 405);
}

function ensureExitRequestsTable(PDO $db){
    try {
        $db->exec("CREATE TABLE IF NOT EXISTS exit_requests (
            id INT AUTO_INCREMENT PRIMARY KEY,
            company_id INT NOT NULL,
            employee_id INT NOT NULL,
            requested_by INT NOT NULL,
            exit_type VARCHAR(50) NULL,
            proposed_last_day DATE NULL,
            reason TEXT NULL,
            status ENUM('pending','approved','rejected','cancelled') NOT NULL DEFAULT 'pending',
            approved_by INT NULL,
            approved_at TIMESTAMP NULL DEFAULT NULL,
            rejection_reason TEXT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            KEY idx_company_status (company_id, status),
            KEY idx_employee (employee_id),
            KEY idx_company_created (company_id, created_at)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
        $db->exec("CREATE TABLE IF NOT EXISTS exit_approval_logs (
            id INT AUTO_INCREMENT PRIMARY KEY,
            exit_request_id INT NOT NULL,
            action ENUM('approved','rejected') NOT NULL,
            approver_user_id INT NOT NULL,
            notes TEXT NULL,
            created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
            KEY idx_request (exit_request_id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci");
    } catch (Throwable $e) { /* ignore */ }
}

function getExitRequests(PDO $db){
    $user = getCurrentUser();
    $cid = (int)$user['company_id'];
    $status = isset($_GET['status']) && $_GET['status'] !== '' ? $_GET['status'] : null;
    $q = "SELECT er.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name,
                 e.employee_number, d.name AS department_name,
                 rb.username AS requested_by_username, ab.username AS approved_by_username
          FROM exit_requests er
          JOIN employees e ON e.id = er.employee_id
          LEFT JOIN departments d ON d.id = e.department_id
          LEFT JOIN users rb ON rb.id = er.requested_by
          LEFT JOIN users ab ON ab.id = er.approved_by
          WHERE er.company_id = :cid" . ($status ? " AND er.status = :st" : '') . "
          ORDER BY er.created_at DESC";
    $st = $db->prepare($q);
    $st->bindValue(':cid', $cid, PDO::PARAM_INT);
    if ($status) $st->bindValue(':st', $status);
    $st->execute();
    ApiResponse::success($st->fetchAll());
}

function getExitRequest(PDO $db, int $id){
    $user = getCurrentUser();
    $cid = (int)$user['company_id'];
    $q = "SELECT er.*, CONCAT(e.first_name,' ',e.last_name) AS employee_name,
                 e.employee_number, d.name AS department_name
          FROM exit_requests er
          JOIN employees e ON e.id = er.employee_id
          LEFT JOIN departments d ON d.id = e.department_id
          WHERE er.id = :id AND er.company_id = :cid LIMIT 1";
    $st = $db->prepare($q);
    $st->bindValue(':id', $id, PDO::PARAM_INT);
    $st->bindValue(':cid', $cid, PDO::PARAM_INT);
    $st->execute();
    $row = $st->fetch();
    if (!$row) ApiResponse::notFound('Exit request not found');
    ApiResponse::success($row);
}

function createExitRequest(PDO $db){
    $user = getCurrentUser();
    // Creator must be allowed to edit employees (HR Officer/HR Head/Admin)
    $role = $user['role_slug'] ?? '';
    if (!in_array($role, ['super_admin','admin','hr_head','hr_officer'], true)) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $employeeId = isset($input['employee_id']) ? (int)$input['employee_id'] : 0;
    if ($employeeId <= 0) ApiResponse::error('employee_id is required');

    // Verify employee belongs to same company
    $emp = $db->prepare("SELECT id, company_id, phone, CONCAT(first_name,' ',last_name) AS name FROM employees WHERE id=:id LIMIT 1");
    $emp->bindValue(':id', $employeeId, PDO::PARAM_INT);
    $emp->execute();
    $row = $emp->fetch();
    if (!$row || (int)$row['company_id'] !== (int)$user['company_id']) {
        ApiResponse::forbidden('Employee not found in your company');
    }

    $exitType = isset($input['exit_type']) ? trim((string)$input['exit_type']) : null;
    $proposedLastDay = isset($input['proposed_last_day']) ? $input['proposed_last_day'] : null;
    $reason = isset($input['reason']) ? trim((string)$input['reason']) : null;

    try {
        $ins = $db->prepare("INSERT INTO exit_requests (company_id, employee_id, requested_by, exit_type, proposed_last_day, reason, status, created_at)
                             VALUES (:cid, :eid, :uid, :type, :last, :reason, 'pending', NOW())");
        $ins->bindValue(':cid', (int)$user['company_id'], PDO::PARAM_INT);
        $ins->bindValue(':eid', $employeeId, PDO::PARAM_INT);
        $ins->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $ins->bindValue(':type', $exitType);
        $ins->bindValue(':last', $proposedLastDay);
        $ins->bindValue(':reason', $reason);
        $ins->execute();
        $rid = (int)$db->lastInsertId();

        // Notify HR Head approvers
        if (function_exists('ensureNotificationsTable')) ensureNotificationsTable($db);
        $title = 'Exit Approval Required';
        $content = sprintf('%s submitted exit request for %s (type: %s).',
            $user['first_name'] . ' ' . $user['last_name'], $row['name'] ?? ('#'.$employeeId), $exitType ?: 'exit');
        $recipients = function_exists('getApproverUserIdsByRole') ? getApproverUserIdsByRole($db, (int)$user['company_id'], ['hr_head']) : [];
        if (!$recipients) {
            $recipients = function_exists('getApproverUserIdsByRole') ? getApproverUserIdsByRole($db, (int)$user['company_id'], ['admin','super_admin']) : [];
        }
        foreach ($recipients as $uid) {
            $n = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, 'exit_request_needs_approval', :t, :c, NOW())");
            $n->bindValue(':uid', $uid, PDO::PARAM_INT);
            $n->bindValue(':t', $title);
            $n->bindValue(':c', $content);
            $n->execute();
        }

        // Optional SMS to employee (informational)
        try { if (!empty($row['phone'])) { sms_send((int)$user['company_id'], $row['phone'], 'Exit request submitted and pending approval.'); } } catch (Throwable $e) { /* ignore */ }

        // Log activity
        recordActivity($db, (int)$user['company_id'], (int)$user['id'], 'exit_request_created', 'exit_request', $rid, null, ['employee_id'=>$employeeId]);

        ApiResponse::success(['id' => $rid], 'Exit request submitted');
    } catch (Throwable $e) {
        ApiResponse::error('Failed to submit exit request: ' . $e->getMessage());
    }
}

function approveExitRequest(PDO $db, int $id){
    $user = getCurrentUser();
    if (!canApprove('exit')) ApiResponse::forbidden('Insufficient permissions to approve exits');
    $input = json_decode(file_get_contents('php://input'), true) ?: [];

    // Verify pending request in same company
    $q = $db->prepare("SELECT er.*, e.company_id, e.id AS emp_id, e.phone AS emp_phone FROM exit_requests er JOIN employees e ON e.id = er.employee_id WHERE er.id = :id AND er.status = 'pending' LIMIT 1");
    $q->bindValue(':id', $id, PDO::PARAM_INT);
    $q->execute();
    $req = $q->fetch();
    if (!$req || (int)$req['company_id'] !== (int)$user['company_id']) ApiResponse::error('Exit request not found or already processed');

    try {
        $db->beginTransaction();
        // Approval log
        $log = $db->prepare("INSERT INTO exit_approval_logs (exit_request_id, action, approver_user_id, notes, created_at) VALUES (:rid, 'approved', :uid, :notes, NOW())");
        $log->bindValue(':rid', $id, PDO::PARAM_INT);
        $log->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $log->bindValue(':notes', $input['notes'] ?? null);
        $log->execute();

        // Finalize request
        $u = $db->prepare("UPDATE exit_requests SET status='approved', approved_by=:uid, approved_at=NOW(), updated_at=NOW() WHERE id=:id");
        $u->bindValue(':id', $id, PDO::PARAM_INT);
        $u->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $u->execute();

        // Update employee status and exit date based on exit_type
        $lastDay = $req['proposed_last_day'] ?: date('Y-m-d');
        $etype = strtolower(trim((string)($req['exit_type'] ?? '')));
        $map = [ 'resignation' => 'resigned', 'retirement' => 'retired', 'termination' => 'terminated', 'exit' => 'exited' ];
        $newStatus = $map[$etype] ?? 'exited';
        $ue = $db->prepare("UPDATE employees SET employment_status = :st, staff_exited_date = :d, updated_at=NOW() WHERE id=:eid");
        $ue->bindValue(':st', $newStatus);
        $ue->bindValue(':d', $lastDay);
        $ue->bindValue(':eid', (int)$req['emp_id'], PDO::PARAM_INT);
        $ue->execute();

        // Notify requester (if exists)
        if (function_exists('ensureNotificationsTable')) ensureNotificationsTable($db);
        $n = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, 'exit_request_approved', :t, :c, NOW())");
        $n->bindValue(':uid', (int)$req['requested_by'], PDO::PARAM_INT);
        $n->bindValue(':t', 'Exit Request Approved');
        $n->bindValue(':c', 'Your exit request has been approved.');
        $n->execute();

        // Optional SMS to employee
        try { if (!empty($req['emp_phone'])) { sms_send((int)$user['company_id'], $req['emp_phone'], 'Exit approved. Last working day: ' . $lastDay); } } catch (Throwable $e) { /* ignore */ }

        recordActivity($db, (int)$user['company_id'], (int)$user['id'], 'exit_request_approved', 'exit_request', $id, null, ['employee_id'=>(int)$req['emp_id']]);

        $db->commit();
        ApiResponse::success(null, 'Exit request approved');
    } catch (Throwable $e) {
        if ($db->inTransaction()) $db->rollBack();
        ApiResponse::error('Failed to approve exit request: ' . $e->getMessage());
    }
}

function rejectExitRequest(PDO $db, int $id){
    $user = getCurrentUser();
    if (!canApprove('exit')) ApiResponse::forbidden('Insufficient permissions to reject exits');
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    if (empty($input['rejection_reason'])) ApiResponse::error('rejection_reason is required');

    // Verify pending request in same company
    $q = $db->prepare("SELECT er.*, e.company_id, e.phone AS emp_phone FROM exit_requests er JOIN employees e ON e.id = er.employee_id WHERE er.id = :id AND er.status = 'pending' LIMIT 1");
    $q->bindValue(':id', $id, PDO::PARAM_INT);
    $q->execute();
    $req = $q->fetch();
    if (!$req || (int)$req['company_id'] !== (int)$user['company_id']) ApiResponse::error('Exit request not found or already processed');

    try {
        $db->beginTransaction();
        // Log rejection
        $log = $db->prepare("INSERT INTO exit_approval_logs (exit_request_id, action, approver_user_id, notes, created_at) VALUES (:rid, 'rejected', :uid, :notes, NOW())");
        $log->bindValue(':rid', $id, PDO::PARAM_INT);
        $log->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $log->bindValue(':notes', $input['rejection_reason']);
        $log->execute();

        // Update request
        $u = $db->prepare("UPDATE exit_requests SET status='rejected', approved_by=:uid, approved_at=NOW(), rejection_reason=:reason, updated_at=NOW() WHERE id=:id");
        $u->bindValue(':id', $id, PDO::PARAM_INT);
        $u->bindValue(':uid', (int)$user['id'], PDO::PARAM_INT);
        $u->bindValue(':reason', $input['rejection_reason']);
        $u->execute();

        // Notify requester
        if (function_exists('ensureNotificationsTable')) ensureNotificationsTable($db);
        $n = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, 'exit_request_rejected', :t, :c, NOW())");
        $n->bindValue(':uid', (int)$req['requested_by'], PDO::PARAM_INT);
        $n->bindValue(':t', 'Exit Request Rejected');
        $n->bindValue(':c', 'Your exit request has been rejected.');
        $n->execute();

        // Optional SMS to employee
        try { if (!empty($req['emp_phone'])) { sms_send((int)$user['company_id'], $req['emp_phone'], 'Exit request rejected. Reason: ' . (string)($input['rejection_reason'] ?? '')); } } catch (Throwable $e) { /* ignore */ }

        recordActivity($db, (int)$user['company_id'], (int)$user['id'], 'exit_request_rejected', 'exit_request', $id, null, null);

        $db->commit();
        ApiResponse::success(null, 'Exit request rejected');
    } catch (Throwable $e) {
        if ($db->inTransaction()) $db->rollBack();
        ApiResponse::error('Failed to reject exit request: ' . $e->getMessage());
    }
}

function updateExitRequest(PDO $db, int $id){
    $user = getCurrentUser();
    // Only allow update by creator while pending, or HR roles
    $input = json_decode(file_get_contents('php://input'), true) ?: [];
    $q = $db->prepare("SELECT * FROM exit_requests WHERE id=:id LIMIT 1");
    $q->bindValue(':id', $id, PDO::PARAM_INT);
    $q->execute();
    $req = $q->fetch();
    if (!$req) ApiResponse::notFound('Exit request not found');
    $role = $user['role_slug'] ?? '';
    $canEdit = ($req['status'] === 'pending' && (int)$req['requested_by'] === (int)$user['id']) || in_array($role, ['super_admin','admin','hr_head'], true);
    if (!$canEdit) ApiResponse::forbidden('Insufficient permissions');

    $fields = ['exit_type','proposed_last_day','reason'];
    $sets = [];
    $params = [':id'=>$id];
    foreach ($fields as $f){
        if (array_key_exists($f, $input)){
            $sets[] = "$f = :$f";
            $params[":$f"] = $input[$f];
        }
    }
    if (!$sets) ApiResponse::success(null, 'No changes');
    $sql = 'UPDATE exit_requests SET '.implode(', ', $sets).', updated_at = NOW() WHERE id = :id AND status = \'pending\'';
    $st = $db->prepare($sql);
    foreach ($params as $k=>$v){ $st->bindValue($k, $v); }
    $st->execute();
    ApiResponse::success(null, 'Updated');
}
