<?php
/**
 * Documents Library API
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$database = new Database();
$db = $database->getConnection();
$user = getCurrentUser();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? 'list';

ensureDocTables($db);

switch ($method) {
  case 'GET':
    if ($action === 'download') download($db, $user);
    elseif ($action === 'employees') employeesSearch($db, $user);
    elseif ($action === 'acks') acks($db, $user);
    elseif ($action === 'recent') recentDocs($db, $user);
    else listDocs($db, $user);
    break;
  case 'POST':
    if ($action === 'upload') upload($db, $user);
    elseif ($action === 'delete') del($db, $user);
    elseif ($action === 'update_access') updateAccess($db, $user);
    elseif ($action === 'ack') ack($db, $user);
    elseif ($action === 'notify_due') notifyDueDocs($db, $user);
    else ApiResponse::error('Invalid action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function notifyDueDocs(PDO $db, array $user){
  // Generate reminder notifications for documents with ack_required=1 and due within next 60 minutes
  $cid = (int)$user['company_id'];
  // Find candidate documents
  $docs = $db->prepare("SELECT id, title, visibility, ack_due_at FROM documents WHERE company_id=:cid AND status='active' AND ack_required=1 AND ack_due_at IS NOT NULL AND ack_due_at BETWEEN NOW() AND DATE_ADD(NOW(), INTERVAL 60 MINUTE)");
  $docs->execute([':cid'=>$cid]);
  $drows = $docs->fetchAll();
  if (!$drows){ ApiResponse::success(['count'=>0]); }
  $total=0;
  foreach ($drows as $d){
    $docId = (int)$d['id']; $vis = $d['visibility']; $title = $d['title'];
    // targets without acknowledgment
    if ($vis==='all'){
      $tq = $db->prepare("SELECT u.id AS user_id, e.id AS employee_id FROM users u JOIN employees e ON u.id=e.user_id WHERE u.company_id=:cid AND u.status='active' AND NOT EXISTS(SELECT 1 FROM document_acknowledgments da WHERE da.document_id=:doc AND da.employee_id=e.id)");
      $tq->execute([':cid'=>$cid, ':doc'=>$docId]);
    } else {
      $tq = $db->prepare("SELECT u.id AS user_id, e.id AS employee_id FROM document_access da JOIN employees e ON da.employee_id=e.id JOIN users u ON e.user_id=u.id WHERE da.document_id=:doc AND NOT EXISTS(SELECT 1 FROM document_acknowledgments da2 WHERE da2.document_id=:doc AND da2.employee_id=e.id)");
      $tq->execute([':doc'=>$docId]);
    }
    $targets = $tq->fetchAll(); if (!$targets) continue;
    $ins = $db->prepare("INSERT INTO notifications(user_id, type, title, content, data, created_at) SELECT :u,'document_ack_reminder',:title,:content,:data,NOW() FROM DUAL WHERE NOT EXISTS(SELECT 1 FROM notifications n WHERE n.user_id=:u AND n.type='document_ack_reminder' AND JSON_EXTRACT(n.data,'$.document_id')=:doc AND n.created_at>DATE_SUB(NOW(), INTERVAL 6 HOUR))");
    $content = 'Reminder: Please acknowledge document: ' . $title;
    $data = json_encode(['document_id'=>$docId]);
    foreach ($targets as $t){ $ins->execute([':u'=>(int)$t['user_id'], ':title'=>$title, ':content'=>$content, ':data'=>$data, ':doc'=>$docId]); $total += (int)$ins->rowCount(); }
  }
  ApiResponse::success(['count'=>$total]);
}

function updateAccess(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $vis = ($in['visibility'] ?? 'all') === 'selected' ? 'selected' : 'all';
  $selArr = is_array($in['selected_employees'] ?? null) ? ($in['selected_employees']) : [];
  // Update visibility
  $st = $db->prepare("UPDATE documents SET visibility=:v WHERE id=:id AND company_id=:cid");
  $st->execute([':v'=>$vis, ':id'=>$id, ':cid'=>$user['company_id']]);
  // Reset and insert access if selected
  $db->prepare('DELETE FROM document_access WHERE document_id=:d')->execute([':d'=>$id]);
  if ($vis==='selected' && count($selArr)){
    $ins = $db->prepare('INSERT INTO document_access(document_id, employee_id) VALUES (:d,:e)');
    foreach ($selArr as $eid){ if (!is_numeric($eid)) continue; $ins->execute([':d'=>$id, ':e'=>(int)$eid]); }
  }
  ApiResponse::success(null, 'Updated');
}

function ensureDocTables(PDO $db){
  $db->exec("CREATE TABLE IF NOT EXISTS documents (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT NULL,
    category VARCHAR(100) NULL,
    file_path VARCHAR(255) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    mime_type VARCHAR(128) NULL,
    size BIGINT NULL,
    visibility ENUM('all','selected') NOT NULL DEFAULT 'all',
    ack_required TINYINT(1) NOT NULL DEFAULT 0,
    ack_due_at DATETIME NULL,
    status ENUM('active','archived') NOT NULL DEFAULT 'active',
    uploaded_by INT NOT NULL,
    created_at DATETIME NOT NULL,
    INDEX(company_id), INDEX(category), INDEX(visibility), INDEX(status)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS document_access (
    id INT AUTO_INCREMENT PRIMARY KEY,
    document_id INT NOT NULL,
    employee_id INT NOT NULL,
    UNIQUE KEY uniq_doc_emp (document_id, employee_id),
    INDEX(document_id), INDEX(employee_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  // Upgrade path: ensure ack columns exist
  try { $db->exec("ALTER TABLE documents ADD COLUMN IF NOT EXISTS ack_required TINYINT(1) NOT NULL DEFAULT 0"); } catch (Throwable $e) {}
  try { $db->exec("ALTER TABLE documents ADD COLUMN IF NOT EXISTS ack_due_at DATETIME NULL"); } catch (Throwable $e) {}
  // Acknowledgments table
  $db->exec("CREATE TABLE IF NOT EXISTS document_acknowledgments (
    id INT AUTO_INCREMENT PRIMARY KEY,
    document_id INT NOT NULL,
    employee_id INT NOT NULL,
    acknowledged_at DATETIME NOT NULL,
    comment TEXT NULL,
    UNIQUE KEY uniq_doc_ack (document_id, employee_id),
    INDEX(document_id), INDEX(employee_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

function listDocs(PDO $db, array $user){
  $q = trim((string)($_GET['q'] ?? ''));
  $cat = trim((string)($_GET['category'] ?? ''));
  $manage = isset($_GET['manage']) && $_GET['manage']=='1';
  $params = [':cid'=>$user['company_id']];

  // Access filter
  $where = "d.company_id = :cid AND d.status='active'";
  $role = $user['role_slug'] ?? '';
  $empId = getEmpId($db, $user['id']);
  // Non-HR roles: restrict by visibility/selection. HR/Admin see all regardless of visibility.
  if (!in_array($role, ['super_admin','admin','hr_head','hr_officer'])){
    if ($empId){
      $where .= " AND (d.visibility='all' OR (d.visibility='selected' AND EXISTS(SELECT 1 FROM document_access da WHERE da.document_id = d.id AND da.employee_id = :eid)))";
      $params[':eid'] = $empId;
    } else {
      // fallback: show only 'all'
      $where .= " AND d.visibility='all'";
    }
  }
  if ($q !== ''){
    $where .= " AND (d.title LIKE :q OR d.description LIKE :q OR d.category LIKE :q)"; $params[':q'] = '%'.$q.'%';
  }
  if ($cat !== ''){
    $where .= " AND d.category = :cat"; $params[':cat'] = $cat;
  }

  $extra = '';
  if ($empId){
    $extra = ", (SELECT acknowledged_at FROM document_acknowledgments da2 WHERE da2.document_id = d.id AND da2.employee_id = :eid LIMIT 1) AS acked_at";
    if (!array_key_exists(':eid', $params)) $st_dummy=true; // marker to avoid lints
    $params[':eid'] = $empId;
  } else {
    $extra = ", NULL AS acked_at";
  }
  $sql = "SELECT d.*, (SELECT COUNT(*) FROM document_access da WHERE da.document_id = d.id) AS selected_count $extra FROM documents d WHERE $where ORDER BY d.created_at DESC, d.id DESC LIMIT 500";
  $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); } $st->execute();
  $rows = $st->fetchAll();
  ApiResponse::success($rows);
}

function upload(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');

  // Using multipart/form-data
  $title = trim($_POST['title'] ?? ''); if ($title==='') ApiResponse::error('title required');
  $desc = trim($_POST['description'] ?? '') ?: null;
  $cat = trim($_POST['category'] ?? '') ?: null;
  $vis = ($_POST['visibility'] ?? 'all') === 'selected' ? 'selected' : 'all';
  $ackRequired = (isset($_POST['ack_required']) && ((string)$_POST['ack_required'] === '1' || $_POST['ack_required'] === 'true')) ? 1 : 0;
  $ackDue = trim((string)($_POST['ack_due_at'] ?? '')) ?: null;
  $sel = $_POST['selected_employees'] ?? '[]';
  $selArr = json_decode($sel, true); if (!is_array($selArr)) $selArr = [];

  if (!isset($_FILES['file'])) ApiResponse::error('file required');
  $f = $_FILES['file']; if ($f['error'] !== UPLOAD_ERR_OK) ApiResponse::error('upload failed');

  $dir = realpath(__DIR__.'/../storage/uploads/documents');
  if ($dir === false){ @mkdir(__DIR__.'/../storage/uploads/documents', 0777, true); $dir = realpath(__DIR__.'/../storage/uploads/documents'); }
  if ($dir === false) ApiResponse::error('storage unavailable');
  $subdir = $dir.DIRECTORY_SEPARATOR.$user['company_id'];
  if (!is_dir($subdir)) @mkdir($subdir, 0777, true);
  $ext = pathinfo($f['name'], PATHINFO_EXTENSION);
  $rand = bin2hex(random_bytes(6));
  $stored = $subdir.DIRECTORY_SEPARATOR.'doc-'.$user['company_id'].'-'.date('YmdHis').'-'.$rand.($ext?('.'.$ext):'');
  if (!@move_uploaded_file($f['tmp_name'], $stored)) ApiResponse::error('save failed');

  $relPath = 'storage/uploads/documents/'.$user['company_id'].'/'.basename($stored);
  $st = $db->prepare("INSERT INTO documents (company_id,title,description,category,file_path,file_name,mime_type,size,visibility,status,uploaded_by,created_at,ack_required,ack_due_at)
                      VALUES (:cid,:t,:d,:c,:p,:fn,:mt,:sz,:v,'active',:uid,NOW(),:ar,:ad)");
  $st->execute([
    ':cid'=>$user['company_id'], ':t'=>$title, ':d'=>$desc, ':c'=>$cat,
    ':p'=>$relPath, ':fn'=>$f['name'], ':mt'=>$f['type'] ?? null, ':sz'=>$f['size'] ?? null,
    ':v'=>$vis, ':uid'=>$user['id'], ':ar'=>$ackRequired, ':ad'=>$ackDue
  ]);
  $docId = (int)$db->lastInsertId();

  if ($vis==='selected' && count($selArr)){
    $ins = $db->prepare('INSERT IGNORE INTO document_access (document_id, employee_id) VALUES (:d,:e)');
    foreach ($selArr as $eid){ if (!is_numeric($eid)) continue; $ins->execute([':d'=>$docId, ':e'=>(int)$eid]); }
  }

  if ($ackRequired){
    try { notifyAckTargets($db, $user['company_id'], $docId, $title, $vis); } catch (Throwable $e) { /* ignore */ }
  }

  ApiResponse::success(['id'=>$docId, 'file'=>$relPath], 'Uploaded');
}

function del(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $st = $db->prepare("UPDATE documents SET status='archived' WHERE id=:id AND company_id=:cid");
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  ApiResponse::success(null, 'Deleted');
}

function download(PDO $db, array $user){
  $id = (int)($_GET['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  // Check access
  $doc = getDoc($db, $user, $id); if (!$doc) ApiResponse::forbidden('Not allowed');
  $abs = realpath(__DIR__.'/../'. $doc['file_path']);
  if (!$abs || !is_file($abs)) ApiResponse::error('file missing', 404);
  $mt = $doc['mime_type'] ?: 'application/octet-stream';
  header('Content-Type: '.$mt);
  header('Content-Disposition: attachment; filename="'.basename($doc['file_name']).'"');
  header('Content-Length: '.filesize($abs));
  readfile($abs);
  exit;
}

function getDoc(PDO $db, array $user, int $id){
  $st = $db->prepare('SELECT * FROM documents WHERE id=:id AND company_id=:cid AND status=\'active\'');
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  $d = $st->fetch(); if (!$d) return null;
  $role = $user['role_slug'] ?? '';
  if ($d['visibility']==='all') return $d;
  if (in_array($role, ['super_admin','admin','hr_head','hr_officer'])) return $d;
  $empId = getEmpId($db, $user['id']); if (!$empId) return null;
  $c = $db->prepare('SELECT 1 FROM document_access WHERE document_id=:d AND employee_id=:e');
  $c->execute([':d'=>$id, ':e'=>$empId]);
  return $c->fetch() ? $d : null;
}

function getEmpId(PDO $db, int $userId){
  try { $q=$db->prepare('SELECT id FROM employees WHERE user_id = :u'); $q->execute([':u'=>$userId]); return (int)($q->fetchColumn() ?: 0); } catch (Throwable $e){ return 0; }
}

function employeesSearch(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $q = trim((string)($_GET['q'] ?? ''));
  if ($q==='') ApiResponse::success([]);
  $sql = "SELECT e.id, e.employee_number, e.first_name, e.last_name, d.name AS department_name
          FROM employees e
          LEFT JOIN departments d ON e.department_id = d.id
          WHERE e.company_id = :cid AND (
            e.first_name LIKE :q OR e.last_name LIKE :q OR e.employee_number LIKE :q
          ) ORDER BY e.first_name, e.last_name LIMIT 20";
  $st = $db->prepare($sql);
  $st->execute([':cid'=>$user['company_id'], ':q'=>'%'.$q.'%']);
  ApiResponse::success($st->fetchAll());
}

function recentDocs(PDO $db, array $user){
  $days = isset($_GET['days']) && is_numeric($_GET['days']) ? max(1, min(60, (int)$_GET['days'])) : 7;
  $allOnly = isset($_GET['all_only']) && $_GET['all_only']=='1';
  $role = $user['role_slug'] ?? '';
  $params = [':cid'=>$user['company_id'], ':from'=>date('Y-m-d H:i:s', time() - $days*86400)];
  $where = "d.company_id=:cid AND d.status='active' AND d.created_at >= :from";
  if ($allOnly){ $where .= " AND d.visibility='all'"; }
  if (!in_array($role, ['super_admin','admin','hr_head','hr_officer'])){
    $empId = getEmpId($db, $user['id']); if ($empId){
      $where .= " AND (d.visibility='all' OR (d.visibility='selected' AND EXISTS(SELECT 1 FROM document_access da WHERE da.document_id=d.id AND da.employee_id=:eid)))";
      $params[':eid']=$empId;
    } else {
      $where .= " AND d.visibility='all'";
    }
  }
  $sql = "SELECT d.*, (SELECT acknowledged_at FROM document_acknowledgments da2 WHERE da2.document_id=d.id AND da2.employee_id = :eid LIMIT 1) AS acked_at
          FROM documents d WHERE $where ORDER BY d.created_at DESC LIMIT 50";
  $st = $db->prepare($sql); foreach($params as $k=>$v){ $st->bindValue($k,$v); }
  // bind :eid if missing
  if (!array_key_exists(':eid', $params)){
    $empId = getEmpId($db, $user['id']); if ($empId) $st->bindValue(':eid',$empId, PDO::PARAM_INT); else $st->bindValue(':eid', 0, PDO::PARAM_INT);
  }
  $st->execute();
  ApiResponse::success($st->fetchAll());
}

function ack(PDO $db, array $user){
  $empId = getEmpId($db, $user['id']); if (!$empId) ApiResponse::forbidden('Only employees can acknowledge');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $comment = trim((string)($in['comment'] ?? '')) ?: null;
  // Check access and requirement
  $st = $db->prepare("SELECT * FROM documents WHERE id=:id AND company_id=:cid AND status='active'");
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  $doc = $st->fetch(); if (!$doc) ApiResponse::forbidden('Not allowed');
  if ($doc['visibility'] !== 'all'){
    $c = $db->prepare('SELECT 1 FROM document_access WHERE document_id=:d AND employee_id=:e');
    $c->execute([':d'=>$id, ':e'=>$empId]);
    if (!$c->fetch()) ApiResponse::forbidden('Not allowed');
  }
  if (!(int)$doc['ack_required']) ApiResponse::error('Acknowledgment not required for this document', 400);
  $ins = $db->prepare("INSERT INTO document_acknowledgments(document_id, employee_id, acknowledged_at, comment) VALUES(:d,:e,NOW(),:c)
                       ON DUPLICATE KEY UPDATE acknowledged_at=VALUES(acknowledged_at), comment=VALUES(comment)");
  $ins->execute([':d'=>$id, ':e'=>$empId, ':c'=>$comment]);
  ApiResponse::success(['id'=>$id, 'employee_id'=>$empId], 'Acknowledged');
}

function acks(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $id = (int)($_GET['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $st = $db->prepare("SELECT * FROM documents WHERE id=:id AND company_id=:cid");
  $st->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  $doc = $st->fetch(); if (!$doc) ApiResponse::error('Document not found', 404);
  // Target employees
  if ($doc['visibility']==='all'){
    $targets = $db->prepare("SELECT e.id, e.first_name, e.last_name, u.id AS user_id FROM employees e JOIN users u ON e.user_id=u.id WHERE e.company_id=:cid AND u.status='active'");
    $targets->execute([':cid'=>$user['company_id']]);
  } else {
    $targets = $db->prepare("SELECT e.id, e.first_name, e.last_name, u.id AS user_id FROM document_access da JOIN employees e ON da.employee_id=e.id JOIN users u ON e.user_id=u.id WHERE da.document_id=:d");
    $targets->execute([':d'=>$id]);
  }
  $trows = $targets->fetchAll();
  $targetIds = array_map(fn($r)=> (int)$r['id'], $trows);
  $ackMap = [];
  if ($targetIds){
    $inQ = implode(',', array_fill(0, count($targetIds), '?'));
    $q = $db->prepare("SELECT employee_id, acknowledged_at FROM document_acknowledgments WHERE document_id=? AND employee_id IN ($inQ)");
    $params = array_merge([$id], $targetIds); $q->execute($params);
    foreach ($q->fetchAll() as $r){ $ackMap[(int)$r['employee_id']] = $r['acknowledged_at']; }
  }
  $acked = []; $pending = [];
  foreach ($trows as $r){
    $eid = (int)$r['id'];
    $row = ['employee_id'=>$eid, 'first_name'=>$r['first_name'], 'last_name'=>$r['last_name'], 'acknowledged_at'=>($ackMap[$eid] ?? null)];
    if (isset($ackMap[$eid])) $acked[] = $row; else $pending[] = $row;
  }
  ApiResponse::success([
    'document'=>['id'=>$doc['id'], 'title'=>$doc['title'], 'ack_required'=>(int)$doc['ack_required'], 'ack_due_at'=>$doc['ack_due_at'], 'visibility'=>$doc['visibility']],
    'acked_count'=>count($acked),
    'pending_count'=>count($pending),
    'acked'=>$acked,
    'pending'=>$pending
  ]);
}

function notifyAckTargets(PDO $db, int $companyId, int $docId, string $title, string $visibility){
  // determine user targets
  $userIds = [];
  if ($visibility==='all'){
    $q = $db->prepare("SELECT u.id FROM users u WHERE u.company_id=:cid AND u.status='active'");
    $q->execute([':cid'=>$companyId]);
    $userIds = array_map(fn($r)=> (int)$r['id'], $q->fetchAll());
  } else {
    $q = $db->prepare("SELECT u.id FROM document_access da JOIN employees e ON da.employee_id=e.id JOIN users u ON e.user_id=u.id WHERE da.document_id=:id");
    $q->execute([':id'=>$docId]);
    $userIds = array_map(fn($r)=> (int)$r['id'], $q->fetchAll());
  }
  if (!$userIds) return;
  $ins = $db->prepare("INSERT INTO notifications(user_id, type, title, content, data, created_at) VALUES(:u,'document_ack',:title,:content,:data,NOW())");
  $content = 'Please read and acknowledge: ' . $title;
  $data = json_encode(['document_id'=>$docId]);
  foreach ($userIds as $uid){ $ins->execute([':u'=>$uid, ':title'=>$title, ':content'=>$content, ':data'=>$data]); }
}
