<?php
/**
 * Calendar Planner API
 */
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$database = new Database();
$db = $database->getConnection();
$user = getCurrentUser();
$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? 'range';

ensureCalTables($db);

switch ($method) {
  case 'GET':
    if ($action === 'range') rangeEvents($db, $user);
    else ApiResponse::error('Invalid action', 400);
    break;
  case 'POST':
    if ($action === 'save') saveEvent($db, $user);
    elseif ($action === 'delete') deleteEvent($db, $user);
    elseif ($action === 'notify_due') notifyDue($db, $user);
    else ApiResponse::error('Invalid action', 400);
    break;
  default:
    ApiResponse::error('Method not allowed', 405);
}

function ensureCalTables(PDO $db){
  $db->exec("CREATE TABLE IF NOT EXISTS calendar_events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT NULL,
    start_at DATETIME NOT NULL,
    end_at DATETIME NULL,
    visibility ENUM('all','selected') NOT NULL DEFAULT 'all',
    reminder_minutes INT NOT NULL DEFAULT 60,
    reminder_sent TINYINT(1) NOT NULL DEFAULT 0,
    created_by INT NOT NULL,
    created_at DATETIME NOT NULL,
    updated_at DATETIME NULL,
    INDEX(company_id), INDEX(start_at), INDEX(visibility), INDEX(reminder_sent)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
  $db->exec("CREATE TABLE IF NOT EXISTS calendar_event_targets (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL,
    employee_id INT NOT NULL,
    UNIQUE KEY uniq_evt_emp (event_id, employee_id),
    INDEX(event_id), INDEX(employee_id)
  ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4");
}

function rangeEvents(PDO $db, array $user){
  $from = $_GET['from'] ?? null; $to = $_GET['to'] ?? null;
  if (!$from || !$to) ApiResponse::error('from/to required');
  $params = [':cid'=>$user['company_id'], ':from'=>$from.' 00:00:00', ':to'=>$to.' 23:59:59'];
  $where = 'e.company_id = :cid AND e.start_at BETWEEN :from AND :to';
  $empId = getEmpId($db, $user['id']);
  if ($empId){
    $where .= " AND (e.visibility='all' OR (e.visibility='selected' AND EXISTS(SELECT 1 FROM calendar_event_targets t WHERE t.event_id = e.id AND t.employee_id = :eid)))";
    $params[':eid'] = $empId;
  } else {
    $where .= " AND e.visibility='all'";
  }
  $st = $db->prepare("SELECT e.* FROM calendar_events e WHERE $where ORDER BY e.start_at ASC");
  foreach($params as $k=>$v){ $st->bindValue($k,$v); }
  $st->execute();
  ApiResponse::success($st->fetchAll());
}

function saveEvent(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0);
  $title = trim((string)($in['title'] ?? '')); if ($title==='') ApiResponse::error('title required');
  $desc = trim((string)($in['description'] ?? '')) ?: null;
  $start = trim((string)($in['start_at'] ?? '')); if ($start==='') ApiResponse::error('start_at required');
  $end = trim((string)($in['end_at'] ?? '')) ?: null;
  $vis = ($in['visibility'] ?? 'all') === 'selected' ? 'selected' : 'all';
  $rem = is_numeric($in['reminder_minutes'] ?? null) ? max(0, (int)$in['reminder_minutes']) : 60;
  $sel = is_array($in['selected_employees'] ?? null) ? $in['selected_employees'] : [];

  if ($id>0){
    $st = $db->prepare("UPDATE calendar_events SET title=:t, description=:d, start_at=:s, end_at=:e, visibility=:v, reminder_minutes=:r, reminder_sent=0, updated_at=NOW() WHERE id=:id AND company_id=:cid");
    $st->execute([':t'=>$title, ':d'=>$desc, ':s'=>$start, ':e'=>$end, ':v'=>$vis, ':r'=>$rem, ':id'=>$id, ':cid'=>$user['company_id']]);
    $db->prepare('DELETE FROM calendar_event_targets WHERE event_id=:id')->execute([':id'=>$id]);
    if ($vis==='selected'){
      $ins = $db->prepare('INSERT INTO calendar_event_targets(event_id, employee_id) VALUES (:id,:e)');
      foreach ($sel as $eid){ if (!is_numeric($eid)) continue; $ins->execute([':id'=>$id, ':e'=>(int)$eid]); }
    }
    ApiResponse::success(['id'=>$id], 'Updated');
  } else {
    $st = $db->prepare("INSERT INTO calendar_events(company_id,title,description,start_at,end_at,visibility,reminder_minutes,reminder_sent,created_by,created_at) VALUES(:cid,:t,:d,:s,:e,:v,:r,0,:uid,NOW())");
    $st->execute([':cid'=>$user['company_id'], ':t'=>$title, ':d'=>$desc, ':s'=>$start, ':e'=>$end, ':v'=>$vis, ':r'=>$rem, ':uid'=>$user['id']]);
    $eid = (int)$db->lastInsertId();
    if ($vis==='selected'){
      $ins = $db->prepare('INSERT INTO calendar_event_targets(event_id, employee_id) VALUES (:id,:e)');
      foreach ($sel as $emp){ if (!is_numeric($emp)) continue; $ins->execute([':id'=>$eid, ':e'=>(int)$emp]); }
    }
    ApiResponse::success(['id'=>$eid], 'Created');
  }
}

function deleteEvent(PDO $db, array $user){
  if (!in_array($user['role_slug'] ?? '', ['super_admin','admin','hr_head','hr_officer'])) ApiResponse::forbidden('Insufficient permissions');
  $in = json_decode(file_get_contents('php://input'), true) ?? [];
  $id = (int)($in['id'] ?? 0); if ($id<=0) ApiResponse::error('id required');
  $db->prepare('DELETE FROM calendar_event_targets WHERE event_id=:id')->execute([':id'=>$id]);
  $db->prepare('DELETE FROM calendar_events WHERE id=:id AND company_id=:cid')->execute([':id'=>$id, ':cid'=>$user['company_id']]);
  ApiResponse::success(null, 'Deleted');
}

function notifyDue(PDO $db, array $user){
  // Only HR/Admin can trigger server-side notification generation, but we can allow any role to trigger with no data leak
  $cid = (int)$user['company_id'];
  $now = date('Y-m-d H:i:s');
  // Pick events whose reminder window started and not yet sent
  $sql = "SELECT * FROM calendar_events WHERE company_id = :cid AND reminder_sent = 0 AND (TIMESTAMPDIFF(MINUTE, NOW(), start_at) <= reminder_minutes) AND start_at >= NOW() ORDER BY start_at ASC LIMIT 50";
  $st = $db->prepare($sql); $st->execute([':cid'=>$cid]);
  $events = $st->fetchAll();
  if (!$events){ ApiResponse::success(['created'=>0]); }
  $created = 0;
  foreach ($events as $ev){
    $targets = [];
    if ($ev['visibility']==='all'){
      // all active users in company
      $q = $db->prepare("SELECT id FROM users WHERE company_id=:cid AND status='active'");
      $q->execute([':cid'=>$cid]);
      $targets = array_map(fn($r)=> (int)$r['id'], $q->fetchAll());
    } else {
      // selected employees -> map to users
      $q = $db->prepare("SELECT u.id FROM calendar_event_targets t JOIN employees e ON t.employee_id=e.id JOIN users u ON e.user_id = u.id WHERE t.event_id=:id");
      $q->execute([':id'=>$ev['id']]);
      $targets = array_map(fn($r)=> (int)$r['id'], $q->fetchAll());
    }
    if ($targets){
      $ins = $db->prepare("INSERT INTO notifications(user_id, type, title, content, data, created_at) VALUES(:u,'calendar_event',:title,:content,:data,NOW())");
      $title = 'Upcoming event: '.$ev['title'];
      $content = 'Starts at '. $ev['start_at'];
      $data = json_encode(['event_id'=>$ev['id']]);
      foreach ($targets as $uid){ $ins->execute([':u'=>$uid, ':title'=>$title, ':content'=>$content, ':data'=>$data]); $created++; }
    }
    // mark as sent
    $db->prepare('UPDATE calendar_events SET reminder_sent=1 WHERE id=:id')->execute([':id'=>$ev['id']]);
  }
  ApiResponse::success(['created'=>$created], 'Notified');
}

function getEmpId(PDO $db, int $userId){
  try { $q=$db->prepare('SELECT id FROM employees WHERE user_id = :u'); $q->execute([':u'=>$userId]); return (int)($q->fetchColumn() ?: 0); } catch (Throwable $e){ return 0; }
}
