<?php
/**
 * Announcements API
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

requireAuth();

$method = $_SERVER['REQUEST_METHOD'];
$action = $_GET['action'] ?? '';
$id = $_GET['id'] ?? null;

$database = new Database();
$db = $database->getConnection();

switch ($method) {
    case 'GET':
        if ($action === 'latest') {
            listAnnouncements($db, true);
        } elseif ($id) {
            getAnnouncement($db, $id);
        } else {
            listAnnouncements($db, false);
        }
        break;

    case 'POST':
        if ($action === 'publish') {
            publishAnnouncement($db, $id);
        } elseif ($action === 'archive') {
            archiveAnnouncement($db, $id);
        } else {
            createAnnouncement($db);
        }
        break;

    case 'PUT':
        updateAnnouncement($db, $id);
        break;

    case 'DELETE':
        deleteAnnouncement($db, $id);
        break;

    default:
        ApiResponse::error('Method not allowed', 405);
}

function canManageAnnouncements($user) {
    return in_array($user['role_slug'], ['super_admin', 'admin', 'hr_head', 'hr_officer']);
}

function listAnnouncements($db, $latestOnly = false) {
    $user = getCurrentUser();
    $limit = intval($_GET['limit'] ?? 10);

    $query = "SELECT a.*, CONCAT(u.first_name, ' ', u.last_name) AS author_name
              FROM announcements a
              JOIN users u ON a.author_id = u.id
              WHERE a.company_id = :company_id";

    $params = [':company_id' => $user['company_id']];

    if (!$latestOnly) {
        // allow filters for admin/HR to view drafts too
        if (!canManageAnnouncements($user)) {
            $query .= " AND a.status = 'published'";
        } else if (isset($_GET['status']) && $_GET['status'] !== '') {
            $query .= " AND a.status = :status";
            $params[':status'] = $_GET['status'];
        }
    } else {
        $query .= " AND a.status = 'published'";
    }

    $query .= " ORDER BY COALESCE(a.published_at, a.created_at) DESC";
    if ($latestOnly) {
        $query .= " LIMIT :limit";
    }

    $stmt = $db->prepare($query);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v);
    }
    if ($latestOnly) {
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    }
    $stmt->execute();

    $items = $stmt->fetchAll();
    ApiResponse::success($items);
}

function getAnnouncement($db, $id) {
    $user = getCurrentUser();

    $query = "SELECT a.*, CONCAT(u.first_name, ' ', u.last_name) AS author_name
              FROM announcements a
              JOIN users u ON a.author_id = u.id
              WHERE a.id = :id AND a.company_id = :company_id";

    $stmt = $db->prepare($query);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':company_id', $user['company_id'], PDO::PARAM_INT);
    $stmt->execute();

    if ($stmt->rowCount() === 0) {
        ApiResponse::notFound('Announcement not found');
    }

    $item = $stmt->fetch();
    ApiResponse::success($item);
}

function createAnnouncement($db) {
    $user = getCurrentUser();
    if (!canManageAnnouncements($user)) {
        ApiResponse::forbidden('Insufficient permissions');
    }

    $input = json_decode(file_get_contents('php://input'), true);
    if (!isset($input['title']) || !isset($input['content'])) {
        ApiResponse::error('Title and content are required');
    }

    $query = "INSERT INTO announcements (
                company_id, title, content, priority, target_audience, target_criteria,
                status, author_id, created_at
              ) VALUES (
                :company_id, :title, :content, :priority, :target_audience, :target_criteria,
                :status, :author_id, NOW()
              )";

    $stmt = $db->prepare($query);
    $stmt->bindValue(':company_id', $user['company_id']);
    $stmt->bindValue(':title', $input['title']);
    $stmt->bindValue(':content', $input['content']);
    $stmt->bindValue(':priority', $input['priority'] ?? 'medium');
    $stmt->bindValue(':target_audience', $input['target_audience'] ?? 'all');
    $stmt->bindValue(':target_criteria', isset($input['target_criteria']) ? json_encode($input['target_criteria']) : null);
    $stmt->bindValue(':status', $input['status'] ?? 'draft');
    $stmt->bindValue(':author_id', $user['id']);
    $stmt->execute();

    $id = $db->lastInsertId();

    // Log activity
    $auth = new Auth();
    $auth->logActivity($user['id'], 'announcement_created', ['announcement_id' => $id]);

    ApiResponse::success(['id' => $id], 'Announcement created');
}

function updateAnnouncement($db, $id) {
    $user = getCurrentUser();
    if (!canManageAnnouncements($user)) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Announcement ID required');

    $input = json_decode(file_get_contents('php://input'), true);

    $fields = ['title','content','priority','target_audience','target_criteria','status','expires_at'];
    $set = [];
    $params = [':id' => $id, ':company_id' => $user['company_id']];

    foreach ($fields as $f) {
        if (array_key_exists($f, $input)) {
            $set[] = "$f = :$f";
            $params[":$f"] = $f === 'target_criteria' ? json_encode($input[$f]) : $input[$f];
        }
    }
    if (!$set) ApiResponse::error('No fields to update');

    $query = 'UPDATE announcements SET '.implode(', ', $set).', updated_at = NOW() WHERE id = :id AND company_id = :company_id';
    $stmt = $db->prepare($query);
    foreach ($params as $k => $v) {
        $stmt->bindValue($k, $v);
    }
    $stmt->execute();

    ApiResponse::success(null, 'Announcement updated');
}

function publishAnnouncement($db, $id) {
    $user = getCurrentUser();
    if (!canManageAnnouncements($user)) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Announcement ID required');

    $query = "UPDATE announcements SET status = 'published', published_at = NOW(), updated_at = NOW() WHERE id = :id AND company_id = :company_id";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':company_id', $user['company_id'], PDO::PARAM_INT);
    $stmt->execute();

    ApiResponse::success(null, 'Announcement published');
}

function archiveAnnouncement($db, $id) {
    $user = getCurrentUser();
    if (!canManageAnnouncements($user)) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Announcement ID required');

    $query = "UPDATE announcements SET status = 'archived', updated_at = NOW() WHERE id = :id AND company_id = :company_id";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':company_id', $user['company_id'], PDO::PARAM_INT);
    $stmt->execute();

    ApiResponse::success(null, 'Announcement archived');
}

function deleteAnnouncement($db, $id) {
    $user = getCurrentUser();
    if (!canManageAnnouncements($user)) {
        ApiResponse::forbidden('Insufficient permissions');
    }
    if (!$id) ApiResponse::error('Announcement ID required');

    $query = "DELETE FROM announcements WHERE id = :id AND company_id = :company_id";
    $stmt = $db->prepare($query);
    $stmt->bindValue(':id', $id, PDO::PARAM_INT);
    $stmt->bindValue(':company_id', $user['company_id'], PDO::PARAM_INT);
    $stmt->execute();

    ApiResponse::success(null, 'Announcement deleted');
}
