<?php
/**
 * Admin Seeding Tools (DEV ONLY)
 * Creates test users and employee records for role model testing with proper reporting lines.
 *
 * Security: Requires authenticated user with role super_admin or admin.
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    ApiResponse::success(null, 'OK');
}

requireAuth();
$user = getCurrentUser();

if (!in_array($user['role_slug'], ['super_admin', 'admin'])) {
    ApiResponse::forbidden('Admin only');
}

$action = $_GET['action'] ?? '';
$database = new Database();
$db = $database->getConnection();

switch ($action) {
    case 'create_test_access':
        createTestAccess($db, $user);
        break;
    default:
        ApiResponse::error('Invalid action');
}

function ensureRole($db, $slug, $name, $permissions = []) {
    $sel = $db->prepare("SELECT id FROM roles WHERE slug = :slug");
    $sel->bindParam(':slug', $slug);
    $sel->execute();
    if ($sel->rowCount() > 0) {
        $row = $sel->fetch();
        return (int)$row['id'];
    }
    $ins = $db->prepare("INSERT INTO roles (slug, name, permissions, created_at) VALUES (:slug, :name, :permissions, NOW())");
    $ins->bindParam(':slug', $slug);
    $ins->bindParam(':name', $name);
    $perm = json_encode($permissions);
    $ins->bindParam(':permissions', $perm);
    $ins->execute();
    return (int)$db->lastInsertId();
}

function ensureUserAndEmployee($db, $companyId, $creatorUserId, $username, $first, $last, $email, $roleId, $managerEmployeeId = null) {
    // If user exists, return existing mapping (and ensure role mapping exists)
    $selU = $db->prepare("SELECT id, employee_id FROM users WHERE username = :u");
    $selU->bindParam(':u', $username);
    $selU->execute();

    if ($selU->rowCount() > 0) {
        $u = $selU->fetch();
        $userId = (int)$u['id'];
        $employeeId = $u['employee_id'] ? (int)$u['employee_id'] : null;
        // ensure role mapping
        $selUR = $db->prepare("SELECT 1 FROM user_roles WHERE user_id = :uid AND role_id = :rid");
        $selUR->bindParam(':uid', $userId);
        $selUR->bindParam(':rid', $roleId);
        $selUR->execute();
        if ($selUR->rowCount() === 0) {
            $insUR = $db->prepare("INSERT INTO user_roles (user_id, role_id, assigned_by, assigned_at) VALUES (:uid, :rid, :by, NOW())");
            $insUR->bindParam(':uid', $userId);
            $insUR->bindParam(':rid', $roleId);
            $insUR->bindParam(':by', $creatorUserId);
            $insUR->execute();
        }
        if (!$employeeId) {
            $employeeId = createEmployeeForUser($db, $companyId, $creatorUserId, $userId, $first, $last, $email, $managerEmployeeId);
            $upd = $db->prepare("UPDATE users SET employee_id = :eid WHERE id = :uid");
            $upd->bindParam(':eid', $employeeId);
            $upd->bindParam(':uid', $userId);
            $upd->execute();
        }
        return [$userId, $employeeId];
    }

    // Create user
    $pwd = password_hash('password', PASSWORD_DEFAULT);
    $insU = $db->prepare("INSERT INTO users (company_id, username, email, password, first_name, last_name, phone, status, created_by, created_at) VALUES (:cid, :un, :em, :pw, :fn, :ln, NULL, 'active', :by, NOW())");
    $insU->bindParam(':cid', $companyId);
    $insU->bindParam(':un', $username);
    $insU->bindParam(':em', $email);
    $insU->bindParam(':pw', $pwd);
    $insU->bindParam(':fn', $first);
    $insU->bindParam(':ln', $last);
    $insU->bindParam(':by', $creatorUserId);
    $insU->execute();
    $userId = (int)$db->lastInsertId();

    // Role mapping
    $insUR = $db->prepare("INSERT INTO user_roles (user_id, role_id, assigned_by, assigned_at) VALUES (:uid, :rid, :by, NOW())");
    $insUR->bindParam(':uid', $userId);
    $insUR->bindParam(':rid', $roleId);
    $insUR->bindParam(':by', $creatorUserId);
    $insUR->execute();

    // Employee
    $employeeId = createEmployeeForUser($db, $companyId, $creatorUserId, $userId, $first, $last, $email, $managerEmployeeId);

    // Link back to user
    $upd = $db->prepare("UPDATE users SET employee_id = :eid WHERE id = :uid");
    $upd->bindParam(':eid', $employeeId);
    $upd->bindParam(':uid', $userId);
    $upd->execute();

    return [$userId, $employeeId];
}

function createEmployeeForUser($db, $companyId, $creatorUserId, $userId, $first, $last, $email, $managerEmployeeId = null) {
    $empNum = 'E' . str_pad((string)rand(1, 99999), 5, '0', STR_PAD_LEFT);
    $insE = $db->prepare("INSERT INTO employees (
        company_id, sub_company_id, department_id, branch_id, position_id,
        employee_number, first_name, last_name, middle_name, email, phone, mobile,
        date_of_birth, gender, marital_status, nationality, address,
        emergency_contact_name, emergency_contact_phone, emergency_contact_relationship,
        hire_date, probation_end_date, contract_type, employment_status,
        salary, currency, manager_id, notes, created_by, created_at, user_id
    ) VALUES (
        :cid, NULL, NULL, NULL, NULL,
        :emp_no, :fn, :ln, NULL, :email, NULL, NULL,
        NULL, NULL, NULL, NULL, NULL,
        NULL, NULL, NULL,
        CURDATE(), NULL, 'permanent', 'active',
        NULL, 'USD', :manager_id, NULL, :by, NOW(), :uid
    )");
    $insE->bindParam(':cid', $companyId);
    $insE->bindParam(':emp_no', $empNum);
    $insE->bindParam(':fn', $first);
    $insE->bindParam(':ln', $last);
    $insE->bindParam(':email', $email);
    $insE->bindParam(':manager_id', $managerEmployeeId);
    $insE->bindParam(':by', $creatorUserId);
    $insE->bindParam(':uid', $userId);
    $insE->execute();
    return (int)$db->lastInsertId();
}

function createTestAccess($db, $user) {
    try {
        $db->beginTransaction();

        $companyId = (int)($user['company_id'] ?? 1);
        $creatorUserId = (int)$user['id'];

        // Ensure roles
        $roleHrHead   = ensureRole($db, 'hr_head', 'HR Head/Lead', ['hr.*', 'approve.*']);
        $roleHrOfficer= ensureRole($db, 'hr_officer', 'HR Officer', ['employees.create', 'employees.edit', 'documents.upload']);
        $roleManager  = ensureRole($db, 'manager', 'Manager', ['team.view', 'approve.team']);
        $roleEmployee = ensureRole($db, 'employee', 'Employee', ['self.service']);

        // Create HR Head (top)
        list($hrHeadUserId, $hrHeadEmpId) = ensureUserAndEmployee(
            $db, $companyId, $creatorUserId,
            'hrhead', 'HR', 'Head', 'hrhead@example.com', $roleHrHead, null
        );

        // HR Officer (reports to HR Head, for convenience)
        list($hrOfficerUserId, $hrOfficerEmpId) = ensureUserAndEmployee(
            $db, $companyId, $creatorUserId,
            'hrofficer', 'HR', 'Officer', 'hrofficer@example.com', $roleHrOfficer, $hrHeadEmpId
        );

        // Manager (reports to HR Head)
        list($managerUserId, $managerEmpId) = ensureUserAndEmployee(
            $db, $companyId, $creatorUserId,
            'manager1', 'IT', 'Manager', 'manager1@example.com', $roleManager, $hrHeadEmpId
        );

        // Employees (report to Manager)
        list($emp1UserId, $emp1EmpId) = ensureUserAndEmployee(
            $db, $companyId, $creatorUserId,
            'employee1', 'John', 'Doe', 'employee1@example.com', $roleEmployee, $managerEmpId
        );
        list($emp2UserId, $emp2EmpId) = ensureUserAndEmployee(
            $db, $companyId, $creatorUserId,
            'employee2', 'Jane', 'Smith', 'employee2@example.com', $roleEmployee, $managerEmpId
        );

        $db->commit();

        ApiResponse::success([
            'users' => [
                ['username' => 'hrhead',    'password' => 'password', 'role' => 'hr_head',   'employee_id' => $hrHeadEmpId],
                ['username' => 'hrofficer', 'password' => 'password', 'role' => 'hr_officer','employee_id' => $hrOfficerEmpId],
                ['username' => 'manager1',  'password' => 'password', 'role' => 'manager',   'employee_id' => $managerEmpId],
                ['username' => 'employee1', 'password' => 'password', 'role' => 'employee',  'employee_id' => $emp1EmpId],
                ['username' => 'employee2', 'password' => 'password', 'role' => 'employee',  'employee_id' => $emp2EmpId],
            ]
        ], 'Test access users created (or already existed).');

    } catch (Exception $e) {
        $db->rollBack();
        ApiResponse::error('Failed to create test access: ' . $e->getMessage());
    }
}
