<?php
/**
 * Simple SMS helper (mNotify provider)
 * Stores per-company config in storage/config/sms-<company_id>.json
 */

if (!function_exists('sms_config_path')) {
    function sms_config_path(int $companyId): string {
        $base = realpath(__DIR__ . '/../storage');
        if ($base === false) {@mkdir(__DIR__ . '/../storage', 0777, true); $base = realpath(__DIR__ . '/../storage'); }
        $cfgDir = rtrim($base, '/\\') . '/config';
        if (!is_dir($cfgDir)) {@mkdir($cfgDir, 0777, true);}    
        return $cfgDir . '/sms-' . $companyId . '.json';
    }
}

if (!function_exists('sms_load_config')) {
    function sms_load_config(int $companyId): array {
        $file = sms_config_path($companyId);
        if (is_file($file)) {
            $cfg = json_decode(@file_get_contents($file), true);
            if (is_array($cfg)) return $cfg;
        }
        return [
            'provider' => 'mnotify',
            'enabled' => 0,
            'sender_id' => '',
            'api_key' => ''
        ];
    }
}

if (!function_exists('sms_save_config')) {
    function sms_save_config(int $companyId, array $cfg): bool {
        $curr = sms_load_config($companyId);
        $merged = array_merge($curr, $cfg);
        $file = sms_config_path($companyId);
        return (bool)@file_put_contents($file, json_encode($merged, JSON_PRETTY_PRINT));
    }
}

if (!function_exists('sms_send')) {
    /**
     * Send an SMS using configured provider (mNotify).
     * @param int $companyId
     * @param string|array $to One phone or list of phones. Keep in international format without '+' if possible.
     * @param string $message Message text
     * @return bool true if queued/sent, false otherwise
     */
    function sms_send(int $companyId, $to, string $message): bool {
        $cfg = sms_load_config($companyId);
        if (!($cfg['enabled'] ?? 0)) return false;
        $provider = strtolower((string)($cfg['provider'] ?? 'mnotify'));
        $sender = trim((string)($cfg['sender_id'] ?? ''));
        $apiKey = trim((string)($cfg['api_key'] ?? ''));
        if ($provider !== 'mnotify' || $apiKey === '' || $message === '') return false;
        $recipients = is_array($to) ? array_values(array_filter(array_map('strval', $to))) : [trim((string)$to)];
        $recipients = array_values(array_filter($recipients, fn($s)=>$s!=='') );
        if (!$recipients) return false;
        $url = 'https://api.mnotify.com/api/sms/quick?key=' . urlencode($apiKey);
        $payload = [
            'recipient' => $recipients,
            'sender' => $sender !== '' ? $sender : 'SMARTQHR',
            'message' => $message,
            'is_schedule' => false,
        ];
        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer ' . $apiKey
        ];
        $body = json_encode($payload);
        $responseOk = false; $httpCode = 0; $resp = '';
        if (function_exists('curl_init')) {
            $ch = curl_init($url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            $resp = curl_exec($ch);
            if (!curl_errno($ch)) { $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE); }
            curl_close($ch);
            $responseOk = ($httpCode >= 200 && $httpCode < 300);
        } else {
            $ctx = stream_context_create([
                'http' => [
                    'method' => 'POST',
                    'header' => implode("\r\n", $headers),
                    'content' => $body,
                    'timeout' => 15,
                    'ignore_errors' => true,
                ]
            ]);
            $resp = @file_get_contents($url, false, $ctx);
            if (isset($http_response_header) && is_array($http_response_header)){
                foreach ($http_response_header as $h){ if (preg_match('/^HTTP\/\d+\.\d+\s+(\d+)/',$h,$m)){ $httpCode = (int)$m[1]; break; } }
            }
            $responseOk = ($httpCode >= 200 && $httpCode < 300);
        }
        if (!$responseOk) {
            if (function_exists('app_log')) {
                try { app_log('error','sms_send_failed',['provider'=>$provider,'http_code'=>$httpCode,'response'=>$resp]); } catch (Throwable $e) { /* ignore */ }
            }
        }
        return $responseOk;
    }
}
