<?php
// Workflow helpers for multi-stage leave approvals

if (!function_exists('ensureApprovalLogsTable')) {
    function ensureApprovalLogsTable(PDO $db){
        $sql = "CREATE TABLE IF NOT EXISTS leave_approval_logs (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    leave_request_id INT NOT NULL,
                    stage TINYINT NOT NULL,
                    action ENUM('approved','rejected') NOT NULL,
                    approver_user_id INT NOT NULL,
                    notes TEXT NULL,
                    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    KEY idx_request (leave_request_id),
                    KEY idx_request_stage (leave_request_id, stage)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        try { $db->exec($sql); } catch (Throwable $e) { /* ignore */ }
    }
}

// Ensure notifications table exists (in-app notifications)
if (!function_exists('ensureNotificationsTable')) {
    function ensureNotificationsTable(PDO $db){
        $sql = "CREATE TABLE IF NOT EXISTS notifications (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    user_id INT NOT NULL,
                    type VARCHAR(64) NOT NULL,
                    title VARCHAR(255) NOT NULL,
                    content TEXT NULL,
                    data LONGTEXT CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL,
                    read_at TIMESTAMP NULL DEFAULT NULL,
                    created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                    KEY idx_user_read (user_id, read_at),
                    KEY idx_created_at (created_at)
               ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        try { $db->exec($sql); } catch (Throwable $e) { /* ignore */ }
    }
}

if (!function_exists('getRequestContext')) {
    function getRequestContext(PDO $db, int $requestId): ?array {
        $q = "SELECT lr.id, lr.employee_id, e.manager_id,
                     e.user_id AS employee_user_id,
                     mgr.user_id AS manager_user_id,
                     e.company_id
              FROM leave_requests lr
              JOIN employees e ON e.id = lr.employee_id
              LEFT JOIN employees mgr ON mgr.id = e.manager_id
              WHERE lr.id = :id LIMIT 1";
        $st = $db->prepare($q);
        $st->bindValue(':id', $requestId, PDO::PARAM_INT);
        $st->execute();
        if ($st->rowCount() === 0) return null;
        return $st->fetch();
    }
}

if (!function_exists('getApprovedStageCount')) {
    function getApprovedStageCount(PDO $db, int $requestId): int {
        $st = $db->prepare("SELECT COUNT(*) AS c FROM leave_approval_logs WHERE leave_request_id = :id AND action = 'approved'");
        $st->bindValue(':id', $requestId, PDO::PARAM_INT);
        $st->execute();
        $row = $st->fetch();
        return (int)($row['c'] ?? 0);
    }
}

if (!function_exists('getNextStageRole')) {
    function getNextStageRole(PDO $db, int $requestId): ?string {
        $ctx = getRequestContext($db, $requestId);
        if (!$ctx) return null;
        $stages = [];
        if (!empty($ctx['manager_user_id'])) $stages[] = 'manager';
        $stages[] = 'hr_head';
        $stages[] = 'admin'; // final stage
        $approvedCount = getApprovedStageCount($db, $requestId);
        if ($approvedCount >= count($stages)) return null; // already finalized
        return $stages[$approvedCount];
    }
}

if (!function_exists('canUserApproveStage')) {
    function canUserApproveStage(array $user, string $stageRole, ?int $managerUserId): bool {
        $role = $user['role_slug'] ?? '';
        $uid = (int)($user['id'] ?? 0);
        switch ($stageRole) {
            case 'manager':
                // Must be the exact manager user
                return ($role === 'manager') && $managerUserId !== null && $uid === (int)$managerUserId;
            case 'hr_head':
                // HR Head or higher can act on this stage
                return in_array($role, ['hr_head','admin','super_admin'], true);
            case 'admin':
                // Only admin or super admin
                return in_array($role, ['admin','super_admin'], true);
            default:
                return false;
        }
    }
}

// Fetch approver user IDs for a company by role slug(s)
if (!function_exists('getApproverUserIdsByRole')) {
    function getApproverUserIdsByRole(PDO $db, int $companyId, $roleSlugs): array {
        $slugs = is_array($roleSlugs) ? $roleSlugs : [$roleSlugs];
        if (!$slugs) return [];
        $placeholders = [];
        $params = [':cid' => $companyId];
        foreach ($slugs as $i => $slug) {
            $ph = ':r' . $i;
            $placeholders[] = $ph;
            $params[$ph] = $slug;
        }
        $sql = "SELECT u.id AS user_id
                FROM users u
                JOIN user_roles ur ON ur.user_id = u.id
                JOIN roles r ON r.id = ur.role_id
                JOIN employees e ON e.user_id = u.id
                WHERE e.company_id = :cid AND r.slug IN (" . implode(',', $placeholders) . ")";
        $st = $db->prepare($sql);
        foreach ($params as $k => $v) $st->bindValue($k, $v);
        $st->execute();
        $rows = $st->fetchAll();
        return array_values(array_filter(array_map(function($r){ return (int)($r['user_id'] ?? 0); }, $rows)));
    }
}

// Notify next-stage approvers when a request advances to a new stage
if (!function_exists('notifyNextStageApprovers')) {
    function notifyNextStageApprovers(PDO $db, int $requestId, string $nextStageRole): void {
        $ctx = getRequestContext($db, $requestId);
        if (!$ctx) return;
        // Ensure notifications table exists
        if (function_exists('ensureNotificationsTable')) { ensureNotificationsTable($db); }
        // Load details for message
        $q = "SELECT lr.id, lr.days_requested, lt.name AS leave_type_name,
                     CONCAT(e.first_name,' ',e.last_name) AS employee_name
              FROM leave_requests lr
              JOIN employees e ON e.id = lr.employee_id
              JOIN leave_types lt ON lt.id = lr.leave_type_id
              WHERE lr.id = :id";
        $st = $db->prepare($q);
        $st->bindValue(':id', $requestId, PDO::PARAM_INT);
        $st->execute();
        $info = $st->fetch();
        if (!$info) return;

        $title = 'Leave Approval Required';
        $content = sprintf('%s requested %d day(s) of %s. Action required.',
            $info['employee_name'] ?? 'Employee', (int)($info['days_requested'] ?? 0), $info['leave_type_name'] ?? 'leave');

        $recipients = [];
        if ($nextStageRole === 'hr_head') {
            $recipients = getApproverUserIdsByRole($db, (int)$ctx['company_id'], ['hr_head']);
            if (!$recipients) { // fallback to admin if no HR Head
                $recipients = getApproverUserIdsByRole($db, (int)$ctx['company_id'], ['admin','super_admin']);
            }
        } elseif ($nextStageRole === 'admin') {
            $recipients = getApproverUserIdsByRole($db, (int)$ctx['company_id'], ['admin','super_admin']);
        }
        if (!$recipients) return;

        foreach ($recipients as $uid) {
            $ins = $db->prepare("INSERT INTO notifications (user_id, type, title, content, created_at) VALUES (:uid, :type, :title, :content, NOW())");
            $ins->bindValue(':uid', $uid, PDO::PARAM_INT);
            $ins->bindValue(':type', 'leave_request_needs_approval');
            $ins->bindValue(':title', $title);
            $ins->bindValue(':content', $content);
            $ins->execute();
        }
    }
}
